
/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz 04/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MHBlindPixels
//
////////////////////////////////////////////////////////////////////////////
#include "MHBlindPixels.h" 

#include <TCanvas.h>

#include "MMcEvt.hxx"
#include "MBlindPixels.h"
#include "MGeomCam.h"
#include "MPedPhotCam.h"
#include "MParList.h"
#include "MBinning.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHBlindPixels);

using namespace std;

// -------------------------------------------------------------------------
//
//  Default Constructor.
//
MHBlindPixels::MHBlindPixels(const char *name, const char *title)
{
    fName  = name  ? name  : "MHBlindPixels";
    fTitle = title ? title : "Histogram for Blind Pixels vs. Theta";

    //  - we initialize the histogram
    //  - we have to give different names for the different histograms because
    //    root don't allow us to have diferent histograms with the same name

    fBlindId.SetName("2D-IdBlindPixels");
    fBlindId.SetTitle("2D-IdBlindPixels");
    fBlindId.SetDirectory(NULL);
    fBlindId.SetXTitle("\\Theta [\\circ]");
    fBlindId.SetYTitle("pixel Id");

    fBlindN.SetName("2D-NBlindPixels");
    fBlindN.SetTitle("2D-NBlindPixels");
    fBlindN.SetDirectory(NULL);
    fBlindN.SetXTitle("\\Theta [\\circ]");
    fBlindN.SetYTitle("number of blind pixels");
}

// --------------------------------------------------------------------------
//
// Set the binnings and prepare the filling of the histogram
//
Bool_t MHBlindPixels::SetupFill(const MParList *plist)
{
    MGeomCam *fCam = (MGeomCam*)plist->FindObject(AddSerialNumber("MGeomCam"));
    if (!fCam)    
    {
        *fLog << err << "MHBlindPixels::SetupFill; MGeomCam not found... aborting." << endl;
        return kFALSE;
    }

    fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << err << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    
    fPedPhot = (MPedPhotCam*)plist->FindObject("MPedPhotCam");
    if (!fPedPhot)
    {
        *fLog << err << "MPedPhotCam not found... aborting." << endl;
        return kFALSE;
    }
    fPedPhot->InitSize(fCam->GetNumPixels());
    

    // Get Theta Binning
    MBinning* binstheta  = (MBinning*)plist->FindObject("BinningTheta", "MBinning");
    if (!binstheta)
    {
        *fLog << err << "Object 'BinningTheta' [MBinning] not found... aborting" << endl;
        return kFALSE;
    }

    // Get binning for pixel number
    const UInt_t npix1 = fPedPhot->GetSize()+1;

    MBinning binspix("BinningPixel");
    binspix.SetEdges(npix1, -0.5, npix1-0.5);

    // Set binnings in histograms
    SetBinning(&fBlindId, binstheta, &binspix);
    SetBinning(&fBlindN,  binstheta, &binspix);

    return kTRUE;
}

// ------------------------------------------------------------------------
// 
// Drawing function. It creates its own canvas.
//
void MHBlindPixels::Draw(Option_t *option)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    pad->Divide(2,2);

    TH1D *h;

    pad->cd(1);
    fBlindId.Draw(option);

    pad->cd(2);
    fBlindN.Draw(option);

    pad->cd(3);
    gPad->SetBorderMode(0);
    h = ((TH2*)&fBlindId)->ProjectionY("ProjY-pixId", -1, 9999, "");
    h->SetDirectory(NULL);
    h->SetTitle("Distribution of blind pixel Id");
    h->SetXTitle("Id of blind pixel");
    h->SetYTitle("No. of events");
    h->Draw(option);
    h->SetBit(kCanDelete);

    pad->cd(4);
    gPad->SetBorderMode(0);
    h = ((TH2*)&fBlindN)->ProjectionY("ProjY-pixN", -1, 9999, "");
    h->SetDirectory(NULL);
    h->SetTitle("Distribution of no.of blind pixels");
    h->SetXTitle("No. of blind pixels");
    h->SetYTitle("No. of events");
    h->Draw(option);
    h->SetBit(kCanDelete);

    pad->Modified();
    pad->Update();
}

Bool_t MHBlindPixels::Fill(const MParContainer *par, const Stat_t w)
{
    if (!par)
        return kFALSE;

    Double_t theta = fMcEvt->GetTelescopeTheta()*kRad2Deg;
    const MBlindPixels &bp = *(MBlindPixels*)par;

    // FIXME: Slow.
    const UInt_t npix = fPedPhot->GetSize();

    UInt_t nb = 0;
    for (UInt_t i=0; i<npix; i++)
    {
        if (bp.IsBlind(i))
	{
          fBlindId.Fill(theta, i, w);
          nb++;
	}   
    }
    fBlindN.Fill(theta, nb, w);

    return kTRUE;
}












