/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Javier Lopez 12/2003 <mailto:jlopez@ifae.es>
!   Author(s): Javier Rico  01/2004 <mailto:jrico@ifae.es>
!   Author(s): Markus Gaug  02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MCalibrate
//
//   This task takes the integrated charge from MExtractedSignal and apply
//   the calibration constants from MCalibraitionCam to convert the summed FADC 
//   slices to photons. The number of photons obtained is stored in MCerPhotEvt. 
//
//   Selection of different calibration methods is possible through the 
//   SetCalibrationMode member function
//
//   The calibration modes which exclude non-valid pixels are the following: 
//
//   kFfactor:    calibrates using the F-Factor method
//   kBlindpixel: calibrates using the BlindPixel method 
//   kBlindpixel: calibrates using the BlindPixel method 
//   kDummy:      calibrates with fixed conversion factors of 1 and errors of 0.
//
//   The calibration modes which include all pixels regardless of their validity is:
//
//   kNone:       calibrates with fixed conversion factors of 1 and errors of 0.
//
//   Use the kDummy and kNone methods ONLY FOR DEBUGGING!   
//
//   Input Containers:
//    MExtractedSingal
//    MCalibrationChargeCam
//
//   Output Containers:
//    MCerPhotEvt
//
//////////////////////////////////////////////////////////////////////////////
#include "MCalibrate.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"
#include "MH.h"

#include "MGeomCam.h"

#include "MCalibrationChargeCam.h"
#include "MCalibrationChargePix.h"

#include "MCalibrationQECam.h"
#include "MCalibrationQEPix.h"

#include "MExtractedSignalCam.h"
#include "MExtractedSignalPix.h"

#include "MBadPixelsCam.h"
#include "MBadPixelsPix.h"

#include "MCerPhotEvt.h"

ClassImp(MCalibrate);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
MCalibrate::MCalibrate(CalibrationMode_t calmode,const char *name, const char *title) 
    : fGeomCam(NULL), fCalibrations(NULL), fQEs(NULL), fBadPixels(NULL), fSignals(NULL), 
      fCerPhotEvt(NULL), fCalibrationMode(calmode)
{
    fName  = name  ? name  : "MCalibrate";
    fTitle = title ? title : "Task to calculate the number of photons in one event";
}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MGeomCam
//  - MCalibrationChargeCam
//  - MExtractedSignalCam
//  - MBadPixelsCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MCerPhotEvt
//
Int_t MCalibrate::PreProcess(MParList *pList)
{

    fSignals = (MExtractedSignalCam*)pList->FindObject(AddSerialNumber("MExtractedSignalCam"));

    if (!fSignals)
    {
      *fLog << err << AddSerialNumber("MExtractedSignalCam") << " not found ... aborting" << endl;
        return kFALSE;
    }

    fBadPixels = (MBadPixelsCam*)pList->FindObject(AddSerialNumber("MBadPixelsCam"));
    if (!fBadPixels)
      *fLog << warn << AddSerialNumber("MBadPixelsCam") << " not found ... no action" << endl;
    
    if(fCalibrationMode>kNone)
      {

	fCalibrations = (MCalibrationChargeCam*)pList->FindObject(AddSerialNumber("MCalibrationChargeCam"));
	if (!fCalibrations)
	  {
	    *fLog << err << AddSerialNumber("MCalibrationChargeCam") << " not found ... aborting." << endl;
	    return kFALSE;
	  }

	fQEs = (MCalibrationQECam*)pList->FindObject(AddSerialNumber("MCalibrationQECam"));
	if (!fQEs)
	  {
	    *fLog << err << AddSerialNumber("MCalibrationQECam") << " not found ... aborting." << endl;
	    return kFALSE;
	  }
        
      }

    fCerPhotEvt = (MCerPhotEvt*)pList->FindCreateObj(AddSerialNumber("MCerPhotEvt"));
    if (!fCerPhotEvt)
        return kFALSE;
    
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for validity of the selected calibration method, switch to a 
// different one in case of need
//
Bool_t MCalibrate::ReInit(MParList *pList)
{

  if(fCalibrationMode == kBlindPixel && !fQEs->IsBlindPixelMethodValid())
    {
      *fLog << warn << GetDescriptor() 
            << "Warning: Blind pixel calibration method not valid, switching to F-factor method" << endl;
      fCalibrationMode = kFfactor;
    }
  
  if(fCalibrationMode == kPinDiode && !fQEs->IsPINDiodeMethodValid())
    { 
      *fLog << warn << GetDescriptor() 
            << "Warning: PIN diode calibration method not valid, switching to F-factor method" << endl;
      fCalibrationMode = kFfactor;
    }

  if(fCalibrationMode == kCombined && !fQEs->IsCombinedMethodValid())
    { 
      *fLog << warn << GetDescriptor() 
            << "Warning: Combined calibration method not valid, switching to F-factor method" << endl;
      fCalibrationMode = kFfactor;
    }

  switch(fCalibrationMode)
    {
    case kBlindPixel:
      break;
    case kFfactor:
      break;
    case kPinDiode:
      *fLog << err << GetDescriptor() 
                    << ": PIN Diode Calibration mode not yet available " << endl;
      return kFALSE;
      break;
    case kCombined:
      *fLog << err << GetDescriptor() 
                    << ": Combined Calibration mode not yet available " << endl;
      return kFALSE;
      break;
    case kDummy:
      *fLog << warn << GetDescriptor() 
                    << ": WARNING: Dummy calibration, no calibration applied!!" << endl;
      break;
    case kNone:
      *fLog << warn << GetDescriptor() 
                    << ": WARNING: No calibration applied!!" << endl;
      break;
    default:
      *fLog << warn << GetDescriptor() 
            << ": WARNING: Calibration mode value ("
            <<fCalibrationMode<<") not known" << endl;
      return kFALSE;
    }
  
  return kTRUE;
}
// --------------------------------------------------------------------------
//
// Apply the calibration factors to the extracted signal according to the 
// selected calibration method
//
Int_t MCalibrate::Process()
{

  //
  // For the moment, we use only a dummy zenith for the calibration:
  //
  const Float_t zenith = 0;

  /*
    if (fCalibrations->GetNumPixels() != (UInt_t)fSignals->GetSize())
    {
        // FIXME: MExtractedSignal must be of variable size -
        //        like MCerPhotEvt - because we must be able
        //        to reduce size by zero supression
        //        For the moment this check could be done in ReInit...
        *fLog << err << "MExtractedSignal and MCalibrationCam have different sizes... abort." << endl;
        return kFALSE;
    }
  */

  UInt_t npix = fSignals->GetSize();

  Float_t hiloconv      = 0.;
  Float_t hiloconverr   = 0.;
  Float_t calibConv     = 0.;
  Float_t calibConvVar  = 0.;
  Float_t calibFFactor  = 0.;
  Float_t calibQE       = 1.;
  Float_t calibQEVar    = 0.;
  
  for (UInt_t pixidx=0; pixidx<npix; pixidx++)
    {

      if(fCalibrationMode!=kNone)
	{

	  
	  MCalibrationChargePix &pix = (MCalibrationChargePix&)(*fCalibrations)[pixidx];
	  MCalibrationQEPix     &qe  = (MCalibrationQEPix&)    (*fQEs         )[pixidx];

          if (fBadPixels)
            {
              MBadPixelsPix         &bad = (*fBadPixels)[pixidx];
              if (!bad.IsCalibrationResultOK())
                continue;
            }
            
          calibConv      = pix.GetMeanConvFADC2Phe();
          calibConvVar   = pix.GetMeanConvFADC2PheVar();
          calibFFactor   = pix.GetMeanFFactorFADC2Phot();

	  switch(fCalibrationMode)
	    {
	    case kBlindPixel:
              if (qe.IsBlindPixelMethodValid())
		{
                  calibQE      = qe.GetQECascadesBlindPixel   ( zenith );
                  calibQEVar   = qe.GetQECascadesBlindPixelVar( zenith );
		}
              else
                continue;
	      break;
	    case kPinDiode:
              if (qe.IsPINDiodeMethodValid())
		{
                  calibQE      = qe.GetQECascadesPINDiode   ( zenith );
                  calibQEVar   = qe.GetQECascadesPINDiodeVar( zenith );
		}
		else
                  continue;
	      break;
	    case kFfactor:
              if (pix.IsFFactorMethodValid())
		{
                  calibQE      = qe.GetQECascadesFFactor   ( zenith );
                  calibQEVar   = qe.GetQECascadesFFactorVar( zenith );
		}
              else
                continue;
	      break;
	    case kCombined:
              if (qe.IsCombinedMethodValid())
		{
                  calibQE      = qe.GetQECascadesCombined   ( zenith );
                  calibQEVar   = qe.GetQECascadesCombinedVar( zenith );
		}
              else
                continue;
              break;
            case kDummy:
 	      hiloconv     = 1.;
	      hiloconverr  = 0.;
	      calibQE      = 1.;
	      calibQEVar   = 0.;
              break;

	    } /* switch calibration mode */
	  
	  hiloconv   = pix.GetConversionHiLo   ();
	  hiloconverr= pix.GetConversionHiLoErr();
	  
	} /* if(fCalibrationMode!=kNone) */
      else
      {
	  hiloconv       = 1.;
	  hiloconverr    = 0.;
	  calibConv      = 1.;
	  calibConvVar   = 0.;
	  calibFFactor   = 0.;
	  calibQE        = 1.;
	  calibQEVar     = 0.;
      }      
      MExtractedSignalPix &sig =  (*fSignals)[pixidx];
      
      Float_t signal;
      Float_t signalErr = 0.;
      Float_t nphot,nphotErr;
            
      if (sig.IsLoGainUsed())
        {
          signal    = sig.GetExtractedSignalLoGain()*hiloconv;
          signalErr = signal*hiloconverr;
        }
      else
        {
	  if (sig.GetExtractedSignalHiGain() > 9999.)
	    {
	      signal    = 0.;
	      signalErr = 0.;
	    }
	  else
	    signal = sig.GetExtractedSignalHiGain();
        }
      
      nphot    = signal*calibConv/calibQE;
      nphotErr = calibFFactor*TMath::Sqrt(nphot);

      //
      // The following part is the outcommented first version of the error calculation
      // Contact Markus Gaug for questions (or wait for the next documentation update...)
      //
      /*
        nphotErr = signal    > 0 ? signalErr*signalErr / (signal * signal)  : 0.
                 + calibConv > 0 ? calibConvVar  / (calibConv * calibConv ) : 0.
                 + calibQE   > 0 ? calibQEVar    / (calibQE   * calibQE   ) : 0.;
        nphotErr  = TMath::Sqrt(nphotErr) * nphot;
      */

      MCerPhotPix *cpix = fCerPhotEvt->AddPixel(pixidx, nphot, nphotErr);

      if (sig.GetNumLoGainSaturated() > 0)
	cpix->SetPixelSaturated();

      if (sig.GetNumHiGainSaturated() > 0)
	cpix->SetPixelHGSaturated();


    } /* for (UInt_t pixidx=0; pixidx<npix; pixidx++) */
  
  fCerPhotEvt->FixSize();
  fCerPhotEvt->SetReadyToSave();
  
  return kTRUE;
}
