/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 10/2001 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */
#include "MEventDisplay.h"

//
// C-lib
//
#include <stdlib.h>              // atoi

//
// root
//
#include <TFile.h>               // TFile
#include <TList.h>               // TList::Add
#include <TStyle.h>              // gStyle->SetOptStat
#include <TCanvas.h>             // TCanvas::cd

//
// root GUI
//
#include <TGLabel.h>             // TGLabel
#include <TGButton.h>            // TGPictureButton
#include <TG3DLine.h>            // TGHorizontal3DLine
#include <TGTextEntry.h>         // TGTextEntry
#include <TGButtonGroup.h>       // TGVButtonGroup
#include <TRootEmbeddedCanvas.h> // TRootEmbeddedCanvas

//
// General
//
#include "MGList.h"              // MGList

#include "MParList.h"            // MParList::AddToList
#include "MEvtLoop.h"            // MEvtLoop::GetParList
#include "MTaskList.h"           // MTaskList::AddToList

//
// Tasks
//
#include "MReadMarsFile.h"        // MReadMarsFile
#include "MGeomApply.h"           // MGeomApply
#include "MFDataMember.h"         // MFDataMember
#include "MMcPedestalCopy.h"      // MMcPedestalCopy
#include "MMcPedestalNSBAdd.h"    // MMcPedestalNSBAdd

#include "MCerPhotAnal2.h"        // MCerPhotAnal2
#include "MImgCleanStd.h"         // MImgCleanStd
#include "MHillasCalc.h"          // MHillasCalc
#include "MHillasSrcCalc.h"       // MHillasSrcCalc
#include "MBlindPixelCalc.h"      // MBlindPixelCalc
#include "MArrivalTimeCalc.h"     // MArrivalTimeCalc
#include "MFillH.h"               // MFillH
#include "MExtractSignal.h"       // MExtractsignal
#include "MMcCalibrationUpdate.h" // MMcCalibrationUpdate
#include "MCalibrate.h"           // MCalibrate

//
// Container
//
#include "MHillas.h"               // MHillas::Print(const MGeomCam&)
#include "MHillasExt.h"            // MHillasExt::Print(const MGeomCam&)
#include "MHillasSrc.h"            // MHillasSrc::Print(const MGeomCam&)
#include "MNewImagePar.h"          // MNewImagePar::Print(const MGeomCam&)
#include "MHEvent.h"               // MHEvent
#include "MHCamera.h"              // MHCamera
#include "MRawEvtData.h"           // MRawEvtData
#include "MArrivalTimeCam.h"       // MArrivalTimeCam
#include "MBadPixelsCam.h"         // MBadPixelsCam
#include "MPedPhotCam.h"           // MPedPhotCam
#include "MCalibrationChargeCam.h" // MCalibrationChargeCam

ClassImp(MEventDisplay);

// --------------------------------------------------------------------------
//
//  Constructor.
//
MEventDisplay::MEventDisplay(const char *fname, const char *pname, const char *cname) : MStatusDisplay()
{
    //
    // Setup Task list for hillas calculation
    //
    SetupTaskList("Events", fname, pname, cname);

    //
    // Add MEventDisplay GUI elements to the display
    //
    AddUserFrame(fname);

    //
    // Show new part of the window, resize to correct aspect ratio
    //
    // FIXME: This should be done by MStatusDisplay automatically
    Resize(GetWidth(), GetHeight() + fUserFrame->GetDefaultHeight());
    SetWindowName("Event Display");
    MapSubwindows();

    //
    // Readin first event and display it
    //
    ReadFirstEvent();
}

// --------------------------------------------------------------------------
//
//  Destructor: PostProcess eventloop, delete eventloop with all containers
//
MEventDisplay::~MEventDisplay()
{
    fEvtLoop->PostProcess();
    delete fEvtLoop;
}

// --------------------------------------------------------------------------
//
//  Setup Task and parameter list for hillas calculation,
//  preprocess tasks and read in first event (process)
//
void MEventDisplay::SetupTaskList(const char *tname, const char *fname,
                                  const char *pname, const char *cname)
{
    MCalibrationChargeCam *ccam=0;
    MPedPhotCam           *pcam=0;

    MBadPixelsCam *badpix = new MBadPixelsCam;

    TFile file(pname, "READ");
    if (!file.IsZombie())
        pcam = new MPedPhotCam;
    if (pcam)
    {
        if (pcam->Read()<=0)
        {
            delete pcam;
            pcam = NULL;
        }

        if (file.FindKey("MBadPixelsCam"))
        {
            MBadPixelsCam bad;
            if (bad.Read()>0)
                badpix->Merge(bad);
        }
    }
    file.Close();
    file.Open(cname, "READ");
    if (!file.IsZombie())
        ccam = new MCalibrationChargeCam;
    if (ccam)
    {
        if (ccam->Read()<=0)
        {
            delete ccam;
            ccam = NULL;
        }

        if (file.FindKey("MBadPixelsCam"))
        {
            MBadPixelsCam bad;
            if (bad.Read()>0)
                badpix->Merge(bad);
        }
    }
    file.Close();

    //
    // Setup an empty job, with a reader task only.
    // All tasks and parameter containers are deleted automatically
    // (via SetOwner())
    //
    MTaskList *tlist = new MTaskList;
    tlist->SetOwner();

    MReadMarsFile *read = new MReadMarsFile(tname, fname);
    read->DisableAutoScheme();
    tlist->AddToList(read);

    MGeomApply *apl = new MGeomApply;
    tlist->AddToList(apl);

    MParList *plist = new MParList;
    plist->SetOwner();
    plist->AddToList(tlist);
    plist->AddToList(badpix);
    if (pcam)
        plist->AddToList(pcam);
    if (ccam)
        plist->AddToList(ccam);

    fEvtLoop = new MEvtLoop;
    fEvtLoop->SetOwner();
    fEvtLoop->SetParList(plist);

    MHEvent *evt1 = new MHEvent(MHEvent::kEvtSignalRaw);
    MHEvent *evt2 = new MHEvent(MHEvent::kEvtSignalRaw);
    MHEvent *evt3 = new MHEvent(MHEvent::kEvtPedestal);
    MHEvent *evt4 = new MHEvent(MHEvent::kEvtPedestalRMS);
    MHEvent *evt5 = new MHEvent(MHEvent::kEvtRelativeSignal);
    MHEvent *evt6 = new MHEvent(MHEvent::kEvtCleaningLevels);
    MHEvent *evt7 = new MHEvent(MHEvent::kEvtIdxMax);
    MHEvent *evt8 = new MHEvent(MHEvent::kEvtArrTime);

    evt1->SetName("Signal");
    evt2->SetName("Cleaned");
    evt3->SetName("Pedestal");
    evt4->SetName("PedRMS");
    evt5->SetName("Signal/PedRMS");
    evt6->SetName("CleanLevels");
    evt7->SetName("Max Slice Idx");
    evt8->SetName("Arrival Time");

    // This makes sure, that the containers are deleted...
    plist->AddToList(evt1);
    plist->AddToList(evt2);
    plist->AddToList(evt3);
    plist->AddToList(evt4);
    plist->AddToList(evt5);
    plist->AddToList(evt6);
    plist->AddToList(evt7);
    plist->AddToList(evt8);

    MCerPhotAnal2     *nanal = new MCerPhotAnal2;
    MFillH            *fill1 = new MFillH(evt1, "MCerPhotEvt", "MFillH1");
    MImgCleanStd      *clean = new MImgCleanStd;
    MFillH            *fill2 = new MFillH(evt2, "MCerPhotEvt", "MFillH2");
    MFillH            *fill3 = new MFillH(evt3, "MPedPhotCam", "MFillH3");
    MFillH            *fill4 = new MFillH(evt4, "MPedPhotCam", "MFillH4");
    MFillH            *fill5 = new MFillH(evt5, "MCameraData", "MFillH5");
    MFillH            *fill6 = new MFillH(evt6, "MCameraData", "MFillH6");
    MBlindPixelCalc   *blind = new MBlindPixelCalc;
    MHillasCalc       *hcalc = new MHillasCalc;
    MHillasSrcCalc    *scalc = new MHillasSrcCalc;

    // If no pedestal or no calibration file is availble
    if (!pcam || !ccam)
    {
        MFilter *f1=new MFDataMember("MRawRunHeader.fRunType", '>', 255.5);
        MFilter *f2=new MFDataMember("MRawRunHeader.fRunType", '<', 255.5);
        f1->SetName("MFMonteCarlo");
        f2->SetName("MFRealData");

        MMcPedestalCopy   *pcopy = new MMcPedestalCopy;
        MMcPedestalNSBAdd *pdnsb = new MMcPedestalNSBAdd;

        MExtractSignal* extra = new MExtractSignal();
	extra->SetRange(5, 10, 5, 10);
	extra->SetSaturationLimit(240);

        MMcCalibrationUpdate* mcupd = new MMcCalibrationUpdate;
        mcupd->SetOuterPixelsGainScaling(kFALSE);

        MCalibrate* mccal = new MCalibrate;
        mccal->SetCalibrationMode(MCalibrate::kFfactor);

        // MC
        extra->SetFilter(f1);
        mcupd->SetFilter(f1);
        mccal->SetFilter(f1);

        // Data
        nanal->SetFilter(f2);

        // TaskList
        tlist->AddToList(f1);
        tlist->AddToList(f2);
        tlist->AddToList(pcopy);
        tlist->AddToList(pdnsb);

        tlist->AddToList(extra);
        tlist->AddToList(mcupd);
        tlist->AddToList(mccal);

        tlist->AddToList(nanal);
    }
    else
    {
        MCalibrate* calib = new MCalibrate;
        tlist->AddToList(calib);
    }

    tlist->AddToList(fill1);
    tlist->AddToList(clean);
    tlist->AddToList(fill2);
    tlist->AddToList(fill3);
    tlist->AddToList(fill4);
    tlist->AddToList(fill5);
    tlist->AddToList(fill6);
    tlist->AddToList(blind);
    tlist->AddToList(hcalc);
    tlist->AddToList(scalc);
    if (!pcam || !ccam)
    {
        MArrivalTimeCalc  *tcalc = new MArrivalTimeCalc;
        MFillH            *fill7 = new MFillH(evt7, "MRawEvtData",     "MFillH7");
        MFillH            *fill8 = new MFillH(evt8, "MArrivalTimeCam", "MFillH8");
        tlist->AddToList(tcalc);
        tlist->AddToList(fill7);
        tlist->AddToList(fill8);
    }

    //
    // Now distribute Display to all tasks
    //
    tlist->SetDisplay(this);
}

// --------------------------------------------------------------------------
//
//  Add the top part of the frame: This is filename and treename display
//
void MEventDisplay::AddTopFramePart1(TGCompositeFrame *frame,
                                    const char *filename,
                                    const char *treename)
{
    //
    //  --- the top1 part of the window ---
    //
    TGHorizontalFrame *top1 = new TGHorizontalFrame(frame, 1, 1);
    fList->Add(top1);

    //
    // create gui elements
    //
    TGLabel *file = new TGLabel(top1, new TGString(Form("%s#%s", filename, treename)));
    fList->Add(file);

    //
    // layout and add gui elements in/to frame
    //
    TGLayoutHints *laystd = new TGLayoutHints(kLHintsCenterX, 5, 5);
    fList->Add(laystd);

    top1->AddFrame(file,  laystd);

    //
    //  --- the top1 part of the window ---
    //
    TGHorizontalFrame *top2 = new TGHorizontalFrame(frame, 1, 1);
    fList->Add(top2);

    //
    // layout and add frames
    //
    TGLayoutHints *laytop1 = new TGLayoutHints(kLHintsCenterX, 5, 5, 5);
    fList->Add(laytop1);
    frame->AddFrame(top1, laytop1);
    frame->AddFrame(top2, laytop1);
}

// --------------------------------------------------------------------------
//
//  Add the second part of the top frame: This are the event number controls
//
void MEventDisplay::AddTopFramePart2(TGCompositeFrame *frame)
{
    //
    // --- the top2 part of the window ---
    //
    TGHorizontalFrame *top2 = new TGHorizontalFrame(frame, 1, 1);
    fList->Add(top2);

    //
    // Create the gui elements
    //
    TGTextButton *prevevt = new TGTextButton(top2, " << ", kEvtPrev);
    prevevt->Associate(this);

    TGLabel *evtnr = new TGLabel(top2, new TGString("Event:"));

    TGTextEntry *entry=new TGTextEntry(top2, new TGTextBuffer(100), kEvtNumber);
    entry->Resize(60, entry->GetDefaultHeight());
    entry->Associate(this);

    fNumOfEvts = new TGLabel(top2, "of           .");

    TGTextButton *nextevt = new TGTextButton (top2, " >> ", kEvtNext);
    nextevt->Associate(this);

    //
    // Add gui elements to 'atotodel'
    //
    fList->Add(prevevt);
    fList->Add(evtnr);
    fList->Add(entry);
    fList->Add(fNumOfEvts);
    fList->Add(nextevt);

    //
    // add the gui elements to the frame
    //
    TGLayoutHints *laystd = new TGLayoutHints(kLHintsLeft|kLHintsCenterY, 5, 5);
    fList->Add(laystd);

    top2->AddFrame(prevevt,    laystd);
    top2->AddFrame(evtnr,      laystd);
    top2->AddFrame(entry,      laystd);
    top2->AddFrame(fNumOfEvts, laystd);
    top2->AddFrame(nextevt,    laystd);

    TGLayoutHints *laystd2 = new TGLayoutHints(kLHintsCenterX, 5, 5, 5, 5);
    fList->Add(laystd2);
    frame->AddFrame(top2, laystd2);

    //
    // Add trailing line...
    //
    TGHorizontal3DLine *line = new TGHorizontal3DLine(frame);
    TGLayoutHints *layline = new TGLayoutHints(kLHintsExpandX);
    fList->Add(line);
    fList->Add(layline);
    frame->AddFrame(line, layline);
}

// --------------------------------------------------------------------------
//
//  Add the user frame part of the display
//
void MEventDisplay::AddUserFrame(const char* filename)
{
    fUserFrame->ChangeOptions(kHorizontalFrame);

    TGCompositeFrame *vf1 = new TGVerticalFrame(fUserFrame, 1, 1);
    TGCompositeFrame *vf2 = new TGVerticalFrame(fUserFrame, 1, 1);

    AddTopFramePart1(vf1, filename, "Events");
    AddTopFramePart2(vf1);

    // create root embedded canvas and add it to the tab
    TRootEmbeddedCanvas *ec = new TRootEmbeddedCanvas("Slices", vf2, vf1->GetDefaultHeight()*3/2, vf1->GetDefaultHeight(), 0);
    vf2->AddFrame(ec);
    fList->Add(ec);

    // set background and border mode of the canvas
    fCanvas = ec->GetCanvas();
    fCanvas->SetBorderMode(0);
    gROOT->GetListOfCanvases()->Add(fCanvas);
    //fCanvas->SetBorderSize(1);
    //fCanvas->SetBit(kNoContextMenu);
    //fCanvas->SetFillColor(16);

    TGLayoutHints *lay = new TGLayoutHints(kLHintsExpandX);
    fUserFrame->AddFrame(vf1, lay);
    fUserFrame->AddFrame(vf2);
}

// --------------------------------------------------------------------------
//
//  Checks if the event number is valid, and if so reads the new event
//  and updates the display
//
void MEventDisplay::ReadinEvent(Int_t dir)
{
    MParList    *plist = (MParList*)   fEvtLoop->GetParList();
    MTaskList   *tlist = (MTaskList*)  fEvtLoop->GetTaskList();
    MGeomCam    *geom  = (MGeomCam*)   plist->FindObject("MGeomCam");
    MRawEvtData *raw   = (MRawEvtData*)plist->FindObject("MRawEvtData");

    //
    // Read first event.
    //
    MReadTree *reader = (MReadTree*)fEvtLoop->FindTask("MRead");

    const Int_t num = reader->GetNumEntry();

    do
    {
        if (dir<0 && !reader->DecEventNum())
        {
            reader->SetEventNum(num);
            return;
        }
        if (dir>0 && !reader->IncEventNum())
        {
            reader->SetEventNum(num);
            return;
        }

        if (!tlist->Process())
            return;

        reader->DecEventNum();

    } while (raw && raw->GetNumPixels()<1 && dir!=0);

    //
    // Cleare the 'FADC canvas'
    //
    fCanvas->Clear();
    fCanvas->Modified();
    fCanvas->Update();

    //
    // Print parameters
    //
    ((MHillas*)     plist->FindObject("MHillas"))->Print(*geom);
    ((MHillasExt*)  plist->FindObject("MHillasExt"))->Print(*geom);
    ((MHillasSrc*)  plist->FindObject("MHillasSrc"))->Print(*geom);
    ((MNewImagePar*)plist->FindObject("MNewImagePar"))->Print(*geom);

    //
    // UpdateDisplay
    //
    gStyle->SetOptStat(1101);
    Update();

    TGTextEntry *entry = (TGTextEntry*)fList->FindWidget(kEvtNumber);
    entry->SetText(Form("%d", reader->GetNumEntry()+1));
}

// --------------------------------------------------------------------------
//
//  Read first event to get display booted
//
void MEventDisplay::ReadFirstEvent()
{
    if (!fEvtLoop->PreProcess())
        return;

    //
    // Get parlist
    //
    MParList *plist = (MParList*)fEvtLoop->GetParList();

    //
    // Add Geometry tab
    //
    AddGeometryTabs();

    //
    // Now read event...
    //
    ReadinEvent();

    MReadTree *reader = (MReadTree*)fEvtLoop->FindTask("MRead");
    TGString *txt = new TGString(Form("of %d", reader->GetEntries()));
    fNumOfEvts->SetText(txt);

    //
    // Draw ellipse on top of all pads
    //
    TObject *hillas = plist->FindObject("MHillas");
    for (int i=1; i<7;i++)
    {
        TCanvas *c = GetCanvas(i);
        c->GetPad(1)->cd(1);
        hillas->Draw();
    }
}

// --------------------------------------------------------------------------
//
//  Adds the geometry tab 
//
void MEventDisplay::AddGeometryTabs()
{
    MGeomCam *geom = (MGeomCam*)fEvtLoop->GetParList()->FindObject("MGeomCam");
    if (!geom)
        return;

    TCanvas &c1=AddTab("Geometry");

    MHCamera *cam = new MHCamera(*geom);
    cam->SetBit(TH1::kNoStats|MHCamera::kNoLegend|kCanDelete);
    cam->Draw("pixelindex");

    c1.Modified();
    c1.Update();

    TCanvas &c2=AddTab("Sectors");

    cam = new MHCamera(*geom);
    cam->SetBit(TH1::kNoStats|MHCamera::kNoLegend|kCanDelete);
    cam->Draw("sectorindex");

    c2.Modified();
    c2.Update();
}

// --------------------------------------------------------------------------
//
//  ProcessMessage(Long_t msg, Long_t parm1, Long_t parm2)
//
// Processes information from all GUI items.
// Selecting an item usually generates an event with 4 parameters.
// The first two are packed into msg (first and second bytes).
// The other two are parm1 and parm2.
//
Bool_t MEventDisplay::ProcessMessage(Long_t msg, Long_t mp1, Long_t mp2)
{
    switch (GET_MSG(msg))
    {
    case kC_TEXTENTRY:
        switch(GET_SUBMSG(msg))
        {
        case kTE_ENTER:
            switch(GET_SUBMSG(msg))
            {
            case kTE_ENTER:
                {
                    TGTextEntry *entry = (TGTextEntry*)fList->FindWidget(kEvtNumber);
                    MReadTree *reader  = (MReadTree*)fEvtLoop->FindTask("MRead");
                    if (reader->SetEventNum(atoi(entry->GetText())-1))
                        ReadinEvent();
                }
                return kTRUE;
            }
            return kTRUE;
        }
        break;

    case kC_COMMAND:
        switch (GET_SUBMSG(msg))
        {
        case kCM_TAB:
            {
                //
                // Set name for 'FADC canvas'. The name is the anchor for MHCamera.
                // and clear the canvas
                TCanvas *c = GetCanvas(mp1);
                if (!c)
                    break;
                MHEvent *o = (MHEvent*)fEvtLoop->GetParList()->FindObject(c->GetName());
                if (!o)
                    break;
                fCanvas->SetName(Form("%p;%p;PixelContent", o->GetHist(), c->GetPad(1)));
            }
            break;

        case kCM_BUTTON:
            switch (mp1)
            {
            case kEvtPrev:
                ReadinEvent(-1);
                return kTRUE;

            case kEvtNext:
                ReadinEvent(+1);
                return kTRUE;
            }
            return kTRUE;
        }
        break;
    }

    return MStatusDisplay::ProcessMessage(msg, mp1, mp2);
}
