/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!   Author(s): Markus Gaug 04/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2002-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MTimeExtractor  
//
//   Base class for the time extractor, see also MExtractor 
//
// Input Containers:
//   MPedestalCam
//
// Output Containers:
//   MArrivalTime
//
//////////////////////////////////////////////////////////////////////////////
#include "MTimeExtractor.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MRawRunHeader.h"
#include "MRawEvtPixelIter.h"

#include "MArrivalTimeCam.h"
#include "MArrivalTimePix.h"

#include "MPedestalCam.h"

ClassImp(MTimeExtractor);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor.
//
// Set: 
// - all pointers to NULL
//
MTimeExtractor::MTimeExtractor(const char *name, const char *title) 
    : fArrTime(NULL)
{
 
  fName  = name  ? name  : "MTimeExtractor";
  fTitle = title ? title : "Base class for arrival time extractors";

}


// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawEvtData
//  - MRawRunHeader
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MArrivalTimeCam
//
Int_t MTimeExtractor::PreProcess(MParList *pList)
{

    fRawEvt = (MRawEvtData*)pList->FindObject(AddSerialNumber("MRawEvtData"));
    if (!fRawEvt)
    {
        *fLog << err << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }
    
    fRunHeader = (MRawRunHeader*)pList->FindObject(AddSerialNumber("MRawRunHeader"));
    if (!fRunHeader)
      {
        *fLog << err << AddSerialNumber("MRawRunHeader") << " not found... aborting." << endl;
        return kFALSE;
      }
    
    fPedestals = (MPedestalCam*)pList->FindObject(AddSerialNumber("MPedestalCam"));
    if (!fPedestals)
    {
        *fLog << warn << GetDescriptor() 
              << ": MPedestalCam not found, calculate times without pedestal information..." << endl;
    }


    fArrTime = (MArrivalTimeCam*)pList->FindCreateObj(AddSerialNumber("MArrivalTimeCam"));
    if (!fArrTime)
    {
        *fLog << err << GetDescriptor() 
              << ": Cannot create MArrivalTimeCam... abort" << endl;
        return kFALSE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// The ReInit searches for:
// -  MRawRunHeader::GetNumSamplesHiGain()
// -  MRawRunHeader::GetNumSamplesLoGain()
//
// In case that the variables fHiGainLast and fLoGainLast are smaller than 
// the even part of the number of samples obtained from the run header, a
// warning is given an the range is set back accordingly. A call to:  
// - SetRange(fHiGainFirst, fHiGainLast-diff, fLoGainFirst, fLoGainLast) or 
// - SetRange(fHiGainFirst, fHiGainLast, fLoGainFirst, fLoGainLast-diff) 
// is performed in that case. The variable diff means here the difference 
// between the requested range (fHiGainLast) and the available one. Note that 
// the functions SetRange() are mostly overloaded and perform more checks, 
// modifying the ranges again, if necessary.
//
// Call: 
// - MArrivalTimeCam::SetUsedFADCSlices(fHiGainFirst, fHiGainLast, fLoGainFirst, fLoGainLast);
//
Bool_t MTimeExtractor::ReInit(MParList *pList)
{
  
  Int_t lastdesired   = (Int_t)fHiGainLast;
  Int_t lastavailable = (Int_t)fRunHeader->GetNumSamplesHiGain()-1;
  
  if (lastdesired > lastavailable)
    {
      const Int_t diff = lastdesired - lastavailable;
      *fLog << endl;
      *fLog << warn << GetDescriptor()
            << Form("%s%2i%s%2i%s%2i%s",": Selected Hi Gain FADC Window [",
                    (int)fHiGainFirst,",",lastdesired,
                    "] ranges out of the available limits: [0,",lastavailable,"].") << endl;
      *fLog << GetDescriptor() << ": Will reduce the upper edge to " << (int)(fHiGainLast - diff) << endl;
      SetRange(fHiGainFirst, fHiGainLast-diff, fLoGainFirst, fLoGainLast);
    }

  lastdesired   = (Int_t)(fLoGainLast);
  lastavailable = (Int_t)fRunHeader->GetNumSamplesLoGain()-1;
  
  if (lastdesired > lastavailable)
    {
      const Int_t diff = lastdesired - lastavailable;
      *fLog << endl; 
      *fLog << warn << GetDescriptor()
            << Form("%s%2i%s%2i%s%2i%s",": Selected Lo Gain FADC Window [",
                    (int)fLoGainFirst,",",lastdesired,
                    "] ranges out of the available limits: [0,",lastavailable,"].") << endl;
      *fLog << GetDescriptor() << ": Will reduce the upper edge to " << (int)(fLoGainLast - diff) << endl;
      SetRange(fHiGainFirst, fHiGainLast, fLoGainFirst, fLoGainLast-diff);
    }

  fArrTime->SetUsedFADCSlices(fHiGainFirst, fHiGainLast, fLoGainFirst, fLoGainLast);

  return kTRUE;
}


void MTimeExtractor::FindTimeHiGain(Byte_t *firstused, Float_t &time, Float_t &dtime, 
                                Byte_t &sat, const MPedestalPix &ped) const
{
  return;
}

void MTimeExtractor::FindTimeLoGain(Byte_t *firstused, Float_t &time, Float_t &dtime, 
                                Byte_t &sat, const MPedestalPix &ped) const
{
  return;
}


// --------------------------------------------------------------------------
//
// Evaluation of the mean arrival times 
// per pixel and store them in the MArrivalTime container.
//
Int_t MTimeExtractor::Process()
{

  MRawEvtPixelIter pixel(fRawEvt);
  fArrTime->Clear();

  while (pixel.Next())
    {
      
      //
      // Find signal in hi- and lo-gain
      //
      Float_t timehi, deltatimehi;
      Byte_t sathi = 0;
      
      //
      // Take correspodning pedestal
      //
      const UInt_t pixid = pixel.GetPixelId();
      const MPedestalPix  &ped = (*fPedestals)[pixid];

      FindTimeHiGain(pixel.GetHiGainSamples()+fHiGainFirst, timehi, deltatimehi, sathi, ped);

      Float_t timelo      =0.;
      Float_t deltatimelo =0.;
      Byte_t  satlo       =0;

      if (pixel.HasLoGain() && sathi>0)
        FindTimeLoGain(pixel.GetLoGainSamples()+fLoGainFirst, timelo, deltatimelo, satlo, ped);

      MArrivalTimePix &pix = (*fArrTime)[pixid];

      pix.SetArrivalTime(timehi,deltatimehi, timelo, deltatimelo);
      pix.SetGainSaturation(sathi, sathi, satlo);

    }
    
  fArrTime->SetReadyToSave();
  
  return kTRUE;
}


