/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!   Author(s): Javier Rico     04/2004 <mailto:jrico@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
// MSrcTranslate
//
// Task to translate the position of the source, starting from previous source
// position (fTranslationIsRelative=kTRUE) or from camera center 
// (fTranslationIsRelative=kFALSE)
//
//  Input Containers:
//    MSrcPosCam
//
//  Output Containers:
//    MSrcPosCam
//    MDCA
//
//////////////////////////////////////////////////////////////////////////////

#include <fstream>
#include <math.h>

#include "MParList.h"

#include "MSrcTranslate.h"
#include "MSrcPosCam.h"
#include "MDCA.h"


#include "MLog.h"
#include "MLogManip.h"

ClassImp(MSrcTranslate);

using namespace std;

static const TString gsDefName  = "MSrcTranslate";
static const TString gsDefTitle = "De-rotate position of the source";

// -------------------------------------------------------------------------
//
// Default constructor. The first (second) argument is the name of a container
// containing the source position in the camera plain, MScrPosCam (MDCA).
// The default is "MSrcPosCam" ("MDCA"). 
//
MSrcTranslate::MSrcTranslate(const char* srcPos, const char* dca, const char *name, const char *title)
  : fSrcPos(NULL), fDCA(NULL), fShiftX(0.), fShiftY(0.), fTranslationIsRelative(kTRUE)
{
    fName  = name  ? name  : gsDefName.Data();
    fTitle = title ? title : gsDefTitle.Data();

    fSrcPosName  = srcPos;
    fDCAName     = dca;
}

// -------------------------------------------------------------------------
//
// Look for/create the needed containers
//
Int_t MSrcTranslate::PreProcess(MParList *pList)
{
    // look for/create MSrcPosCam
  fSrcPos = (MSrcPosCam*)pList->FindObject(AddSerialNumber(fSrcPosName), "MSrcPosCam");
  if (!fSrcPos)
    {
      *fLog << warn << AddSerialNumber(fSrcPosName) << " [MSrcPosCam] not found... creating default container." << endl;
      fSrcPos = (MSrcPosCam*)pList->FindCreateObj("MSrcPosCam", AddSerialNumber(fSrcPosName));
      if(!fSrcPos)
	return kFALSE;
    }

  // look for/create MDCA
  fDCA = (MDCA*)pList->FindObject(AddSerialNumber(fDCAName), "MDCA");
  if (!fDCA)
    {
      *fLog << warn << AddSerialNumber(fDCAName) << " [MDCA] not found... creating default container." << endl;
      fDCA = (MDCA*)pList->FindCreateObj("MDCA", AddSerialNumber(fDCAName));
      if(!fDCA)
	return kFALSE;
    }

  if(fShiftX==0. && fShiftY==0.)
    *fLog << warn << "MSrcTranslate::PreProcess Warning: Null translation" << endl;    

  return kTRUE;
}

// -------------------------------------------------------------------------
//
// Perform the translation of the source position 
// 
Int_t MSrcTranslate::Process()
{  
  Double_t newX,newY;

  if(fTranslationIsRelative)
    {
      newX=fSrcPos->GetX()+fShiftX;
      newY=fSrcPos->GetY()+fShiftY;
    }
  else
    {
      newX=fShiftX;
      newY=fShiftY;
    }

  fSrcPos->SetX(newX);
  fSrcPos->SetY(newY);
  fDCA->SetRefPoint(newX,newY);

  return kTRUE;
}
