/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Javier Lpez , 4/2004 <mailto:jlopez@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
#include "MStarLocalPos.h"

#include <TEllipse.h>

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MStarLocalPos);

using namespace std;

MStarLocalPos::MStarLocalPos(const char *name, const char *title)
{

    fName  = name  ? name  : "MStarLocalPos";
    fTitle = title ? title : "";
    
    Reset();
}

void MStarLocalPos::Reset()
{

    //Expected position on camera
     fMagExp = 0.;
     fXExp = 0.;
     fYExp = 0.;

    //Info from calculation

     fMagCalc = 0.;
     fMaxCalc = 0.;
     fMeanXCalc = 0.;
     fMeanYCalc = 0.;
     fSigmaMinorAxisCalc = 0.;
     fSigmaMajorAxisCalc = 0.;

    //Info from fit

     fMagFit = 0.;
     fMaxFit = 0.;
     fMeanXFit = 0.;
     fMeanYFit = 0.;
     fSigmaMinorAxisFit = 0.;
     fSigmaMajorAxisFit = 0.;
     fChiSquare = 0.;
     fNdof = 1;

}

void MStarLocalPos::SetExpValues(Float_t mag, Float_t x, Float_t y)
{
     fMagExp = mag;
     fXExp = x;
     fYExp = y;
}

void MStarLocalPos::SetCalcValues(Float_t mag, Float_t max, Float_t x, Float_t y, Float_t sigmaMinorAxis, Float_t sigmaMajorAxis)
{
     fMagCalc = mag;
     fMaxCalc = max;
     fMeanXCalc = x;
     fMeanYCalc = y;
     fSigmaMinorAxisCalc = sigmaMinorAxis;
     fSigmaMajorAxisCalc = sigmaMajorAxis;
}

void MStarLocalPos::SetFitValues(Float_t mag, Float_t max, Float_t x, Float_t y, Float_t sigmaMinorAxis, Float_t sigmaMajorAxis, Float_t chiSquare, Int_t ndof)
{
     fMagFit = mag;
     fMaxFit = max;
     fMeanXFit = x;
     fMeanYFit = y;
     fSigmaMinorAxisFit = sigmaMinorAxis;
     fSigmaMajorAxisFit = sigmaMajorAxis;
     fChiSquare = chiSquare;
     fNdof = ndof;
}

// --------------------------------------------------------------------------
//
// Paint the ellipse corresponding to the parameters
//
void MStarLocalPos::Paint(Option_t *opt)
{
  //Print a cross in the expected position

  if (fSigmaMinorAxisCalc>0. && fSigmaMajorAxisCalc>0.)
    {
      TEllipse ecalc(fMeanXCalc, fMeanYCalc, fSigmaMinorAxisCalc, fSigmaMajorAxisCalc, 0, 360, 0);
      ecalc.SetLineWidth(2);
      ecalc.SetLineColor(kRed);
      ecalc.Paint();
    }

  if (fSigmaMinorAxisFit>0. || fSigmaMajorAxisFit>0.)
    {
      TEllipse efit(fMeanXFit, fMeanYFit, fSigmaMinorAxisFit, fSigmaMajorAxisFit, 0, 360, 0);
      efit.SetLineWidth(2);
      efit.SetLineColor(kBlack);
      efit.Paint();
    }
}
  
void MStarLocalPos::Print(Option_t *opt) const
{
  TString o = opt;
  
  if (o.Contains("mag", TString::kIgnoreCase) || opt == NULL)
    {
      *fLog << inf << "Star maginitude:" << endl;
      *fLog << inf << " Expected \t" << setw(4) << fMagExp << endl;
      *fLog << inf << " Calcultated \t " << setw(4) << fMagCalc << endl;
      *fLog << inf << " Fitted \t " << setw(4) << fMagFit << endl;
    }
  
  if (o.Contains("max", TString::kIgnoreCase) || opt == NULL)
    {
      *fLog << inf << "Star Maximum:" << endl;
      *fLog << inf << " Calcultated \t " << setw(4) << fMaxCalc << " uA" << endl;
      *fLog << inf << " Fitted \t " << setw(4) << fMaxFit << " uA" << endl;
    }
  
  if (o.Contains("pos", TString::kIgnoreCase) || opt == NULL)
    {
      *fLog << inf << "Star position:" << endl;
      *fLog << inf << " Expected \t X " << setw(4) << fXExp << " mm \tY " << setw(4) << fYExp << " mm" << endl;
      *fLog << inf << " Calcultated \t X " << setw(4) << fMeanXCalc << " mm \tY " << setw(4) << fMeanYCalc << " mm" << endl;
      *fLog << inf << " Fitted \t X " << setw(4) << fMeanXFit << " mm \tY " << setw(4) << fMeanYFit << " mm" << endl;
    }

  if (o.Contains("siz", TString::kIgnoreCase) || opt == NULL)
    {
      *fLog << inf << "Star size:" << endl;
      *fLog << inf << " Calcultated \t X " << setw(4) << fSigmaMinorAxisCalc << " mm \tY " << setw(4) << fSigmaMajorAxisCalc << " mm" << endl;
      *fLog << inf << " Fitted \t X " << setw(4) << fSigmaMinorAxisFit << " mm \tY " << setw(4) << fSigmaMajorAxisFit << " mm" << endl;
    }

  if (o.Contains("chi", TString::kIgnoreCase) || opt == NULL)
    {
      *fLog << inf << "Star Fit Quality:" << endl;
      *fLog << inf << " ChiSquare/Ndof \t " << setw(3) << fChiSquare << "/" << fNdof << endl;
    }
}
