/***************************************************************************
                          videodisplay.cpp  -  description
                             -------------------
    begin                : Sat Apr 5 2003
    copyright            : (C) 2003 by Martin Merck
    email                : merck@astro.uni-wuerzburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "amcerror.h"
#include "amclog.h"
#include "amcmotor.h"
#include "amcdefs.h"
#include "amcmirrorpanel.h"
#include "amcprogdlg.h"
#include "amcframegrabber.h"
#include "calibratepaneldialog.h"
#include "centerthread.h"
#include "magicmirror.h"
#include "threadcontroller.h"	// needed for the definition of DEFOCUS_THREAD_CTRL
#include "threadevent.h"			// needed for the definition of thread events
#include "videodisplay.h"
#include "videoframe.h"
#include <qlayout.h>
#include <qtimer.h>
#include <qslider.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qmessagebox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <kapp.h>
#include <math.h>
#include <unistd.h>

#define MAX(a,b) ((a>b) ? a : b )

extern KApplication*			g_theApp;
extern AMCLog*						g_pLog;
extern AMCFrameGrabber*		g_pFG;
extern MagicMirror*				g_theMirror;
extern int								g_iRefX, g_iRefY;

VideoDisplay::VideoDisplay(QWidget *parent, const char *name, bool modal )
		: VideoDisplayBase(parent, name, modal)
{
//	m_theFG = new AMCFrameGrabber();

  qDebug("Thread for VideoDisplay PID: %d - PPID: %d", getpid(), getppid() );

	Brightness->setValue( g_pFG->getBrightness() );
	Contrast->setValue( g_pFG->getContrast() );
	connect( Brightness, SIGNAL( valueChanged(int) ), g_pFG ,SLOT( setBrightness( int ) ) );
	connect( Contrast, SIGNAL( valueChanged(int) ), g_pFG ,SLOT( setContrast( int ) ) );

  qDebug("Creating Videoframe");
	m_videoFrame = new VideoFrame( this, "Video", g_pFG );
  m_videoFrame->setMinimumSize( QSize( MY_WIDTH, MY_HEIGHT ) );
  m_videoFrame->setMaximumSize( QSize( MY_WIDTH, MY_HEIGHT ) );
  m_videoFrame->setFrameShape( QFrame::NoFrame );
  qDebug("Adding Videoframe as widget");
	((QGridLayout*) this->layout())->addWidget( m_videoFrame, 0, 0 );
	connect( m_videoFrame, SIGNAL( gref(int, int ) ), this ,SLOT( setGRefInfoSlot( int, int ) ) );
	connect( m_videoFrame, SIGNAL( spot(int, int ) ), this ,SLOT( setSpotInfoSlot( int, int ) ) );
	connect( m_videoFrame, SIGNAL( framesGrabbed(int) ), (QObject*) GrabbedLCD ,SLOT( display( int ) ) );
	connect( m_videoFrame, SIGNAL( framesProcessed(int) ), (QObject*) ProcessedLCD ,SLOT( display( int ) ) );
	connect( thresholdSlider, SIGNAL( valueChanged(int) ), m_videoFrame ,SLOT( threshold(int) ) );
  thresholdSlider->setValue(15);
	connect( scaleBox, SIGNAL( toggled(bool) ), m_videoFrame ,SLOT( scaleImage(bool) ) );
	connect( scaleValSlider, SIGNAL( valueChanged(int) ), m_videoFrame ,SLOT( scaleValue(int) ) );
  scaleValSlider->setValue(200);
	connect( darksBox, SIGNAL( toggled(bool) ), m_videoFrame ,SLOT( substractDarks(bool) ) );
	connect( Take_Dark_Button, SIGNAL( clicked() ), m_videoFrame ,SLOT( takeDark() ) );
	connect( Clear_Darks_Button, SIGNAL( clicked() ), m_videoFrame ,SLOT( clearDarks() ) );

	// Constructing a timer. The destructor of the parent of the timer, in this case "this"
	// VideoDisplay will automatically destroy the timer so we do not need to delete it
	// ourselves.
  qDebug("Creating and starting timer");
	m_pTimer = new QTimer( this );
	connect( m_pTimer, SIGNAL( timeout() ), m_videoFrame, SLOT( timerDone() ) );
	m_pTimer->start( 1000 );        	// start 1 seconds timer

}

VideoDisplay::~VideoDisplay()
{
	disconnect( Brightness, SIGNAL( valueChanged(int) ), g_pFG ,SLOT( setBrightness( int ) ) );
	disconnect( Contrast, SIGNAL( valueChanged(int) ), g_pFG ,SLOT( setContrast( int ) ) );
	disconnect( m_pTimer, SIGNAL( timeout() ), m_videoFrame, SLOT( timerDone() ) );
	delete m_pTimer;
	qDebug("Deleting VideoFrame");
	delete m_videoFrame;
	qDebug("Deleting VideoDisplay");
}

void VideoDisplay::storeSlot(){
	m_videoFrame->saveFrame();
}

/** No descriptions */
void VideoDisplay::setSpotInfoSlot( int p_iX, int p_iY )
{
	SpotX->setText( QString::number(p_iX) );
	SpotY->setText( QString::number(p_iY) );
}

/** No descriptions */
void VideoDisplay::setGRefInfoSlot( int p_iX, int p_iY )
{
	GRefX->setText( QString::number( p_iX ) );
	GRefY->setText( QString::number( p_iY ) );
}

/** Set the motor of the panel to use.
 */
void VideoDisplay::setMotor( AMCMotor* p_pMotor )
{
	m_pMotor = p_pMotor;
}

/** Set the motor of the panel to use.
 */
void VideoDisplay::setPanel( AMCMirrorPanel* p_pPanel )
{
	m_pPanel = p_pPanel;
	QString info;
	Panel_Info->setTitle( info.sprintf( "Panel: %d,%d", m_pPanel->i(), m_pPanel->j() ) );
	Device_Label->setText( info.sprintf( "Device: /dev/ttyS%d", m_pPanel->port()+3 ) );
	Box_Label->setText( info.sprintf( "Box: %02d", m_pPanel->box() ) );
	Driver_Label->setText( info.sprintf( "Driver: %d", m_pPanel->driver() ) );
	Laser_Check->setChecked( m_pPanel->isLaserOn() );

	RefX->setText( QString::number(	m_pPanel->getLaserX() ) );
	RefY->setText( QString::number( m_pPanel->getLaserY() ) );
	m_videoFrame->setRef( m_pPanel->getLaserX(), m_pPanel->getLaserY() );
	GRefX->setText( QString::number( g_iRefX ) );
	GRefY->setText( QString::number( g_iRefY ) );

  enableButtons( true );
}

/** Switch the laser on. */
void VideoDisplay::switchLaserSlot( bool p_zOn )
{
	try
	{
		m_pMotor->switchLaser( p_zOn );
		m_pPanel->setLaserOn( p_zOn );
	}
	catch( AMCError& e )
	{
		QMessageBox::information( this,
								  "AMCError",
  							  	  e.getErrorText(),
  							  	  QMessageBox::Ok | QMessageBox:: Default );
	}		
}

/** Center the motors. */
void VideoDisplay::centerSlot()
{
//	m_videoFrame->halt( true );
	enableButtons( false );
	g_theApp->processEvents();
	qDebug("Creating center thread");
	m_pCenterThread = new CenterThread( this );
	m_pCenterThread->setPanel( m_pPanel );
	qDebug("Starting center thread");
	m_pCenterThread->start();
/*	try
	{
		m_pMotor->centerMotors();
		m_pMotor->waitForMotors( m_pMotor->calcTimeout( 17000 )+2 );
	}
	catch( AMCError& e )
	{
		QMessageBox::information( this,
								  "AMCError",
  							  	  e.getErrorText(),
  							  	  QMessageBox::Ok | QMessageBox:: Default );
		enableButtons( true );
		m_videoFrame->halt( false );
		return;
	}
	m_pPanel->setX( 0 );
	m_pPanel->setY( 0 );
	enableButtons( true );
	m_videoFrame->halt( false );
//	g_theApp->processEvents();
*/
}

/** Goto the position given by X. */
void VideoDisplay::goXSlot()
{
  bool zOk;
	int iX = X_Goto_Value->text().toInt( &zOk );
	moveMotors( iX - m_pPanel->getX(), 0 );
}

/** Goto the position given by Y. */
void VideoDisplay::goYSlot()
{
  bool zOk;
	int iY = Y_Goto_Value->text().toInt( &zOk );
	moveMotors( 0, iY - m_pPanel->getY() );
}

/** Move motors relative by the given amount of steps. */
void VideoDisplay::moveMotors( int p_iX, int p_iY )
{
	if( (p_iX == 0) && (p_iY == 0) )
		return;
	
	enableButtons( false );
	m_videoFrame->halt( true );
	g_theApp->processEvents();
	try
	{
		m_pMotor->moveMotors( p_iX, p_iY );
		m_pMotor->waitForMotors( m_pMotor->calcTimeout( MAX( abs(p_iX), abs(p_iY) ) ) );			
	
		int x = m_pPanel->getX() + p_iX;
		int y = m_pPanel->getY() + p_iY;
		m_pPanel->setX( x );
		m_pPanel->setY( y );
	}
	catch( AMCError& e )
	{
		QMessageBox::information( this,
								  "AMCError",
  							  	  e.formatError(),
  							  	  QMessageBox::Ok | QMessageBox:: Default );
	}
	m_videoFrame->halt( false );
	enableButtons( true );
	return;
	
}


/** No descriptions */
void VideoDisplay::enableButtons( bool p_zEnable ) const
{
	Laser_Check->setEnabled( p_zEnable );
	Calibrate_Button->setEnabled( p_zEnable );
	DefocusButton->setEnabled( p_zEnable );
	FocusButton->setEnabled( p_zEnable );
	DefocusAllButton->setEnabled( p_zEnable );
	
	X_Inc_100_Button->setEnabled( p_zEnable );
	X_Inc_10_Button->setEnabled( p_zEnable );
	X_Goto_Value->setEnabled( p_zEnable );
	X_Go_Button->setEnabled( p_zEnable );
	X_Dec_10_Button->setEnabled( p_zEnable );
	X_Dec_100_Button->setEnabled( p_zEnable );
	
	Y_Inc_100_Button->setEnabled( p_zEnable );
	Y_Inc_10_Button->setEnabled( p_zEnable );
	Y_Goto_Value->setEnabled( p_zEnable );
	Y_Go_Button->setEnabled( p_zEnable );
	Y_Dec_10_Button->setEnabled( p_zEnable );
	Y_Dec_100_Button->setEnabled( p_zEnable );

	if( p_zEnable )
	{
		int x = m_pPanel->getX();
		int y = m_pPanel->getY();
		
		QString str;
		X_Mot_Box->setTitle( str.sprintf("X=%d",x ) );
		Y_Mot_Box->setTitle( str.sprintf("Y=%d",y ) );
		X_Inc_100_Button->setEnabled( (x > 2400) ? false : true );
		X_Inc_10_Button->setEnabled( (x > 2490) ? false : true );
		X_Dec_10_Button->setEnabled( (x < -2490) ? false : true );
		X_Dec_100_Button->setEnabled( (x < -2400) ? false : true );
		
		Y_Inc_100_Button->setEnabled( (y > 2400) ? false : true );
		Y_Inc_10_Button->setEnabled( (y > 2490) ? false : true );
		Y_Dec_10_Button->setEnabled( (y < -2490) ? false : true );
		Y_Dec_100_Button->setEnabled( (y < -2400) ? false : true );

	}
	
	Center_Button->setEnabled( p_zEnable );
}


/** Save actual laser position as reference. */
void VideoDisplay::saveRefSlot()
{
	int x = SpotX->text().toInt();
	int y = SpotY->text().toInt();

	if( x != -1)
	{
		m_pPanel->setLaserX( x );
		m_pPanel->setLaserY( y );
		m_pPanel->setRefX( m_pPanel->getX() );
		m_pPanel->setRefY( m_pPanel->getY() );
		RefX->setText( QString::number( x ) );
		RefY->setText( QString::number( y ) );
		m_videoFrame->setRef( x, y );
	}
}

/** Save laser position as global reference. */
void VideoDisplay::saveGRefSlot()
{
	int x = SpotX->text().toInt();
	int y = SpotY->text().toInt();

	if( x != -1)
	{
		g_iRefX = x;
		g_iRefY = y;
		GRefX->setText( QString::number( x ) );
		GRefY->setText( QString::number( y ) );
	}
}

/** Adjust laser spot to the reference position of this panel */
void VideoDisplay::adjustRefSlot()
{
	int iRefX = m_pPanel->getLaserX();
	int iRefY = m_pPanel->getLaserY();
  qDebug("Reference: %d %d", iRefX, iRefY );
	if( ( iRefX == -1 ) && ( iRefY == -1 ) )
		return;
	int iDx, iDy;
	iDx = m_videoFrame->m_iCenterX - CAMERA_X;
	iDy = m_videoFrame->m_iCenterY - CAMERA_Y;
	iRefX += iDx;
	iRefY += iDy;
	int iStepX, iStepY;
	calcSteps( iRefX, iRefY, iStepX, iStepY );
	moveMotors( iStepX, iStepY );
}

/** Adjust laser spot to the global reference position */
void VideoDisplay::adjustGRefSlot()
{
	if( ( g_iRefX == -1 ) && ( g_iRefY == -1 ) )
		return;
	int iStepX, iStepY;
	calcSteps( g_iRefX, g_iRefY, iStepX, iStepY );
	moveMotors( iStepX, iStepY );
}

void VideoDisplay::calcSteps( int p_iRefX, int p_iRefY, int& p_iStepX, int& p_iStepY )
{
	p_iStepX = p_iStepY = 0;
	int iVSpotX = SpotX->text().toInt();
	int iVSpotY = SpotY->text().toInt();
	if( ( iVSpotX == -1 ) && ( iVSpotY == -1 ) )
		return;
	qDebug("Spot at x,y: %4d %4d", iVSpotX, iVSpotY );
	qDebug("Ref  at x,y: %4d %4d", p_iRefX, p_iRefY );

	double dX = p_iRefX - iVSpotX;
	double dY = p_iRefY - iVSpotY;
	double dAlpha1 = atan( m_pPanel->getSlopeX() );
	double dAlpha2 = atan( m_pPanel->getSlopeY() );
	double dSinA1 = sin( dAlpha1 );
	double dSinA2 = sin( dAlpha2 );
	double dCosA1 = cos( dAlpha1 );
	double dCosA2 = cos( dAlpha2 );

	double dFacX = dX - ( dY * dCosA2 / dSinA2 );
	dFacX /= ( dCosA1 - dSinA1 * dCosA2 / dSinA2 );
	double dFacY = (dY - dFacX * dSinA1) / dSinA2;

	p_iStepX = (int) ( dFacX * m_pPanel->getConversionX() );
	p_iStepY = (int) ( dFacY * m_pPanel->getConversionY() );

	qDebug("Steps till reference: %4d %4d", p_iStepX, p_iStepY );	
}

/** No descriptions */
void VideoDisplay::defocusAllSlot()
{
	enableButtons( false );
  QString qsMsg("Defocusing all mirrors");
	AMCProgDlg* pDialog = new AMCProgDlg( DEFOCUS_THREAD_CTRL, this, qsMsg, true );
	pDialog->setText( qsMsg );
	pDialog->start();
	pDialog->exec();
	delete pDialog;

	try
	{
		// reselect the currently selected panel
		m_pMotor->unselectBox();
		int iBox = m_pPanel->box();
		m_pMotor->selectBox( iBox );
		int iDriver = m_pPanel->driver();
		m_pMotor->selectDriver( iDriver );
	}
	catch( AMCError& e )
	{
		QMessageBox::information( this,
								  "AMCError",
 						  	  e.formatError(),
 						  	  QMessageBox::Ok | QMessageBox:: Default );
	}
	enableButtons( true );
	return;
}

/** No descriptions */
void VideoDisplay::defocusSlot()
{
	int x = m_pPanel->getRefX();
	int y = m_pPanel->getRefY();

	if( x >= 0 )
		x -= 1500;
	else
		x += 1500;

	if( y >= 0 )
		y -= 1500;
	else
		y += 1500;

	x -= m_pPanel->getX();
	y -= m_pPanel->getY();
	moveMotors( x, y );

  return;
}

/** No descriptions */
void VideoDisplay::focusAllSlot()
{
	enableButtons( false );

  QString qsMsg("Focusing all mirrors");
	AMCProgDlg* pDialog = new AMCProgDlg( DEFOCUS_THREAD_CTRL, this, qsMsg, true );
	pDialog->setText( qsMsg );
	pDialog->start();
	pDialog->exec();
	delete pDialog;

	try
	{
		// reselect the currently selected panel
		m_pMotor->unselectBox();
		int iBox = m_pPanel->box();
		m_pMotor->selectBox( iBox );
		int iDriver = m_pPanel->driver();
		m_pMotor->selectDriver( iDriver );
	}
	catch( AMCError& e )
	{
		QMessageBox::information( this,
								  "AMCError",
 						  	  e.formatError(),
 						  	  QMessageBox::Ok | QMessageBox:: Default );
	}
	enableButtons( true );
	return;
}

/** No descriptions */
void VideoDisplay::focusSlot()
{
	int x = m_pPanel->getRefX();
	int y = m_pPanel->getRefY();

	x -= m_pPanel->getX();
	y -= m_pPanel->getY();
	moveMotors( x, y );

  return;
}

/** No descriptions */
void VideoDisplay::takeDarkSlot()
{
//	m_videoFrame->m_zTakeDark = true;
}

/** No descriptions */
void VideoDisplay::clearDarksSlot(){
	m_videoFrame->clearDarks();
}

void VideoDisplay::customEvent(QCustomEvent *e)
{
  if ( e->type() == THREAD_EVENT )
	{
		ThreadEvent* te = (ThreadEvent*)e;
	}

  if ( e->type() == THREAD_ERROR_EVENT )
	{
		ThreadErrorEvent* tee = (ThreadErrorEvent*)e;
		QMessageBox::information( this,
								  "AMCError",
  							  tee->getMessage(),
  							  QMessageBox::Ok | QMessageBox:: Default );
		m_pCenterThread->wait();		
		qDebug("Deleting center thread");
		delete m_pCenterThread;
		enableButtons( true );
		g_pLog->logError( tee->getMessage(), tee->getError() );
	}

  if ( e->type() == THREAD_END_EVENT )
	{
		m_pCenterThread->wait();		
		delete m_pCenterThread;
		enableButtons( true );
	}

}
/** No descriptions */
void VideoDisplay::calibrateSlot()
{
  qDebug("Stoping VideoFrame");
	m_videoFrame->halt( true );
  qDebug("Disabling buttons");
	enableButtons( false );
	CalibratePanelDialog* pDlg = new CalibratePanelDialog( this, "Calibrate Panel", g_pFG );
	pDlg->setPanel( m_pPanel );
	pDlg->setTSigmas( thresholdSlider->value() );
	int iRet = pDlg->exec();
	Laser_Check->setChecked( m_pPanel->isLaserOn() );
	delete pDlg;
  qDebug("Enabling buttons");
	enableButtons( true );
  qDebug("Restarting VideoFrame");
	m_videoFrame->halt( false );
}

/** No descriptions */
void VideoDisplay::accept()
{
	g_theMirror->saveMirrorPanels();
	VideoDisplayBase::accept();
}
