/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): David Paneque 02/2004 <mailto:dpaneque@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MTSupercutsApplied
//
// Class derived from TObject and containing a TTree object 
// used to store the values of the Supercuts applied 
// to each single event. The supercuts are dynamic, which means that 
// the cuts depend on the characterics (size, theta, distance) of the 
// individual events. 
// For the time being, only Length, Width and Dist are used, yet in 
// the future more parameters will be added (conc, asym...)
//
/////////////////////////////////////////////////////////////////////////////

#include "MTSupercutsApplied.h"

#include <TObject.h>

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MTSupercutsApplied);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor.
//
MTSupercutsApplied::MTSupercutsApplied(const char* name, const char* title)
{
    fName  = name  ? name  : "MTSupercutsApplied";
    fTitle = title ? title : "Storage container for the dynamical supercuts";


    // TTree object is initialized with same name and titles as parent object
    
    fRootTree = new TTree (name, title);

    fBranchesCreated = kFALSE;

}

// --------------------------------------------------------------------------
//   DESTRUCTOR deletes dynamic memory allocated for TTree and vectors

MTSupercutsApplied::~MTSupercutsApplied()
{

    // If I release the dynamic memory ocupied by this guys a segmentation 
    // fault is produced. I guess this is due to the fact that tryis to 
    // remove something which has been writtenn into a root file... 
    // not sure...

    // Anyhow, this occupies very little memory... it will not load 
    // too much the system.

    //delete fCutParameters;
    //delete fShowerParamBranchName;
    //delete fRootTree;
    

}




// Function that creates the branches of the TTree object

Bool_t MTSupercutsApplied::CreateTreeBranches()
{

    // Names of the branches is fixed for the time being
    
    fCutParamBranchName = ("SupercutsApplied");
    fShowerParamBranchName = ("ShowerParameters");

   
    // Number of parameters for the 2 branches is set in cosntructor. 
    // In later versions I might do it in such a way that can be set 
    // by user

    fNCutParameters = 13;
    fNShowerParameters = 10;

    
    const Int_t dimscparam = fNCutParameters;
    const Int_t dimshowerparam = fNShowerParameters;

    fCutParameters = new Double_t [dimscparam];
    fShowerParameters = new Double_t [dimshowerparam];
    


    // Cut parameters names and types for branch CutParameters
    // will be stored in a member data TString fCutParametersNamesTypes
    // Eventually might be set from outside the class

    TString StringVector[dimscparam];
    StringVector[0] = ("LengthUp/D");
    StringVector[1] = ("LengthLow/D");
    StringVector[2] = ("WidthUp/D");
    StringVector[3] = ("WidthLow/D");
    StringVector[4] = ("DistUp/D");
    StringVector[5] = ("DistLow/D");
    StringVector[6] = ("AsymUp/D");
    StringVector[7] = ("AsymLow/D");
    StringVector[8] = ("ConcUp/D");
    StringVector[9] = ("ConcLow/D");
    StringVector[10] = ("LeakageUp/D");
    StringVector[11] = ("LeakageLow/D");
    StringVector[12] = ("Hadronness/D");

    fCutParametersNamesTypes = StringVector[0];
    for (Int_t i = 1; i < dimscparam; i++)
    {
	fCutParametersNamesTypes += (":");
	fCutParametersNamesTypes += StringVector[i];
    }
	

    // Cut parameters names and types for branch CutParameters
    // will be stored in a member data TString fShowerParametersNamesTypes
    // Eventually might be set from outside the class
	
    TString StringVector2[dimshowerparam];

    StringVector2[0] = ("Length/D");
    StringVector2[1] = ("Width/D");
    StringVector2[2] = ("Dist/D");
    StringVector2[3] = ("NewDist/D");
    StringVector2[4] = ("Asym/D");
    StringVector2[5] = ("Conc/D");
    StringVector2[6] = ("Leakage/D");
    StringVector2[7] = ("Theta/D");
    StringVector2[8] = ("Size/D");
    StringVector2[9] = ("Alpha/D");


 
    fShowerParametersNamesTypes = StringVector2[0];
    for (Int_t i = 1; i < dimshowerparam; i++)
    {
	fShowerParametersNamesTypes += (":");
	fShowerParametersNamesTypes += StringVector2[i];
    }


    *fLog << "MTSupercutsApplied::CreateTreeBranches()" << endl
	  << "Branches created are the following ones (BranchName; Parameters)" << endl
	  << fCutParamBranchName << " ; " << fCutParametersNamesTypes << endl
	  << fShowerParamBranchName << " ; " << fShowerParametersNamesTypes << endl;

    
    // Branches are finally created

    fRootTree -> Branch(fCutParamBranchName.Data(),
			fCutParameters, fCutParametersNamesTypes.Data());

    fRootTree -> Branch(fShowerParamBranchName.Data(),
			fShowerParameters, fShowerParametersNamesTypes.Data());

    fBranchesCreated = kTRUE;


    /*

    fRootTree -> Branch("LengthUp", &fLengthUp, "LengthUp/D");
    fRootTree -> Branch("LengthLow", &fLengthLow, "LengthLow/D");
    fRootTree -> Branch("WidthUp", &fWidthUp, "WidthUp/D");
    fRootTree -> Branch("Widthlow", &fWidthLow, "WidthLow/D");
    fRootTree -> Branch("DistUp", &fDistUp, "DistUp/D");
    fRootTree -> Branch("DistLow", &fDistLow, "DistLow/D");

    */


    *fLog << "MTSupercutsApplied::CreateTreeBranches();" << endl
	  << "Branches created successfully in TTree object" << endl;

    return kTRUE;
    
}

Bool_t MTSupercutsApplied::FillTreeBranches(const TArrayD cutparameters, 
					    const TArrayD showerparameters)
{

    // Check that branches have been created 
    if(!fBranchesCreated)
    {
	*fLog << "MTSupercutsApplied::FillTreeBranches" << endl
	      << "Branches for the Tree object have not been created yet..." 
	      << "Therefore Tree can not be filled" << endl;
	
	return kFALSE;

    }

    // Check that dimensions of the argument vectors match with the
    // dimension of the vectors whose addresses are given to the branches

    if (fNCutParameters != cutparameters.GetSize()
	|| fNShowerParameters != showerparameters.GetSize())
    {
	*fLog << "MTSupercutsApplied::FillTreeBranches" << endl
	      << "Dimension of arrays used in the argument of this function " 
	      << "do not match with the dimension of the arrays used in the branches"
	      << endl;

	return kFALSE;
    }

    // Parameters are transferred to the member data vectors 

    for (Int_t i = 0; i < fNCutParameters; i++)
    {
	fCutParameters[i] = double (cutparameters[i]);
    }

    for (Int_t i = 0; i < fNShowerParameters; i++)
    {
	fShowerParameters[i] = double (showerparameters[i]);
    }


    // Branches are finally filled

    fRootTree -> Fill();
    
    return kTRUE;
}

























