/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz 04/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MHBadPixels
//
////////////////////////////////////////////////////////////////////////////
#include "MHBadPixels.h" 

#include <TCanvas.h>

#include "MPointingPos.h"
#include "MBadPixelsCam.h"
#include "MGeomCam.h"
#include "MPedPhotCam.h"
#include "MParList.h"
#include "MBinning.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHBadPixels);

using namespace std;

// -------------------------------------------------------------------------
//
//  Default Constructor.
//
MHBadPixels::MHBadPixels(const char *name, const char *title)
{
    fName  = name  ? name  : "MHBadPixels";
    fTitle = title ? title : "Histogram for Bad Pixels vs. Theta";

    fBadId.SetName("2D-IdBadPixels");
    fBadId.SetTitle("2D-IdBadPixels");
    fBadId.SetDirectory(NULL);
    fBadId.SetXTitle("\\Theta [\\circ]");
    fBadId.SetYTitle("pixel Id");

    fBadN.SetName("2D-NBadPixels");
    fBadN.SetTitle("2D-NBadPixels");
    fBadN.SetDirectory(NULL);
    fBadN.SetXTitle("\\Theta [\\circ]");
    fBadN.SetYTitle("number of bad pixels");

    fNamePedPhotCam = "MPedPhotCamFromData";
}

// --------------------------------------------------------------------------
//
// Set the binnings and prepare the filling of the histogram
//
Bool_t MHBadPixels::SetupFill(const MParList *plist)
{
    MGeomCam *fCam = (MGeomCam*)plist->FindObject(AddSerialNumber("MGeomCam"));
    if (!fCam)    
    {
        *fLog << err << "MHBadPixels::SetupFill; MGeomCam not found... aborting." << endl;
        return kFALSE;
    }

    fPointPos = (MPointingPos*)plist->FindObject("MPointingPos");
    if (!fPointPos)
    {
        *fLog << err << "MPointingPos not found... aborting." << endl;
        return kFALSE;
    }

    
    fPedPhot = (MPedPhotCam*)plist->FindObject(AddSerialNumber(fNamePedPhotCam), "MPedPhotCam");
    if (!fPedPhot)
    {
        *fLog << err << AddSerialNumber(fNamePedPhotCam) 
              << "[MPedPhotCam] not found... aborting." << endl;
        return kFALSE;
    }
    //fPedPhot->InitSize(fCam->GetNumPixels());


    // Get Theta Binning
    MBinning* binstheta  = (MBinning*)plist->FindObject("BinningTheta", "MBinning");
    if (!binstheta)
    {
        *fLog << err << "Object 'BinningTheta' [MBinning] not found... aborting" << endl;
        return kFALSE;
    }

    // Get binning for pixel number
    const UInt_t npix1 = fPedPhot->GetSize()+1;

    MBinning binspix("BinningPixel");
    binspix.SetEdges(npix1, -0.5, npix1-0.5);

    // Set binnings in histograms
    SetBinning(&fBadId, binstheta, &binspix);
    SetBinning(&fBadN,  binstheta, &binspix);

    return kTRUE;
}

// ------------------------------------------------------------------------
// 
// Drawing function. It creates its own canvas.
//
void MHBadPixels::Draw(Option_t *option)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    pad->Divide(2,2);

    TH1D *h;

    pad->cd(1);
    fBadId.Draw(option);

    pad->cd(2);
    fBadN.Draw(option);

    pad->cd(3);
    gPad->SetBorderMode(0);
    h = ((TH2*)&fBadId)->ProjectionY("ProjY-pixId", -1, 9999, "");
    h->SetDirectory(NULL);
    h->SetTitle("Distribution of bad pixel Id");
    h->SetXTitle("Id of bad pixel");
    h->SetYTitle("No. of events");
    h->Draw(option);
    h->SetBit(kCanDelete);

    pad->cd(4);
    gPad->SetBorderMode(0);
    h = ((TH2*)&fBadN)->ProjectionY("ProjY-pixN", -1, 9999, "");
    h->SetDirectory(NULL);
    h->SetTitle("Distribution of no.of bad pixels");
    h->SetXTitle("No. of bad pixels");
    h->SetYTitle("No. of events");
    h->Draw(option);
    h->SetBit(kCanDelete);

    pad->Modified();
    pad->Update();
}

Bool_t MHBadPixels::Fill(const MParContainer *par, const Stat_t w)
{
    if (!par)
        return kFALSE;

    Double_t theta = fPointPos->GetZd();

    const MBadPixelsCam *fBadPixels = (MBadPixelsCam*)par;

    UShort_t entries = fCam->GetNumPixels();
    UInt_t nb = 0;
    for (UInt_t i=0; i<entries; i++)
    {
      if ( (*fBadPixels)[i].IsUnsuitable() )
      {
          fBadId.Fill(theta, i, w);
          nb++;
      }   
    }
    fBadN.Fill(theta, nb, w);

    return kTRUE;
}












