/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 08/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!         
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
//////////////////////////////////////////////////////////////////////////////
//
//  MCalibColorSet
//
//  Sets the color for events depending on the color which was used for
//  the run. This is a workaround for runs which were taken before the
//  digital module could deliver the color 'per event'
// 
//  Input Containers:
//   MRawRunHeader
//
//  Output Containers:
//   MRawEvtHeader
//
//////////////////////////////////////////////////////////////////////////////
#include "MCalibColorSet.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRawEvtHeader.h"
#include "MRawRunHeader.h"

ClassImp(MCalibColorSet);

using namespace std;

const Int_t MCalibColorSet::gkIFAEBoxInaugurationRun = 20113;

// --------------------------------------------------------------------------
//
//  Default constructor. MGeomCamMagic is the default geometry.
//
MCalibColorSet::MCalibColorSet(const char *name, const char *title)
    : fHeader(0), fIsValid(kFALSE)
{
    fName  = name  ? name  : "MCalibColorSet";
    fTitle = title ? title : "Task to set workaround missing colors calibration events";
}

Int_t MCalibColorSet::PreProcess(MParList *pList)
{
    fHeader = (MRawEvtHeader*)pList->FindObject("MRawEvtHeader");
    if (!fHeader)
    {
       *fLog << err << "MRawEvtHeader not found... abort." << endl;
        return kFALSE;
    }
    return kTRUE;
}

// --------------------------------------------------------------------------
//
//  Try to find 'MGeomCam' in the Parameter List. If it is not found,
//  processing is stopped.
//
Bool_t MCalibColorSet::ReInit(MParList *pList)
{
    fIsValid = kFALSE;

    MRawRunHeader *header = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!header)
    {
       *fLog << err << "MRawRunHeader not found... abort." << endl;
        return kFALSE;
    }

    enum { kNONE, kGREEN, kBLUE, kUV, kCT1 };

    Int_t color = kNONE;

    const Int_t num = header->GetRunNumber();
    if (num<gkIFAEBoxInaugurationRun)
    {
        *fLog << inf << "Run taken before inauguration of IFAE-box... using CT1 pulser." << endl;
        color = kCT1;
        return kTRUE;
    }

    switch (num)
    {
    case 26402:
        color = kBLUE;
        break;

    case 30090:
    case 20660:
    case 20661:
    case 26408:
    case 26409:
    case 26412:
    case 26568:
    case 26924:
        color = kGREEN;
        break;

    case 27474:
        *fLog << err << "Sorry, run 27474 was taken with CLOSED LIDS. It should not be used! " << endl;
        return kFALSE;
    }

    if (color!=kNONE)
        *fLog << inf << "Color determined from the run-number... ";
    else
    {
        const TString proj = header->GetProjectName();

        if (proj.Contains("green",TString::kIgnoreCase))
            color = kGREEN;
        if (proj.Contains("blue",TString::kIgnoreCase))
            color = kBLUE;
        if (proj.Contains("uv",TString::kIgnoreCase))
            color = kUV;
        if (proj.Contains("ct1",TString::kIgnoreCase))
            color = kCT1;

        *fLog << inf << "Color determined from project-name (" << proj << ")... ";
    }

    if (color==kNONE)
    {
        *fLog << err << "Sorry, calibration run was taken before the events could be" << endl;
        *fLog << "flagged with a color by the digital modul and no color" << endl;
        *fLog << "could be determined... abort." << endl;
        return kFALSE;
    }

    switch (color)
    {
    case kGREEN: *fLog << "Green."; fPattern = 0; break;
    case kBLUE:  *fLog << "Blue.";  fPattern = 0; break;
    case kUV:    *fLog << "UV.";    fPattern = 0; break;
    case kCT1:   *fLog << "CT1.";   fPattern = 0; break;
    }
    *fLog << endl;

    fIsValid = kTRUE;

    return kTRUE;
}

Int_t MCalibColorSet::Process()
{
    if (fIsValid)
        fHeader->SetCalibrationPattern(fPattern);
    return kTRUE;
}
