/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 08/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!         
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
//////////////////////////////////////////////////////////////////////////////
//
//  MCalibColorSet
//
//  Sets the color for events depending on the color which was used for
//  the run. This is a workaround for runs which were taken before the
//  digital module could deliver the color 'per event'
// 
//  Input Containers:
//   MRawRunHeader
//
//  Output Containers:
//   MRawEvtHeader
//
//////////////////////////////////////////////////////////////////////////////
#include "MCalibColorSet.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRawEvtHeader.h"
#include "MRawRunHeader.h"

ClassImp(MCalibColorSet);

using namespace std;

const Int_t MCalibColorSet::gkIFAEBoxInaugurationRun = 20113;
// --------------------------------------------------------------------------
//
//  Default constructor. MGeomCamMagic is the default geometry.
//
MCalibColorSet::MCalibColorSet(const char *name, const char *title)
    : fHeader(0)
{
  fName  = name  ? name  : "MCalibColorSet";
  fTitle = title ? title : "Task to set workaround missing colors calibration events";

  Clear();
}

void MCalibColorSet::Clear(const Option_t *o)
{
  
  fPattern = 0;
  fIsValid = kFALSE;
  
}


// -----------------------------------------------------------------------------------
//
// The following container are searched for and execution aborted if not in MParList:
//  - MRawEvtHeader
//
Int_t MCalibColorSet::PreProcess(MParList *pList)
{

  fHeader = (MRawEvtHeader*)pList->FindObject("MRawEvtHeader");
  if (!fHeader)
    {
      *fLog << err << "MRawEvtHeader not found... abort." << endl;
      return kFALSE;
    }

  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Search for the following input containers and abort if not existing:
// - MRawRunHeader
//
// If Runnumber < gkIFAEBoxInaugurationRun, set colour pattern: 0
//
// If Runnumber > gkIFAEBoxInaugurationRun, search for colour in 
// the project name: Set colour pattern according to the following 
// convention: 
//    Green: assume slot 1  ( 5 Leds Green)
//    Blue:  assume slot 14 ( 5 Leds Blue )
//    UV:    assume slot 12 ( 5 Leds UV   )
//    CT1:   take  'slot 17'
//
Bool_t MCalibColorSet::ReInit(MParList *pList)
{

  Clear();

  MRawRunHeader *header = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
  if (!header)
    {
      *fLog << err << "MRawRunHeader not found... abort." << endl;
      return kFALSE;
    }
  
  enum { kNONE, kGREEN, kBLUE, kUV, kCT1 };
  
  enum ColorCode_t 
    {
      k01LedGreen  = BIT(15),
      k1LedGreen   = BIT(14),
      k2LedGreen   = BIT(1 ),
      k3LedGreen   = k1LedGreen & k2LedGreen,
      k5LedGreen   = BIT(0 ),
      k1LedUV      = BIT(3 ),
      k2LedUV      = BIT(4 ),
      k3LedUV      = k1LedUV & k2LedUV,
      k5LedUV1     = BIT(11),
      k5LedUV2     = BIT(12),
      k10LedUV     = k5LedUV1 & k5LedUV2,
      k11LedUV     = k10LedUV & k1LedUV,
      k12LedUV     = k10LedUV & k2LedUV,
      k13LedUV     = k10LedUV & k1LedUV,
      k01LedBlue   = BIT(8 ),
      k1LedBlue    = BIT(10),
      k2LedBlue    = BIT(7 ),
      k3LedBlue    = k1LedBlue & k2LedBlue,
      k5LedBlue1   = BIT(13),
      k5LedBlue2   = BIT(2 ),      
      k5LedBlue3   = BIT(5 ),
      k5LedBlue4   = BIT(6 ),
      k10LedBlue   = k5LedBlue1 & k5LedBlue2,
      k15LedBlue   = k10LedBlue & k5LedBlue3,
      k20LedBlue   = k15LedBlue & k5LedBlue4,
      k21LedBlue   = k20LedBlue & k1LedBlue,
      k22LedBlue   = k20LedBlue & k2LedBlue,
      k23LedBlue   = k22LedBlue & k1LedBlue,
      kCT1Pulser   = BIT(16)
    };

  const Int_t num = header->GetRunNumber();

  if (num<gkIFAEBoxInaugurationRun)
    {
      *fLog << inf << "Run taken before inauguration of IFAE-box... using CT1 pulser." << endl;
      fPattern |= kCT1Pulser;
      fIsValid  = kTRUE;
      return kTRUE;
    }
  
  Int_t color = kNONE;

  switch (num)
    {
    case 26402:
    case 22246:
    case 22253:
      color = kBLUE;
      break;
      
    case 30090:
    case 20660:
    case 20661:
    case 26408:
    case 26409:
    case 26412:
    case 26568:
    case 26924:
      color = kGREEN;
      break;
      
    case 27474:
      *fLog << err << "Sorry, run 27474 was taken with CLOSED LIDS. It should not be used! " << endl;
      return kFALSE;
    }
  
  if (color!=kNONE)
    {
      *fLog << inf << "Color determined from the run-number... ";
      switch (color)
        {
        case kGREEN: *fLog << "Green."; fPattern |= k5LedGreen; break;
        case kBLUE:  *fLog << "Blue.";  fPattern |= k5LedBlue1; break;
        }
      *fLog << endl;
      fIsValid  = kTRUE;
      return kTRUE;
    }
  else
    {
      const TString proj = header->GetProjectName();
      
      // Possible green combinations
      if (proj.Contains("0.1ledgree",TString::kIgnoreCase))
        { fPattern |= k01LedGreen; color = kGREEN;    }
      if (proj.Contains("1ledgree",TString::kIgnoreCase))
        { fPattern |= k1LedGreen; color = kGREEN;    }
      if (proj.Contains("2ledgree",TString::kIgnoreCase))
        { fPattern |= k2LedGreen; color = kGREEN;    }
      if (proj.Contains("3ledgree",TString::kIgnoreCase))
        { fPattern |= k3LedGreen; color = kGREEN;    }
      if (proj.Contains("5ledgree",TString::kIgnoreCase))
        { fPattern |= k5LedGreen; color = kGREEN;    }

      // Possible green combinations
      if (proj.Contains("0.1ledblue",TString::kIgnoreCase))
        { fPattern |= k01LedBlue; color = kBLUE;    }
      if (proj.Contains("1ledblue",TString::kIgnoreCase))
        { fPattern |= k1LedBlue; color = kBLUE;    }
      if (proj.Contains("2ledblue",TString::kIgnoreCase))
        { fPattern |= k2LedBlue; color = kBLUE;    }
      if (proj.Contains("3ledblue",TString::kIgnoreCase))
        { fPattern |= k3LedBlue; color = kBLUE;    }
      if (proj.Contains("5ledblue",TString::kIgnoreCase))
        { fPattern |= k5LedBlue1; color = kBLUE;    }
      if (proj.Contains("10ledblue",TString::kIgnoreCase))
        { fPattern |= k10LedBlue; color = kBLUE;    }
      if (proj.Contains("15ledblue",TString::kIgnoreCase))
        { fPattern |= k15LedBlue; color = kBLUE;    }
      if (proj.Contains("20ledblue",TString::kIgnoreCase))
        { fPattern |= k20LedBlue; color = kBLUE;    }
      if (proj.Contains("21ledblue",TString::kIgnoreCase))
        { fPattern |= k21LedBlue; color = kBLUE;    }
      if (proj.Contains("22ledblue",TString::kIgnoreCase))
        { fPattern |= k22LedBlue; color = kBLUE;    }
      if (proj.Contains("23ledblue",TString::kIgnoreCase))
        { fPattern |= k23LedBlue; color = kBLUE;    }
      
      // Possible UV combinations
      if (proj.Contains("1leduv",TString::kIgnoreCase))
        { fPattern |= k1LedUV; color = kUV;    }
      if (proj.Contains("2leduv",TString::kIgnoreCase))
        { fPattern |= k2LedUV; color = kUV;    }
      if (proj.Contains("3leduv",TString::kIgnoreCase))
        { fPattern |= k3LedUV; color = kUV;    }
      if (proj.Contains("5leduv",TString::kIgnoreCase))
        { fPattern |= k5LedUV1; color = kUV;    }
      if (proj.Contains("10leduv",TString::kIgnoreCase))
        { fPattern |= k10LedUV; color = kUV;    }
      if (proj.Contains("11leduv",TString::kIgnoreCase))
        { fPattern |= k11LedUV; color = kUV;    }
      if (proj.Contains("12leduv",TString::kIgnoreCase))
        { fPattern |= k12LedUV; color = kUV;    }
      if (proj.Contains("13leduv",TString::kIgnoreCase))
        { fPattern |= k13LedUV; color = kUV;    }

      if (color != kNONE)
        *fLog << inf << "Color and Intensity determined from project-name (" << proj << ")... ";          
      else
        {
          if (proj.Contains("gree",TString::kIgnoreCase))
            { fPattern |= k5LedGreen; color = kGREEN; }
          if (proj.Contains("blue",TString::kIgnoreCase))
            { fPattern |=k5LedBlue1; color = kBLUE;}
          if (proj.Contains("uv",TString::kIgnoreCase))
            { fPattern |=k5LedUV1  ; color = kUV;  }
          if (proj.Contains("ct1",TString::kIgnoreCase))
            { fPattern |=kCT1Pulser; color = kCT1; }
          if (color != kNONE)
            *fLog << inf << "Color determined from project-name (" << proj << ")... ";
        }
    }
  
  if (color==kNONE)
    {
      *fLog << err << "Sorry, calibration run was taken before the events could be" << endl;
      *fLog << "flagged with a color by the digital modul and no color" << endl;
      *fLog << "could be determined... abort." << endl;
      return kFALSE;
    }
  
  switch (color)
    {
    case kGREEN: *fLog << "Green.";  break;
    case kBLUE:  *fLog << "Blue.";   break;
    case kUV:    *fLog << "UV.";     break;
    case kCT1:   *fLog << "CT1.";    break;
    }
  *fLog << endl;
  
  fIsValid = kTRUE;
  
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Sets the pattern to MRawEvtHeader from outside, if fIsValid is set.
//
Int_t MCalibColorSet::Process()
{

  if (fIsValid)
    fHeader->SetCalibrationPattern(fPattern);
  return kTRUE;
}
