//
// This File contains the definition of the MGCoordinates-class
//
//   Author: Thomas Bretz
//   Version: V1.0 (1-8-2000)

#include "MGVelocity.h"

#include <iostream.h>  // cout

#include <TArc.h>
#include <TWbox.h>
#include <TLine.h>
#include <TText.h>
#include <TList.h>
#include <TGaxis.h>
#include <TArrow.h>
#include <TCanvas.h>

#include "coord.h"

ClassImp(MGVelocity);

void MGVelocity::DrawCoordinateSystem()
{
    TWbox box;
    box.DrawWbox(-145, 145, -35, 120,  18,  2, 1);

    TText text;
    text.SetTextAlign(22);  // centered, centered (s.TAttText)
    text.DrawText(-90., 132.5, fCanvas->GetName());


    TLine line;
    line.DrawLine(-65*2,     0, 65*2,    0);
    line.DrawLine(    0, -65*2,    0, 65*2);

    //
    // Can be replaced by TGaxis axe; in a later root version
    // than 3.01/06. I talked to Rene
    //
    TGaxis *axe;
    axe = new TGaxis(-60*2, 0, 60*2, 0,  -2, 2,  304, "+-N");
    axe->SetTitle("Az"); // \xb0
    axe->SetBit(kCanDelete);
    axe->Draw();

    axe = new TGaxis(0, -60*2, 0, 60*2,  -2, 2,  304, "+-N");
    axe->SetTitle("Zd"); // \xb0
    axe->SetLabelOffset(-0.02);
    axe->SetBit(kCanDelete);
    axe->Draw();
}

void MGVelocity::InitVelocity()
{
    fArrow    = new TArrow(0, 0, 0, 0, 0.01);
    fArrowX   = new TArrow(0, 0, 0, 0, 0.01);
    fArrowY   = new TArrow(0, 0, 0, 0, 0.01);
    fArrowAvg = new TArrow(0, 0, 0, 0, 0.01);

    fArrow->SetLineColor(10);   // white
    fArrowX->SetLineColor(17);  // light gray
    fArrowY->SetLineColor(17);  // light gray
    fArrowAvg->SetLineColor(137);  // light gray

    fArrow->Draw();
    fArrowX->Draw();
    fArrowY->Draw();
    fArrowAvg->Draw();

    fList->Add(fArrow);
    fList->Add(fArrowX);
    fList->Add(fArrowY);
    fList->Add(fArrowAvg);

    fText = new TText(70*2, -70*2, "x1");
    fText->SetTextColor(10);
    fText->SetTextAlign(31); // right, bottom
    fText->Draw();
    fList->Add(fText);
}

MGVelocity::MGVelocity(const TGWindow* p, const char *name, const UInt_t w)
: MGEmbeddedCanvas(name, p, w, 75*2), fPos(-1), fScale(1)
{
    fOld = new XY;
    fAvg = new XY[10];

    DrawCoordinateSystem();
    InitVelocity();

    InitCanvas();

    SetNoContextMenu();
}

MGVelocity::~MGVelocity()
{
    delete fOld;
    delete fAvg;

    //    cout << "MGVelocity destroyed." << endl;
}

void MGVelocity::UpdateText()
{
    char txt[10];

    if (fScale>1)
        sprintf(txt,  "/%.0f", fScale);
    else
        sprintf(txt,  "x%.0f", 1./fScale);

    fText->SetText(fText->GetX(), fText->GetY(), txt);
}

Bool_t MGVelocity::UpdateAvg(const float x, const float y)
{

    //
    // calculate scale factor from avarage over
    // speed, not pixels
    //
    // different scales for Az and Zd
    //
    // check for the number of the value
    //
    //
    const int num = 10;

    //    static int pos = -1;
    //    static XY avg[num];

    if (fPos<0)
        for (int i=1; i<num; i++)
            fAvg[i].Set(x, y);

    fPos++;
    fPos %= num;

    fAvg[fPos].Set(x, y);

    Float_t avgx = 0;
    Float_t avgy = 0;

    for (int i=0; i<num; i++)
    {
        avgx += fAvg[i].X();
        avgy += fAvg[i].Y();
    }

    avgx /= 10.;
    avgy /= 10.;

    avgx *= fScale;
    avgy *= fScale;

    fArrowAvg->SetX2(avgx);
    fArrowAvg->SetY2(avgy);

    if ((fabs(avgx)>/*40.*/110. || fabs(avgy)>/*40.*/110.))
    {
        fScale /= 2;
        return kTRUE;
    }

    if ((fabs(avgx)< 5. && fabs(avgy)<20.) ||
        (fabs(avgx)<20. && fabs(avgy)< 5.))
    {
        fScale *= 2;
        return kTRUE;
    }

    return kFALSE;
}

void MGVelocity::Update(ZdAz &zdaz)
{
    //
    // calculate actual time for planet positions
    //
    float vx = zdaz.Az(); // [U_mot/min]
    float vy = zdaz.Zd(); // [U_mot/min]

    int pixx = (int)(vx*fScale/fPix);
    int pixy = (int)(vy*fScale/fPix);

    //
    // FIXME! Check for the right place!
    //
    Bool_t rc = kFALSE;

    if (pixx || pixy)
        rc = UpdateAvg(vx, vy);

    if (rc)
        UpdateText();

    if (!rc && (int)fOld->X()==pixx && (int)fOld->Y()==pixy)
        return;

    vx *= fScale;
    vy *= fScale;

    fArrow->SetX2(vx);
    fArrow->SetY2(vy);

    fArrowX->SetX2(vx);
    fArrowY->SetY2(vy);

    fOld->Set(pixx, pixy);

    SetModified();
    UpdateCanvas();
}
