/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Robert Wagner  11/2002 <mailto:magicsoft@rwagner.de>
!
!   Copyright: MAGIC Software Development, 2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MVPTime                                                                 //
//                                                                         //
// Time Container used by the visibility plotter storing times in MJD      //
//                                                                         //
// This is a simple time container for the Visibility Plotter. It takes    //
// "normal" or MJD-formatted time, converts it to MJD and can return MJD.  //
//                                                                         //
// The astronomical algorithms used are taken from SLALIB 2.4-8.           //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MVPTime.h"

#include <slalib.h>

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MVPTime);

MVPTime::MVPTime(const char *name, const char *title)
{
    fName  = name  ? name  : "MVPTime";
    fTitle = title ? title : "Storage container for time";      
}

MVPTime::~MVPTime()
{
  // do nothing special.
}

void MVPTime::SetTime(UInt_t yr, UInt_t mo, UInt_t dy, UInt_t hr, UInt_t mi, UInt_t se)
{
  fUT1 = CalendarToUT1(yr, mo, dy, hr, mi, se);  
  fYear = yr;
  fMonth = mo;
  fDay = dy;
}

void MVPTime::SetMJD(Double_t mjd, Double_t fracMjd)
{
  fUT1 = mjd + fracMjd;
  Double_t fd;
  Int_t status;
  slaDjcl (mjd, &fYear, &fMonth, &fDay, &fd, &status);  
}

// --------------------------------------------------------------------------
//
//
Double_t MVPTime::CalendarToUT1(UInt_t yr, UInt_t mo, UInt_t dy, UInt_t hr, UInt_t mi, UInt_t se)
{
  int status;
  Double_t mjd;
  slaCldj(yr, mo, dy, &mjd, &status); 
  switch (status)
    {
    case 1:
      *fLog << "CalendarToMJD Warn: bad year" << endl;
      break;
    case 2:
      *fLog << "CalendarToMJD Warn: bad month" << endl;
      break;
    case 3:
      *fLog << "CalendarToMJD Warn: bad day" << endl;
      break;
    }
  
  mjd += ((Double_t)hr/24) + ((Double_t)mi/(24*60)) + ((Double_t)se/(24*60*60));
  return mjd;
}

// --------------------------------------------------------------------------
//
//
Double_t MVPTime::MJDStartOfYear(UInt_t year)
{
  int status;
  Double_t mjd;
  slaCldj(year, 1, 1, &mjd, &status); 
  switch (status)
    {
    case 1:
      *fLog << "CalendarToMJD Warn: bad year" << endl;
      break;
    case 2:
      *fLog << "CalendarToMJD Warn: bad month" << endl;
      break;
    case 3:
      *fLog << "CalendarToMJD Warn: bad day" << endl;
      break;
    }
  return mjd;
}

void MVPTime::Print(Option_t *) const
{
  *fLog << all;
  *fLog << "Time (MJD) is: " << fUT1 << endl;
}
