/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz 07/2002 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MagicShow
// ---------
//
// Tool to visualize Next Neighbours.
//
// Start the show by:
//   MagicShow show;
//
// Use the following keys:
// -----------------------
//
//   * Space:
//     Toggle between auto increment and manual increment
//
//   * Right/Left:
//     Increment/Decrement pixel number by 1
//
//   * Right/Left:
//     Increment/Decrement pixel number by 1
//
//   * Up/Down:
//     Increment/Decrement pixel number by 10
//
//   * PageUp/PageDown:
//     Increment/Decrement pixel number by 100
//
//   * Home/End:
//     Jump to first/last pixel
//
////////////////////////////////////////////////////////////////////////////
#include "MagicShow.h"

#include <iostream>

#include <KeySymbols.h>

#include <TCanvas.h>
#include <TInterpreter.h>

#include "MHexagon.h"

#include "MGeomPix.h"
#include "MGeomCamCT1.h"
#include "MGeomCamMagic.h"

ClassImp(MagicShow);

using namespace std;

// ------------------------------------------------------------------------
//
// Free all onbects connected to a special camera geometry
//
void MagicShow::Free()
{
    if (!fGeomCam)
        return;

    fPixels->Delete();

    delete fPixels;

    delete fGeomCam;
}

// ------------------------------------------------------------------------
//
// Draw all pixels of the camera
//  (means apend all pixelobjects to the current pad)
//
void MagicShow::DrawHexagons()
{
    for (UInt_t i=0; i<fNumPixels; i++)
    {
        MHexagon &h = (*this)[i];

        h.SetFillColor(kBackground);
        h.Draw();
    }
}

// ------------------------------------------------------------------------
//
// Change camera from Magic to CT1 and back
//
void MagicShow::ChangeCamera()
{
    static Bool_t ct1=kFALSE;

    cout << "Change to " << (ct1?"Magic":"CT1") << endl;

    if (ct1)
        SetNewCamera(new MGeomCamMagic);
    else
        SetNewCamera(new MGeomCamCT1);

    ct1 = !ct1;

    DrawHexagons();
}

// ------------------------------------------------------------------------
//
// Reset/set all veriables needed for a new camera geometry
//
void MagicShow::SetNewCamera(MGeomCam *geom)
{
    Free();

    //
    //  Reset the display geometry
    //
    fW=0;
    fH=0;

    //
    //  Set new camera
    //
    fGeomCam = geom;

    //
    //  create the hexagons of the display
    //
    fNumPixels = fGeomCam->GetNumPixels();
    fRange     = fGeomCam->GetMaxRadius();

    fNumPixel  = fNumPixels-1;

    //
    // Construct all hexagons. Use new-operator with placement
    //
    fPixels = new TClonesArray("MHexagon", fNumPixels);

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        MHexagon &h = *new ((*fPixels)[i]) MHexagon((*fGeomCam)[i]);
#if ROOT_VERSION_CODE > ROOT_VERSION(3,01,06)
        h.SetBit(kNoContextMenu|kCannotPick);
#endif
    }
}

// ------------------------------------------------------------------------
//
//  default constructor
//
MagicShow::MagicShow()
    : fTimer(this, 250, kTRUE), fGeomCam(NULL), fNumPixel(-1), fAuto(kTRUE), fW(0), fH(0)
{
    SetNewCamera(new MGeomCamMagic);

    memset(fText, 0, sizeof(fText));

    //
    // Make sure, that the object is destroyed when the canvas/pad is
    // destroyed. Make also sure, that the interpreter doesn't try to
    // delete it a second time.
    //
    SetBit(kCanDelete);
    gInterpreter->DeleteGlobal(this);

    Draw();

    fTimer.TurnOn();
}

// ------------------------------------------------------------------------
//
// Destructor. Deletes TClonesArrays for hexagons and legend elements.
//
MagicShow::~MagicShow()
{
    Free();

    for (int i=0; i<6; i++)
        if (fText[i])
            delete fText[i];

    if (fDrawingPad->GetListOfPrimitives()->FindObject(this)==this)
    {
        fDrawingPad->RecursiveRemove(this);
        delete fDrawingPad;
    }
}

// ------------------------------------------------------------------------
//
// This is called at any time the canvas should get repainted.
// Here we maintain an aspect ratio of 5/4=1.15. This makes sure,
// that the camera image doesn't get distorted by resizing the canvas.
//
void MagicShow::Paint(Option_t *opt)
{
    const UInt_t w = (UInt_t)(gPad->GetWw()*gPad->GetAbsWNDC());
    const UInt_t h = (UInt_t)(gPad->GetWh()*gPad->GetAbsHNDC());

    //
    // Check for a change in width or height, and make sure, that the
    // first call also sets the range
    //
    if (w*fH == h*fW && fW && fH)
        return;

    //
    // Calculate aspect ratio (5/4=1.25 recommended)
    //
    const Double_t ratio = (Double_t)w/h;

    Float_t x;
    Float_t y;

    if (ratio>1.0)
    {
        x = fRange*(ratio*2-1);
        y = fRange;
    }
    else
    {
        x = fRange;
        y = fRange/ratio;
    }

    fH = h;
    fW = w;

    //
    // Set new range
    //
    fDrawingPad->Range(-fRange, -y, x, y);
}

// ------------------------------------------------------------------------
//
// Call this function to draw the camera layout into your canvas.
// Setup a drawing canvas. Add this object and all child objects
// (hexagons, etc) to the current pad. If no pad exists a new one is
// created.
//
void MagicShow::Draw(Option_t *option)
{
    //
    // if no canvas is yet existing to draw into, create a new one
    //
    /*TCanvas *c =*/ new TCanvas("MagicShow", "Magic Show Next Neighbours", 0, 0, 800, 800);
    //c->ToggleEventStatus();

    fDrawingPad = gPad;
    fDrawingPad->SetBorderMode(0);
    fDrawingPad->SetFillColor(22);

    //
    // Append this object, so that the aspect ratio is maintained
    // (Paint-function is called)
    //
    AppendPad(option);

    //
    // Reset the game pad
    //
    DrawHexagons();

    fShow.SetTextAlign(23);   // centered/bottom
#if ROOT_VERSION_CODE > ROOT_VERSION(3,01,06)
    fShow.SetBit(kNoContextMenu|kCannotPick);
#endif
    fShow.Draw();
}

// ------------------------------------------------------------------------
//
// Update Status text
//
void MagicShow::Update()
{
    TString txt = "Pixels: ";
    txt += fNumPixels;
    txt += "  Pixel: ";
    txt += fNumPixel;

    if (fAuto)
        txt += "  (auto)";

    fShow.SetText(0, fRange, txt);
}

// ------------------------------------------------------------------------
//
// Execute a mouse event on the camera
//
void MagicShow::ExecuteEvent(Int_t event, Int_t keycode, Int_t keysym)
{
    if (event!=kKeyPress)
        return;

    switch (keysym)
    {
    case kKey_Space:
        fAuto = !fAuto;
        Update();
        fDrawingPad->Update();
        return;

    case kKey_Right:
        ChangePixel(+1);
        return;

    case kKey_Left:
        ChangePixel(-1);
        return;

    case kKey_Up:
        ChangePixel(+10);
        return;

    case kKey_Down:
        ChangePixel(-10);
        return;

    case kKey_PageUp:
        ChangePixel(+100);
        return;

    case kKey_PageDown:
        ChangePixel(-100);
        return;

    case kKey_Home:
        ChangePixel(-fNumPixel);
        return;

    case kKey_End:
        ChangePixel(fNumPixels-fNumPixel-1);
        return;
    }
}

// ------------------------------------------------------------------------
//
// Change the shown pixel by add indices
//
void MagicShow::ChangePixel(Int_t add)
{
    MagicShow &This = *this;

    const MGeomPix &pix1=(*fGeomCam)[fNumPixel];
    This[fNumPixel].SetFillColor(kBackground);
    for (int i=0; i<pix1.GetNumNeighbors(); i++)
    {
        This[pix1.GetNeighbor(i)].SetFillColor(kBackground);
        if (!fText[i])
            continue;

        delete fText[i];
        fText[i] = NULL;
    }

    fNumPixel += add;

    if (fNumPixel>=fNumPixels)
        fNumPixel = 0;
    if (fNumPixel<0)
        fNumPixel = fNumPixels-1;

    const MGeomPix &pix2=(*fGeomCam)[fNumPixel];
    This[fNumPixel].SetFillColor(kBlue);
    for (int i=0; i<pix2.GetNumNeighbors(); i++)
    {
        Int_t idx = pix2.GetNeighbor(i);

        This[idx].SetFillColor(kMagenta);

        TString num;
        num += idx;

        fText[i] = new TText(This[idx].GetX(), This[idx].GetY(), num);
        fText[i]->SetTextSize(0.3*This[idx].GetD()/fGeomCam->GetMaxRadius());
        fText[i]->SetTextFont(122);
        fText[i]->SetTextAlign(22);   // centered/centered
        fText[i]->Draw();
    }

    Update();

    fDrawingPad->Update();
}

// ------------------------------------------------------------------------
//
// If automatic is switched on step one pixel forward
//
Bool_t MagicShow::HandleTimer(TTimer *timer)
{
    if (fAuto)
        ChangePixel(+1);

    return kTRUE;
}
