#ifndef __MMcEvt__
#define __MMcEvt__

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif

//
// Version 4: 
//   Added member fFadcTimeJitter
//

class MMcEvt : public MParContainer
{
public:
    //
    //     ParticleId for Monte Carlo simulation
    //
    enum ParticleId_t
    {
        kGAMMA    =  1,
        kPOSITRON =  2,
        kELECTRON =  3,
        kANTIMUON =  5,
        kMUON     =  6,
        kPI0      =  7,
        kNEUTRON  = 13,
        kPROTON =   14,
        kHELIUM =  402,
        kOXYGEN = 1608,
        kIRON   = 5626
    };

private:
  UInt_t      fEvtNumber;
  UShort_t     fPartId;             // Type of particle
  Float_t      fEnergy;             // [GeV] Energy
  Float_t      fThick0;             // [g/cm2]
  Float_t      fFirstTarget;        // []
  Float_t      fZFirstInteraction;  // [cm]

  Float_t fTheta;           // [rad] Theta angle of event 
  Float_t fPhi;             // [rad] Phi angle of event (see class description)

  Float_t fCoreD;           // [cm] Core d pos
  Float_t fCoreX;           // [cm] Core x pos
  Float_t fCoreY;           // [cm] Core y pos
  Float_t fImpact;          // [cm] impact parameter

  // Up to here, the info from the CORSIKA event header. 

  // Telescope orientation: 
  Float_t	fTelescopePhi;    // [rad] (see class description)
  Float_t	fTelescopeTheta;  // [rad]

  // Time of first and last photon:
  Float_t      fTimeFirst;   // [ns]
  Float_t      fTimeLast;    // [ns]

  // 6 parameters and chi2 of the NKG fit to the longitudinal 
  // particle distribution. See CORSIKA manual for explanation,
  // section 4.42 "Longitudinal shower development": 
  //
  Float_t       fLongiNmax;   // [particles]
  Float_t       fLongit0;     // [g/cm2]
  Float_t       fLongitmax;   // [g/cm2]
  Float_t       fLongia;      // [g/cm2]
  Float_t       fLongib;      // []
  Float_t       fLongic;      // [cm2/g]
  Float_t       fLongichi2;

  UInt_t fPhotIni;          // [ph] Initial number of photons
  UInt_t fPassPhotAtm;      // [ph] Passed atmosphere
  UInt_t fPassPhotRef;      // [ph] Passed reflector(reflectivity + effective area)
  UInt_t fPassPhotCone;     // [ph]  Within any valid pixel, before plexiglas
  UInt_t fPhotElfromShower; // [phe] Passed qe, coming from the shower
  UInt_t fPhotElinCamera;   // [phe] usPhotElfromShower + mean of phe
                            // from NSB

  // Now follow the fraction of photons reaching the camera produced by
  // electrons, muons and other particles respectively:

  Float_t  fElecCphFraction;
  Float_t  fMuonCphFraction;
  Float_t  fOtherCphFraction;
  
  Float_t  fFadcTimeJitter;

 public:
  MMcEvt() ;
  
  MMcEvt( UInt_t, UShort_t, Float_t, Float_t, Float_t,
	  Float_t, Float_t, Float_t, Float_t, Float_t, Float_t,
	  Float_t, Float_t, Float_t, Float_t, Float_t, Float_t, 
	  Float_t, Float_t, Float_t, Float_t, Float_t, Float_t, 
	  UInt_t, UInt_t, UInt_t, UInt_t, UInt_t, UInt_t,
	  Float_t, Float_t, Float_t, Float_t) ; 
  
  ~MMcEvt(); 

  void Clear(Option_t *opt=NULL);

  void Fill( UInt_t, UShort_t, Float_t, Float_t, Float_t, 
	     Float_t, Float_t, Float_t, Float_t, Float_t, Float_t,
	     Float_t, Float_t, Float_t, Float_t, Float_t, Float_t,
	     Float_t, Float_t, Float_t, Float_t, Float_t, Float_t,
	     UInt_t, UInt_t, UInt_t, UInt_t, UInt_t, UInt_t,
	     Float_t, Float_t, Float_t, Float_t) ; 

  //virtual void AsciiWrite(ofstream &fout) const;

  void Print(Option_t *opt=NULL) const;

  UInt_t GetEvtNumber() const { return fEvtNumber; }  //Get Event Number
  Short_t GetPartId() const { return fPartId; }       //Get Type of particle
  Float_t GetEnergy() const { return fEnergy; }        //Get Energy

  Float_t GetTheta() const { return fTheta; }          //Get Theta angle
  Float_t GetPhi() const { return fPhi ;  }            //Get Phi angle

/*    Float_t GetCoreD() { return fCoreD; }          //Get Core d pos */
  Float_t GetCoreX() { return fCoreX; }          //Get Core x pos
  Float_t GetCoreY() { return fCoreY; }          //Get Core y pos
  Float_t GetImpact() const { return fImpact;}         //Get impact parameter 

  UInt_t GetPhotIni() { return fPhotIni; }           //Get Initial photons
  UInt_t GetPassPhotAtm() { return fPassPhotAtm;}    //Get Passed atmosphere
  UInt_t GetPassPhotRef() { return fPassPhotRef; }   //Get Passed reflector
  UInt_t GetPassPhotCone() { return fPassPhotCone; } //Get Passed glas
  UInt_t GetPhotElfromShower() { return fPhotElfromShower; }   //Get Passed qe from shower
  UInt_t GetPhotElinCamera() { return fPhotElinCamera; }   //Get Passed qe total
  Float_t GetZFirstInteraction() const { return fZFirstInteraction; }

  Float_t GetTelescopePhi() const { return fTelescopePhi; }
  Float_t GetTelescopeTheta() const { return fTelescopeTheta; }
  Float_t GetOtherCphFraction() const { return fOtherCphFraction; }

  Float_t GetLongiNmax() const { return fLongiNmax; }
  Float_t GetLongia()    const { return fLongia; }
  Float_t GetLongib()    const { return fLongib; }
  Float_t GetLongic()    const { return fLongic; }
  Float_t GetLongichi2() const { return fLongichi2; }
  Float_t GetLongit0()   const { return fLongit0; }
  Float_t GetLongitmax() const { return fLongitmax; }

  Float_t GetFadcTimeJitter() const { return fFadcTimeJitter; }

  void SetPartId(Short_t PartId)
  {fPartId=PartId;}             //Set Type of particle 

  TString GetParticleName() const
  {
      switch (fPartId)
      {
      case kGAMMA:    return "Gamma";
      case kPOSITRON: return "Positron";
      case kELECTRON: return "Electron";
      case kANTIMUON: return "Anti-Muon";
      case kMUON:     return "Muon";
      case kPI0:      return "Pi-0";
      case kNEUTRON:  return "Neutron";
      case kPROTON:   return "Proton";
      case kHELIUM:   return "Helium";
      case kOXYGEN:   return "Oxygen";
      case kIRON:     return "Iron";
      }

      return Form("Id:%d", fPartId);
  }
  TString GetParticleSymbol() const
  {
      switch (fPartId)
      {
      case kGAMMA:    return "\\gamma";
      case kPOSITRON: return "e^{+}";
      case kELECTRON: return "e^{-}";
      case kANTIMUON: return "\\mu^{+}";
      case kMUON:     return "\\mu^{-}";
      case kPI0:      return "\\pi^{0}";
      case kNEUTRON:  return "n";
      case kPROTON:   return "p";
      case kHELIUM:   return "He";
      case kOXYGEN:   return "O";
      case kIRON:     return "Fe";
      }

      return Form("Id:%d", fPartId);
  }
  TString GetEnergyStr() const
  {
      if (fEnergy>1000)
          return Form("%.1fTeV", fEnergy/1000);

      if (fEnergy>10)
          return Form("%dGeV", (Int_t)(fEnergy+.5));

      if (fEnergy>1)
          return Form("%.1fGeV", fEnergy);

      return Form("%dMeV", (Int_t)(fEnergy*1000+.5));
  }

  void SetEnergy(Float_t Energy)
  { fEnergy=Energy; }              //Set Energy 
 
  void SetTheta(Float_t Theta) 
  { fTheta=Theta; }                //Set Theta angle 

  void SetPhi(Float_t Phi) 
  { fPhi=Phi;  }                   //Set Phi angle 
 
  void SetCoreD(Float_t CoreD) 
  { fCoreD=CoreD; }                //Set Core d pos

  void SetCoreX(Float_t CoreX)
  { fCoreX=CoreX; }                //Set Core x pos 

  void SetCoreY(Float_t CoreY ) 
  { fCoreY=CoreY; }                //Set Core y pos

  void SetImpact(Float_t Impact) 
  { fImpact=Impact;}               //Set impact parameter

  // DO NOT USE THIS IS A WORKAROUND!
  void SetTelescopeTheta(Float_t Theta) { fTelescopeTheta=Theta; }
  void SetTelescopePhi(Float_t Phi)     { fTelescopePhi=Phi; }

  
/*    void SetPhotIni(Short_t PhotIni)  */
/*      { fPhotIni=PhotIni; }                 //Set Initial photons */
/*    void SetPassPhotAtm(Short_t PassPhotAtm)  */
/*      { fPassPhotAtm=PassPhotAtm;}         //Set Passed atmosphere */
/*    void SetPassPhotRef(Short_t PassPhotRef)  */
/*      { fPassPhotRef=PassPhotRef ; }       //Set Passed reflector */
/*    void SetPassPhotCone(Short_t PhotCon)  */
/*      { fPassPhotCone=PhotCon; }           //Set Passed glas */


  ClassDef(MMcEvt, 4)  //Stores Montecarlo Information of one event (eg. the energy)

};

#endif
