/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJCalibrateSignalFromOutside
//
// This class is reading the output written by callisto. It calibrates
// signal and time.
//
// The signal and time extractors are read from the callisto-output. In
// pricipal you could overwrite these default using the resource file,
// but this is NOT recommended!
//
/////////////////////////////////////////////////////////////////////////////
#include "MJCalibrateSignalFromOutside.h"

#include <TEnv.h>
#include <TFile.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MDirIter.h"
#include "MRunIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MStatusDisplay.h"

#include "MGeomCam.h"
#include "MHCamEvent.h"
#include "MPedestalCam.h"
#include "MBadPixelsCam.h"

#include "MCalibrationQECam.h"
#include "MCalibrationBlindCam.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationRelTimeCam.h"
#include "MCalibrationChargePINDiode.h"

#include "MReadReports.h"
#include "MReadMarsFile.h"
#include "MRawFileRead.h"
#include "MGeomApply.h"
#include "MMcPedestalCopy.h"
#include "MPointingPosCalc.h"
#include "MPedCalcFromLoGain.h"
#include "MExtractor.h"
#include "MTaskEnv.h"
#include "MCalibrateData.h"
#include "MCalibrateRelTimes.h"
#include "MBadPixelsMerge.h"
#include "MBadPixelsCalc.h"
#include "MBadPixelsTreat.h"
#include "MFillH.h"
#include "MWriteRootFile.h"

ClassImp(MJCalibrateSignalFromOutside);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets fRuns to 0, fExtractor to NULL, fDataCheck to kFALSE
//
MJCalibrateSignalFromOutside::MJCalibrateSignalFromOutside(const char *name, const char *title)
    : fIsDataCheck(kFALSE), fGeometry("MGeomCamMagic")
{
    fName  = name  ? name  : "MJCalibrateSignalFromOutside";
    fTitle = title ? title : "Tool to calibrate data";
}

Bool_t MJCalibrateSignalFromOutside::WriteResult()
{
    if (fPathOut.IsNull())
    {
        *fLog << inf << "No output path specified via SetPathOut - no output written." << endl;
        return kTRUE;
    }

    const TString oname = Form("%s/signal%06d.root", (const char*)fPathOut, fSequence.GetSequence());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "RECREATE");

    *fLog << inf << " - MStatusDisplay..." << flush;
    if (fDisplay && fDisplay->Write()<=0)
    {
        *fLog << err << "Unable to write MStatusDisplay to " << oname << endl;
        return kFALSE;
    }
    *fLog << inf << "ok." << endl;

    return kTRUE;
}

Bool_t MJCalibrateSignalFromOutside::ProcessFile(MPedestalCam &pedcamab, MPedestalCam &pedcamrms, MCalibrationChargeCam &chargecam, MCalibrationQECam &qecam, MCalibrationRelTimeCam &relcam, Byte_t filetype)
{
    // --------------------------------------------------------------------------------

    *fLog << inf;
    fLog->Separator(GetDescriptor());

    *fLog << "Calibrate data from ";
    *fLog << "Runs " << fRuns->GetRunsAsString() << endl;
    *fLog << endl;

    MDirIter iter;

    *fLog << all;
    if (fExtractor)
    {
        *fLog << underline << "Signal Extractor found in calibration file" << endl;
        fExtractor->Print();
        *fLog << endl;
    }
    else
      {
        *fLog << inf << "No Signal Extractor: ExtractSignal in file." << endl;
        return kFALSE;
      }
    
    fExtractor->SetPedestals(&pedcamab);

    // Setup Parlist
    MParList plist;
    plist.AddToList(this); // take care of fDisplay!
    plist.AddToList(&fBadPixels);
    plist.AddToList(&pedcamrms);
    plist.AddToList(&chargecam);
    plist.AddToList(&qecam);
    plist.AddToList(&relcam);

    // Setup Tasklist
    MTaskList tlist;
    plist.AddToList(&tlist);

    MReadReports readreal;
    readreal.AddTree("Events", "MTime.", kTRUE);
    readreal.AddTree("Trigger");
    readreal.AddTree("Camera");
    readreal.AddTree("Drive");
    readreal.AddTree("CC");
    readreal.AddTree("Currents");

    //    MReadMarsFile read("Events");
    //    read.DisableAutoScheme();
    MRawFileRead rawread(NULL);
    if (IsDataCheck())
        rawread.AddFiles(*fRuns);
    else
      {
        //        read.AddFiles(*fRuns);
        readreal.AddFiles(*fRuns);
      }

    MGeomApply             apply; // Only necessary to create geometry
    apply.SetGeometry(fGeometry);
    MBadPixelsMerge        merge(&fBadPixels);

    MPedCalcFromLoGain     pedlo1("MPedCalcMeanFromLoGain");
    pedlo1.SetPedestalUpdate(kTRUE);
    pedlo1.SetPedestalsOut(&pedcamab);

    MPedCalcFromLoGain     pedlo2("MPedCalcRmsFromLoGain");
    pedlo2.SetPedestalUpdate(kTRUE);
    pedlo2.SetPedestalsIn(&pedcamab);
    pedlo2.SetPedestalsOut(&pedcamrms);

    if (fExtractor->InheritsFrom("MExtractTimeAndCharge"))
      pedlo2.SetExtractor((MExtractTimeAndCharge*)fExtractor);

    MMcPedestalCopy        pcopy;
    MTaskEnv taskenv1("ExtractSignal");
    taskenv1.SetDefault(fExtractor);
    MCalibrateData         calib;
    if (filetype==3) // MC file
    {
        calib.SetCalibrationMode(MCalibrateData::kFfactor);
        calib.SetPedestalFlag(MCalibrateData::kRun);
    }
    else
    {
        calib.AddPedestal("MPedestalCam", "MPedPhotFromExtractorRndm");
        calib.AddPedestal("Fundamental");
        calib.SetPedestalFlag(MCalibrateData::kEvent);
    }

    MCalibrateRelTimes     caltm;
    MBadPixelsCalc         bpcal;
    MBadPixelsTreat        treat;

    MHCamEvent evt0(0, "PedFLG",      "Pedestal from Lo Gain;;P [fadc/sl]");
    MHCamEvent evt1(2, "PedRmsFLG",   "Pedestal RMS from Lo Gain;;\\sigma_{p} [fadc/sl]");
    MHCamEvent evt2(0, "Extra'd",     "Extracted Signal;;S [fadc/sl]");
    MHCamEvent evt3(0, "PedPhot",     "Calibrated Pedestal;;P [\\gamma]");
    MHCamEvent evt4(1, "PedRMS",      "Calibrated Pedestal RMS;;\\sigma_{p} [\\gamma]");
    MHCamEvent evt5(0, "Interp'd",    "Interpolated Signal;;S [\\gamma]");
    MHCamEvent evt6(2, "Unsuitable",  "Unsuitable event ratio;;%");
    MHCamEvent evt7(0, "Times",       "Arrival Time;;T [slice]");
    evt0.EnableVariance();
    evt1.EnableVariance();
    evt2.EnableVariance();
    evt3.EnableVariance();
    evt4.EnableVariance();
    evt5.EnableVariance();
    evt7.EnableVariance();

    MFillH fill0(&evt0, &pedcamab,             "FillPedFLG");
    MFillH fill1(&evt1, "MPedestalCam",        "FillPedRmsFLG");
    MFillH fill2(&evt2, "MExtractedSignalCam", "FillExtracted");
    MFillH fill3(&evt3, "MPedPhotCam",         "FillPedPhot");
    MFillH fill4(&evt4, "MPedPhotCam",         "FillPedRMS");
    MFillH fill5(&evt5, "MCerPhotEvt",         "FillInterpolated");
    MFillH fill6(&evt6, "MBadPixelsCam",       "FillUnsuitable");
    MFillH fill7(&evt7, "MArrivalTime",        "FillTimes");

    MWriteRootFile write(2, Form("%s{s/_D_/_Y_}", fPathOut.Data()), fOverwrite);
    // Run Header
    write.AddContainer("MRawRunHeader",       "RunHeaders");
    write.AddContainer("MBadPixelsCam",       "RunHeaders");
    write.AddContainer("MGeomCam",            "RunHeaders");
    // Monte Carlo Headers
    write.AddContainer("MMcTrigHeader",       "RunHeaders", kFALSE);
    write.AddContainer("MMcConfigRunHeader",  "RunHeaders", kFALSE);
    write.AddContainer("MMcCorsikaRunHeader", "RunHeaders", kFALSE);
    // Monte Carlo
    write.AddContainer("MMcEvt",              "Events", kFALSE);
    write.AddContainer("MMcTrig",             "Events", kFALSE);
    // Data
    write.AddContainer("MCerPhotEvt",         "Events");
    write.AddContainer("MPedPhotCam",         "Events");
    write.AddContainer("MTime",               "Events", kFALSE);
    write.AddContainer("MRawEvtHeader",       "Events");
    write.AddContainer("MArrivalTime",        "Events");
    // Slow-Control: Current
    write.AddContainer("MTimeCurrents",       "Currents", kFALSE);
    write.AddContainer("MCameraDC",           "Currents", kFALSE);
    write.AddContainer("MReportCurrents",     "Currents", kFALSE);
    // Slow-Control: Camera
    write.AddContainer("MReportCamera",       "Camera", kFALSE);
    write.AddContainer("MTimeCamera",         "Camera", kFALSE);
    write.AddContainer("MCameraAUX",          "Camera", kFALSE);
    write.AddContainer("MCameraCalibration",  "Camera", kFALSE);
    write.AddContainer("MCameraCooling",      "Camera", kFALSE);
    write.AddContainer("MCameraHV",           "Camera", kFALSE);
    write.AddContainer("MCameraLV",           "Camera", kFALSE);
    write.AddContainer("MCameraLids",         "Camera", kFALSE);
    // Slow-Control: Trigger
    write.AddContainer("MReportTrigger",      "Trigger", kFALSE);
    write.AddContainer("MTimeTrigger",        "Trigger", kFALSE);
    // Slow-Control: Drive
    write.AddContainer("MPointingPos",        "Drive", kFALSE);
    write.AddContainer("MReportDrive",        "Drive", kFALSE);
    write.AddContainer("MTimeDrive",          "Drive", kFALSE);
    // Slow-Control: Central Control
    write.AddContainer("MReportCC",           "CC", kFALSE);
    write.AddContainer("MTimeCC",             "CC", kFALSE);

    // Now setup tasklist for events
    MTaskList tlist2;
    tlist2.AddToList(&apply);
    tlist2.AddToList(&merge);
    if (filetype==3)
        tlist2.AddToList(&pcopy);
    else
    {
        tlist2.AddToList(&pedlo1);
        tlist2.AddToList(&pedlo2);
    }
    tlist2.AddToList(&fill0);
    tlist2.AddToList(&fill1);
    tlist2.AddToList(&taskenv1);
    tlist2.AddToList(&fill2);
    tlist2.AddToList(&calib);
    if (&relcam)
      tlist2.AddToList(&caltm);
    tlist2.AddToList(&fill3);
    tlist2.AddToList(&bpcal);
    tlist2.AddToList(&treat);
    tlist2.AddToList(&fill4);
    tlist2.AddToList(&fill5);
    tlist2.AddToList(&fill6);
    tlist2.AddToList(&fill7);

    // Setup List for Drive-tree
    MPointingPosCalc pcalc;

    // Now setup main tasklist
    tlist.AddToList(IsDataCheck() ? (MTask*)&rawread : (MTask*)&readreal);
    tlist.AddToList(&tlist2, IsDataCheck()?"All":"Events");
    if (!IsDataCheck())
        tlist.AddToList(&pcalc, "Drive");
    tlist.AddToList(&write);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);
    if (!SetupEnv(evtloop))
        return kFALSE;

    // Execute first analysis
    if (!evtloop.Eventloop(fMaxEvents))
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    if (!WriteResult())
        return kFALSE;

    *fLog << all << GetDescriptor() << ": Done." << endl;
    *fLog << endl << endl;

    return kTRUE;
}
