/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Eva Domingo,     12/2004 <mailto:domingo@ifae.es>
!              Wolfgang Wittek, 12/2004 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2005
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//   MHDisp                                                                //
//                                                                         //
//   container holding the histograms for Disp                             //
//   also computes the Chi^2 of the Disp optimization                      // 
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MHDisp.h"

#include <math.h>

#include <TH1.h>
#include <TH2.h>
#include <TProfile.h>
#include <TArrayI.h>
#include <TPad.h>
#include <TCanvas.h>
#include <TStyle.h>

#include "MGeomCam.h"
#include "MSrcPosCam.h"
#include "MHillas.h"
#include "MHillasExt.h"
#include "MNewImagePar.h"
#include "MMcEvt.hxx"
#include "MPointingPos.h"
#include "MImageParDisp.h"

#include "MHMatrix.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHDisp);

using namespace std;

// --------------------------------------------------------------------------
//
// Constructor 
//
MHDisp::MHDisp(const char *imagepardispname,
	       const char *name, const char *title)
  : fImageParDispName(imagepardispname)
{
    fName  = name  ? name  : "MHDisp";
    fTitle = title ? title : "Histograms for Disp";

    fSelectedPos = 1;  // default MHDisp flag (selects Correct Disp srcpos)

    fMatrix = NULL;

    //--------------------------------------------------
    // creating the Disp related histograms

    fHistEnergy   = new TH1F("fHistEnergy", 
	 "log10(Energy)", 50, 1., 3.);
    fHistEnergy->SetDirectory(NULL);
    fHistEnergy->UseCurrentStyle();
    fHistEnergy->SetXTitle("log10(Energy) [GeV]");
    fHistEnergy->SetYTitle("# events");

    fHistSize   = new TH1F("fHistSize", 
	 "log10(Size)", 50, 2., 4.);
    fHistSize->SetDirectory(NULL);
    fHistSize->UseCurrentStyle();
    fHistSize->SetXTitle("log10(Size) [#phot]");
    fHistSize->SetYTitle("# events");

    fHistcosZA   = new TH1F("fHistcosZA", 
	 "cos(Zenith Angle)", 10, 0., 1.);
    fHistcosZA->SetDirectory(NULL);
    fHistcosZA->UseCurrentStyle();
    fHistcosZA->SetXTitle("cos(Theta)");
    fHistcosZA->SetYTitle("# events");

    fSkymapXY = new TH2F("fSkymapXY", 
         "Disp estimated source positions Skymap", 71, -2., 2., 71, -2., 2.);
    fSkymapXY->SetDirectory(NULL);
    fSkymapXY->UseCurrentStyle();
    fSkymapXY->SetXTitle("X Disp source position [deg]");
    fSkymapXY->SetYTitle("Y Disp source position [deg]");

    fHistChi2   = new TH1F("fHistChi2"  , 
         "Distance^2 between Disp and real srcpos", 100, 0., 2.);
    fHistChi2->SetDirectory(NULL);
    fHistChi2->UseCurrentStyle();
    fHistChi2->SetXTitle("Chi^2 = distance^2 Disp to real srcpos [deg^2]");
    fHistChi2->SetYTitle("# events");
    
    fHistDuDv   = new TH2F("fHistDuDv", 
	 "Du vs. Dv (distances between Disp and real srcpos)", 
	 100, -2., 2., 100, -2., 2.);
    fHistDuDv->SetDirectory(NULL);
    fHistDuDv->UseCurrentStyle();
    fHistDuDv->SetXTitle("Dv = transveral distance [deg]");
    fHistDuDv->SetYTitle("Du = longitudinal distance [deg]");

    fHistChi2Energy   = new TH2F("fHistChi2Energy", 
	 "Chi^2 vs. Energy", 50, 1., 3., 100, 0., 2.);
    fHistChi2Energy->SetDirectory(NULL);
    fHistChi2Energy->UseCurrentStyle();
    fHistChi2Energy->SetXTitle("log10(Energy) [GeV]");
    fHistChi2Energy->SetYTitle("Chi^2 = distance^2 Disp to real srcpos [deg^2]");

    fHistChi2Size   = new TH2F("fHistChi2Size", 
	 "Chi^2 vs. Size", 50, 2., 4., 100, 0., 2.);
    fHistChi2Size->SetDirectory(NULL);
    fHistChi2Size->UseCurrentStyle();
    fHistChi2Size->SetXTitle("log10(Size) [#phot]");
    fHistChi2Size->SetYTitle("Chi^2 = distance^2 Disp to real srcpos [deg^2]");

    fHistDuEnergy   = new TH2F("fHistDuEnergy", 
	 "Du vs. Energy", 50, 1., 3., 100, -2., 2.);
    fHistDuEnergy->SetDirectory(NULL);
    fHistDuEnergy->UseCurrentStyle();
    fHistDuEnergy->SetXTitle("log10(Energy) [GeV]");
    fHistDuEnergy->SetYTitle("Du = longitudinal distance Disp to real srcpos [deg]");

    fHistDuSize   = new TH2F("fHistDuSize", 
	 "Du vs. Size", 50, 2., 4., 100, -2., 2.);
    fHistDuSize->SetDirectory(NULL);
    fHistDuSize->UseCurrentStyle();
    fHistDuSize->SetXTitle("log10(Size) [#phot]");
    fHistDuSize->SetYTitle("Du = longitudinal distance Disp to real srcpos [deg]");

    fHistDvEnergy   = new TH2F("fHistDvEnergy", 
	 "Dv vs. Energy", 50, 1., 3., 100, -2., 2.);
    fHistDvEnergy->SetDirectory(NULL);
    fHistDvEnergy->UseCurrentStyle();
    fHistDvEnergy->SetXTitle("log10(Energy) [GeV]");
    fHistDvEnergy->SetYTitle("Dv = transveral distance Disp to real srcpos [deg]");

    fHistDvSize   = new TH2F("fHistDvSize", 
	 "Dv vs. Size", 50, 2., 4., 100, -2., 2.);
    fHistDvSize->SetDirectory(NULL);
    fHistDvSize->UseCurrentStyle();
    fHistDvSize->SetXTitle("log10(Size) [#phot]");
    fHistDvSize->SetYTitle("Dv = transveral distance Disp to real srcpos [deg]");

    fEvCorrAssign   = new TProfile("fEvCorrAssign", 
	 "Fraction events srcpos well assign vs. Size", 50, 2., 4., 0., 1.);
    fEvCorrAssign->SetDirectory(NULL);
    fEvCorrAssign->UseCurrentStyle();
    fEvCorrAssign->SetXTitle("log10(Size) [#phot]");
    fEvCorrAssign->SetYTitle("Fraction of events with srcpos well assign");
}


// --------------------------------------------------------------------------
//
// Destructor 
//
MHDisp::~MHDisp()
{
  delete fHistEnergy;
  delete fHistSize;
  delete fHistcosZA;
  delete fSkymapXY;
  delete fHistChi2;
  delete fHistDuDv;
  delete fHistChi2Energy;
  delete fHistChi2Size;
  delete fHistDuEnergy;
  delete fHistDuSize;
  delete fHistDvEnergy;
  delete fHistDvSize;
  delete fEvCorrAssign;
}

// --------------------------------------------------------------------------
//
// Set the pointers to the containers 
// 
//
Bool_t MHDisp::SetupFill(const MParList *pList)
{
    // reset all histograms and Chi^2 computing variables
    // before each new eventloop
    fNumEv = 0;
    fSumChi2  = 0.;
    fChi2  = 0.;

    fHistEnergy->Reset();
    fHistSize->Reset();
    fHistcosZA->Reset();
    fSkymapXY->Reset();
    fHistChi2->Reset();
    fHistDuDv->Reset();
    fHistChi2Energy->Reset();
    fHistChi2Size->Reset();
    fHistDuEnergy->Reset();
    fHistDuSize->Reset();
    fHistDvEnergy->Reset();
    fHistDvSize->Reset();
    fEvCorrAssign->Reset();


    // look for the defined camera geometry to get mm to deg conversion factor
    MGeomCam *cam = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!cam)
    {
        *fLog << err << "MGeomCam (Camera Geometry) not found... aborting." 
              << endl;
        return kFALSE;
    }
    fMm2Deg = cam->GetConvMm2Deg();


    // look for Disp related containers
    fImageParDisp = (MImageParDisp*)pList->FindObject(fImageParDispName,
						      "MImageParDisp");
    if (!fImageParDisp)
    {
        *fLog << err << fImageParDispName 
              << " [MImageParDisp] not found... aborting." << endl;
        return kFALSE;
    }


    //-----------------------------------------------------------
    // if fMatrix exists, skip preprocessing and just read events from matrix;
    // if doesn't exist, read variables values from containers, so look for them
    if (fMatrix)
      return kTRUE;
    
    fSrcPos = (MSrcPosCam*)pList->FindObject("MSrcPosCam");
    if (!fSrcPos)
    {
        *fLog << err << "MSrcPosCam not found... aborting." << endl;
        return kFALSE;
    }

    fHil = (MHillas*)pList->FindObject("MHillas");
    if (!fHil)
    {
        *fLog << err << "MHillas not found... aborting." << endl;
        return kFALSE;
    }

    fHilExt = (MHillasExt*)pList->FindObject("MHillasExt");
    if (!fHilExt)
    {
        *fLog << err << "MHillasExt not found... aborting." << endl;
        return kFALSE;
    }

    fNewPar = (MNewImagePar*)pList->FindObject("MNewImagePar");
    if (!fNewPar)
    {
        *fLog << err << "MNewImagePar not found... aborting." << endl;
        return kFALSE;
    }

    fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << err << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    fPointing = (MPointingPos*)pList->FindObject("MPointingPos");
    if (!fPointing)
    {
        *fLog << err << "MPointingPos not found... " 
	      << "MMcEvt is going to be used to get Theta and Phi." 
	      << endl;
	//        return kFALSE;
    }

    //------------------------------------------

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Set which selection algorithm for the Disp estimated source position 
// we want to follow when filling the histograms:
//  * iflag == 1 --> Correct source position, the closest to the real srcpos
//                   (only applicable to MC or well known source real data)
//  * iflag == 2 --> Wrong source position, the furthest to the real srcpos
//                   (same applicability than case 1)
//  * iflag == 3 --> Source position assigned as M3Long sign indicates
//  * iflag == 4 --> Source position assigned as Asym sign indicates
//
void MHDisp::SetSelectedPos(Int_t iflag)
{ 
    fSelectedPos = iflag; 
}    


// --------------------------------------------------------------------------
//
// Sets the Matrix and the array of mapped values for each Matrix column
// (see MDispCalc)
//
void MHDisp::SetMatrixMap(MHMatrix *matrix, TArrayI &map)
{
    fMatrix = matrix;
    fMap = map;
}


// --------------------------------------------------------------------------
//
// Returns a mapped value from the Matrix
//
Double_t MHDisp::GetVal(Int_t i) const
{
    Double_t val = (*fMatrix)[fMap[i]];

    //*fLog << "MHDisp::GetVal; i, fMatrix, fMap, val = "
    //    << i << ",  " << fMatrix << ",  " << fMap[i] << ",  " << val << endl;

    return val;
}


// --------------------------------------------------------------------------
//
// Fill the histograms and sum up the Chi^2 outcome of each processed event
//
Bool_t MHDisp::Fill(const MParContainer *par, const Stat_t w)
{
    Double_t energy = 0.;
    Double_t impact = 0.;
    Double_t xmax = 0.;
    Double_t theta = 0.;
    Double_t phi = 0.;

    if ( fMatrix || (!fMatrix && fMcEvt) )
    {  
      energy   = fMatrix ? GetVal(13)  : fMcEvt->GetEnergy();
      impact   = fMatrix ? GetVal(14)  : fMcEvt->GetImpact();
      xmax     = fMatrix ? GetVal(15)  : fMcEvt->GetLongitmax();
      
      if (fPointing)
      {
	theta  = fMatrix ? GetVal(16)  : fPointing->GetZd();
	phi    = fMatrix ? GetVal(17)  : fPointing->GetAz();
      }
      else
      {
	theta  = fMatrix ? GetVal(16)  : fMcEvt->GetTelescopeTheta()*kRad2Deg;
	phi    = fMatrix ? GetVal(17)  : fMcEvt->GetTelescopePhi()*kRad2Deg;
      }
    }
    else
      *fLog << "No MMcEvt container available (no Energy,ImpactPar or Xmax)" << endl;

    Double_t xsrcpos0   = fMatrix ? GetVal(0)   : fSrcPos->GetX();
    Double_t ysrcpos0   = fMatrix ? GetVal(1)   : fSrcPos->GetY();
    Double_t xmean0     = fMatrix ? GetVal(2)   : fHil->GetMeanX();  
    Double_t ymean0     = fMatrix ? GetVal(3)   : fHil->GetMeanY();
    Double_t delta      = fMatrix ? GetVal(4)   : fHil->GetDelta();
    
    Double_t size       = fMatrix ? GetVal(5)   : fHil->GetSize();
    //    Double_t width0     = fMatrix ? GetVal(6)   : fHil->GetWidth();
    //    Double_t length0    = fMatrix ? GetVal(7)   : fHil->GetLength();
    
    //    Double_t conc       = fMatrix ? GetVal(8)   : fNewPar->GetConc();
    //    Double_t leakage1   = fMatrix ? GetVal(9)   : fNewPar->GetLeakage1();
    //    Double_t leakage2   = fMatrix ? GetVal(10)  : fNewPar->GetLeakage2();
    
    Double_t m3long     = fMatrix ? GetVal(11)  : fHilExt->GetM3Long();
    Double_t asym       = fMatrix ? GetVal(12)  : fHilExt->GetAsym();

    //------------------------------------------
    // convert to deg
    //    Double_t width    = width0  * fMm2Deg;
    //    Double_t length   = length0 * fMm2Deg;
    Double_t xsrcpos  = xsrcpos0 * fMm2Deg;
    Double_t ysrcpos  = ysrcpos0 * fMm2Deg;
    Double_t xmean    = xmean0 * fMm2Deg;
    Double_t ymean    = ymean0 * fMm2Deg;
    
    // calculate cosinus of the angle between d and a vectors
    Double_t a = (xmean-xsrcpos)*cos(delta) + (ymean-ysrcpos)*sin(delta);
    Double_t b = sqrt( (xmean-xsrcpos)*(xmean-xsrcpos) + (ymean-ysrcpos)*(ymean-ysrcpos) );
    Double_t cosda = a/b;
    
    // sign of cosda
    Int_t s0 = cosda>0 ? 1 : -1;   
    
    // get the sign of M3Long and Asym variables
    Int_t sm3 = m3long>0 ? 1 : -1;
    Int_t sa  =   asym>0 ? 1 : -1;
    
    // set the sign "s" that will select one Disp source position for each
    // shower, according to each of the possible algorithms for solution selection:
    //   SelectedPos = 1  means choose the right source position
    //                 2                   wrong
    //                 3               the position according to M3Long
    //                 4               the position according to Asym
    Int_t s = s0;
    if (fSelectedPos == 1)    
      s = s0;  
    else if (fSelectedPos == 2)
      s = -s0;
    else if (fSelectedPos == 3)
      s = sm3;
    else if (fSelectedPos == 4)
      s = sa;
    else
      *fLog << "Illegal value for Disp srcpos selection algorithm: " 
	    << " fSelectedPos = " << fSelectedPos << endl;
    
    // count the number of events where source position has been correctly assigned
    if (s == s0)
      fEvCorrAssign->Fill(log10(size), 1.);
    else
      fEvCorrAssign->Fill(log10(size), 0.);
    
    // get estimated Disp value
    Double_t disp = fImageParDisp->GetDisp();
    
    //------------------------------------------
    // Disp estimated source position
    Double_t xdisp = xmean - s*cos(delta)*disp;
    Double_t ydisp = ymean - s*sin(delta)*disp;
    fSkymapXY->Fill(xdisp,ydisp);
    
    // Distance between estimated Disp and real source position
    Double_t d2 = (xdisp-xsrcpos)*(xdisp-xsrcpos) +  (ydisp-ysrcpos)*(ydisp-ysrcpos); 
    fHistChi2->Fill(d2);
    
    // Longitudinal and transversal distances between Disp and real source positon
    Double_t Du = -s*( (xdisp-xsrcpos)*cos(delta) + (ydisp-ysrcpos)*sin(delta));
    Double_t Dv = -s*(-(xdisp-xsrcpos)*sin(delta) + (ydisp-ysrcpos)*cos(delta));
    fHistDuDv->Fill(Dv,Du);
    
    // Fill Energy, Size and ZA distributions
    fHistEnergy->Fill(log10(energy));
    fHistSize->Fill(log10(size));
    fHistcosZA->Fill(cos(theta/kRad2Deg));
    
    // to check the size dependence of the optimization
    fHistChi2Energy->Fill(log10(energy),d2);
    fHistChi2Size->Fill(log10(size),d2);
    fHistDuEnergy->Fill(log10(energy),Du);
    fHistDuSize->Fill(log10(size),Du);
    fHistDvEnergy->Fill(log10(energy),Dv);
    fHistDvSize->Fill(log10(size),Dv);
    
    // variables for Chi^2 calculation (= d^2)
    fNumEv += 1;
    fSumChi2 += d2;  
    
    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Calculates the final Chi^2 of the Disp optimization
//
Bool_t MHDisp::Finalize()
{
    fChi2 = fSumChi2/fNumEv;
    *fLog << endl;
    *fLog << "MHDisp::Finalize: SumChi2, NumEv = " << fSumChi2 << ", " << fNumEv << endl;
    *fLog << "MHDisp::Finalize: Final Chi2 = " << fChi2 << endl;
    
    fChi2 = fHistChi2->GetMean();
    *fLog << "MHDisp::Finalize: Final Chi2 = " << fChi2 << endl;

    SetReadyToSave();

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the Disp related histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHDisp::Draw(Option_t *opt)
{
    gStyle->SetPalette(1);

    TString title = GetName();
    title += ": ";
    title += GetTitle();

    TCanvas *pad = new TCanvas(GetName(),title,0,0,900,1500);    
    pad->SetBorderMode(0);
    pad->Divide(3,5);

    pad->cd(1);
    gPad->SetBorderMode(0);
    fHistcosZA->SetTitleOffset(1.2,"Y");
    fHistcosZA->DrawClone(opt);
    fHistcosZA->SetBit(kCanDelete);
    gPad->Modified();

    pad->cd(2);
    gPad->SetBorderMode(0);
    fHistEnergy->SetTitleOffset(1.2,"Y");
    fHistEnergy->DrawClone(opt);
    fHistEnergy->SetBit(kCanDelete);
    gPad->Modified();

    pad->cd(3);
    gPad->SetBorderMode(0);
    fHistSize->SetTitleOffset(1.2,"Y");
    fHistSize->DrawClone(opt);
    fHistSize->SetBit(kCanDelete);
    gPad->Modified();

    pad->cd(4);
    gPad->SetBorderMode(0);
    fHistChi2->SetTitleOffset(1.2,"Y");
    fHistChi2->DrawClone(opt);
    fHistChi2->SetBit(kCanDelete);
    gPad->Modified();

    TProfile *profChi2Energy = fHistChi2Energy->ProfileX("Chi^2 vs. Energy",0,99999,"s");
    profChi2Energy->SetXTitle("log10(Energy) [GeV]");
    profChi2Energy->SetYTitle("Chi^2 = distance^2 Disp to real srcpos [deg^2]");
    pad->cd(5);
    gPad->SetBorderMode(0);
    profChi2Energy->SetTitleOffset(1.2,"Y");
    profChi2Energy->DrawClone(opt);
    profChi2Energy->SetBit(kCanDelete);
    gPad->Modified();

    TProfile *profChi2Size = fHistChi2Size->ProfileX("Chi^2 vs. Size",0,99999,"s");
    profChi2Size->SetXTitle("log10(Size) [#phot]");
    profChi2Size->SetYTitle("Chi^2 = distance^2 Disp to real srcpos [deg^2]");
    pad->cd(6);
    gPad->SetBorderMode(0);
    profChi2Size->SetTitleOffset(1.2,"Y");
    profChi2Size->DrawClone(opt);
    profChi2Size->SetBit(kCanDelete);
    gPad->Modified();

    pad->cd(7);
    gPad->SetBorderMode(0);
    fSkymapXY->SetTitleOffset(1.2,"Y");
    fSkymapXY->DrawClone("COLZ");
    fSkymapXY->SetBit(kCanDelete);
    gPad->Modified();

    pad->cd(8);
    gPad->SetBorderMode(0);
    fEvCorrAssign->SetTitleOffset(1.2,"Y");
    fEvCorrAssign->DrawClone(opt);
    fEvCorrAssign->SetBit(kCanDelete);
    gPad->Modified();

    pad->cd(9);
    gPad->SetBorderMode(0);
    fHistDuDv->SetTitleOffset(1.2,"Y");
    fHistDuDv->DrawClone("COLZ");
    fHistDuDv->SetBit(kCanDelete);
    gPad->Modified();

    TH1F *histDu = (TH1F*)fHistDuDv->ProjectionY("histDu");
    histDu->SetTitle("Longitudinal distance Du");
    histDu->SetXTitle("Du = longitudinal distance [deg]");
    pad->cd(10);
    gPad->SetBorderMode(0);
    histDu->SetTitleOffset(1.2,"Y");
    histDu->DrawClone(opt);
    histDu->SetBit(kCanDelete);
    gPad->Modified();

    TProfile *profDuEnergy = fHistDuEnergy->ProfileX("Du vs. Energy",0,99999,"s");
    profDuEnergy->SetXTitle("log10(Energy) [GeV]");
    profDuEnergy->SetYTitle("Du = longitudinal distance [deg]");
    pad->cd(11);
    gPad->SetBorderMode(0);
    profDuEnergy->SetTitleOffset(1.2,"Y");
    profDuEnergy->DrawClone(opt);
    profDuEnergy->SetBit(kCanDelete);
    gPad->Modified();

    TProfile *profDuSize = fHistDuSize->ProfileX("Du vs. Size",0,99999,"s");
    profDuSize->SetXTitle("log10(Size) [#phot]");
    profDuSize->SetYTitle("Du = longitudinal distance [deg]");
    pad->cd(12);
    gPad->SetBorderMode(0);
    profDuSize->SetTitleOffset(1.2,"Y");
    profDuSize->DrawClone(opt);
    profDuSize->SetBit(kCanDelete);
    gPad->Modified();

    TH1F *histDv = (TH1F*)fHistDuDv->ProjectionX("histDv");
    histDv->SetTitle("Transversal distance Dv");
    histDv->SetXTitle("Dv = transversal distance [deg]");
    pad->cd(13);
    gPad->SetBorderMode(0);
    histDv->SetTitleOffset(1.2,"Y");
    histDv->DrawClone(opt);
    histDv->SetBit(kCanDelete);
    gPad->Modified();

    TProfile *profDvEnergy = fHistDvEnergy->ProfileX("Dv vs. Energy",0,99999,"s");
    profDvEnergy->SetXTitle("log10(Energy) [GeV]");
    profDvEnergy->SetYTitle("Dv = transversal distance [deg]");
    pad->cd(14);
    gPad->SetBorderMode(0);
    profDvEnergy->SetTitleOffset(1.2,"Y");
    profDvEnergy->DrawClone(opt);
    profDvEnergy->SetBit(kCanDelete);
    gPad->Modified();

    TProfile *profDvSize = fHistDvSize->ProfileX("Dv vs. Size",0,99999,"s");
    profDvSize->SetXTitle("log10(Size) [#phot]");
    profDvSize->SetYTitle("Dv = transversal distance [deg]");
    pad->cd(15);
    gPad->SetBorderMode(0);
    profDvSize->SetTitleOffset(1.2,"Y");
    profDvSize->DrawClone(opt);
    profDvSize->SetBit(kCanDelete);
    gPad->Modified();
}















