/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
/////////////////////////////////////////////////////////////////////////////
//
//  MJCalibrateSignalFromOutside
//
// This class is reading the output written by callisto. It calibrates
// signal and time.
//
// The signal and time extractors are read from the callisto-output. In
// pricipal you could overwrite these default using the resource file,
// but this is NOT recommended!
//
/////////////////////////////////////////////////////////////////////////////
#include "MJCalibrateSignalFromOutside.h"

#include <TEnv.h>
#include <TFile.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MDirIter.h"
#include "MRunIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MStatusDisplay.h"

#include "MGeomCam.h"
#include "MHCamEvent.h"
#include "MPedestalCam.h"
#include "MPedPhotCam.h"
#include "MCalibrationHiLoCam.h"
#include "MBadPixelsCam.h"

#include "MCalibrationQECam.h"
#include "MCalibrationBlindCam.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationRelTimeCam.h"
#include "MCalibrationChargePINDiode.h"

#include "MCalibrationChargeCalc.h"
#include "MCalibrationRelTimeCalc.h"

#include "MCalibrationIntensityChargeCam.h"
#include "MCalibrationIntensityBlindCam.h"
#include "MCalibrationIntensityRelTimeCam.h"
#include "MCalibrationIntensityQECam.h"
#include "MBadPixelsIntensityCam.h"

#include "MHCalibrationChargeCam.h"
#include "MHCalibrationChargeBlindCam.h"
#include "MHCalibrationChargePINDiode.h"
#include "MHCalibrationRelTimeCam.h"

#include "MCalibCalcFromPast.h"

#include "MReadReports.h"
#include "MReadMarsFile.h"
#include "MRawFileRead.h"
#include "MTriggerPatternDecode.h"
#include "MFTriggerPattern.h"
#include "MContinue.h"
#include "MGeomApply.h"
#include "MMcPedestalCopy.h"
#include "MPointingPosCalc.h"
#include "MPedCalcFromLoGain.h"
#include "MExtractor.h"
#include "MExtractPINDiode.h"
#include "MExtractBlindPixel.h"
#include "MTaskEnv.h"
#include "MCalibrateData.h"
#include "MCalibrateRelTimes.h"
#include "MBadPixelsMerge.h"
#include "MBadPixelsCalc.h"
#include "MBadPixelsTreat.h"
#include "MFillH.h"
#include "MWriteRootFile.h"

ClassImp(MJCalibrateSignalFromOutside);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets fRuns to 0, fExtractor to NULL, fDataCheck to kFALSE
//
MJCalibrateSignalFromOutside::MJCalibrateSignalFromOutside(const char *name, const char *title)
    : fIsDataCheck(kFALSE), fGeometry("MGeomCamMagic"), fInterlaced(kFALSE)
{
    fName  = name  ? name  : "MJCalibrateSignalFromOutside";
    fTitle = title ? title : "Tool to calibrate data";
}

Bool_t MJCalibrateSignalFromOutside::WriteResult()
{
    if (fPathOut.IsNull())
    {
        *fLog << inf << "No output path specified via SetPathOut - no output written." << endl;
        return kTRUE;
    }

    const TString oname = Form("%s/signal%08d.root", (const char*)fPathOut, fSequence.GetSequence());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "RECREATE");

    *fLog << inf << " - MStatusDisplay..." << flush;
    if (fDisplay && fDisplay->Write()<=0)
    {
        *fLog << err << "Unable to write MStatusDisplay to " << oname << endl;
        return kFALSE;
    }
    *fLog << inf << "ok." << endl;

    return kTRUE;
}

Bool_t MJCalibrateSignalFromOutside::ProcessFile(MPedestalCam &pedcamab, MPedestalCam &pedcam2, MCalibrationChargeCam &chargecam, MCalibrationQECam &qecam, MCalibrationRelTimeCam &relcam, Byte_t filetype)
{
    // --------------------------------------------------------------------------------

    *fLog << inf;
    fLog->Separator(GetDescriptor());

    *fLog << "Calibrate data from ";
    *fLog << "Runs " << fRuns->GetRunsAsString() << endl;
    *fLog << endl;

    MDirIter iter;

    *fLog << all;
    if (fExtractor)
    {
        *fLog << underline << "Signal Extractor found in calibration file" << endl;
        fExtractor->Print();
        *fLog << endl;
    }
    else
      {
        *fLog << inf << "No Signal Extractor: ExtractSignal in file." << endl;
        return kFALSE;
      }
    
    fExtractor->SetPedestals(&pedcamab);

    // Setup Parlist
    MParList plist;
    plist.AddToList(this); // take care of fDisplay!
    plist.AddToList(&fBadPixels);
    plist.AddToList(&pedcam2);
    plist.AddToList(&pedcamab);
    plist.AddToList(&chargecam);
    plist.AddToList(&qecam);
    plist.AddToList(&relcam);

    MCalibrationIntensityChargeCam  intenscharge;
    MCalibrationIntensityRelTimeCam intensreltime;
    MCalibrationIntensityQECam      intensqe;
    MCalibrationIntensityBlindCam   intensblind;
    MBadPixelsIntensityCam          intensbad;

    MHCalibrationChargeCam      hchargecam;
    MHCalibrationChargeBlindCam hblindcam;
    MHCalibrationChargePINDiode hpindiode;
    MHCalibrationRelTimeCam     hrelcam;

    //
    // Calibration Results containers
    //
    if (fInterlaced)
      {
        plist.AddToList(&intensqe);
        plist.AddToList(&intenscharge);
        plist.AddToList(&intensblind);
        //        plist.AddToList(&intensCalibrationPINDiode);
        plist.AddToList(&intensreltime);
        plist.AddToList(&intensbad);

        hchargecam.SetLast(499.5);
        hchargecam.SetNbins(300);

        hchargecam.SetLoGain(kFALSE);
        hrelcam.SetLoGain(kFALSE);

        hchargecam.SetOscillations(kFALSE);
        hrelcam.SetOscillations(kFALSE);

        plist.AddToList(&hchargecam);
        plist.AddToList(&hblindcam);
        plist.AddToList(&hrelcam);
        plist.AddToList(&hpindiode);
    }

    MPedPhotCam pedphotcam;
    plist.AddToList(&pedphotcam);

    MFillH fillhilo("MHCalibrationHiLoCam",      "MExtractedSignalCam",        "FillHiLoCam");
    fillhilo.SetBit(MFillH::kDoNotDisplay);

    MCalibrationHiLoCam hilocam;
    plist.AddToList(&hilocam);
    // Setup Tasklist
    MTaskList tlist;
    plist.AddToList(&tlist);

    MReadReports readreal;
    readreal.AddTree("Events", "MTime.", kTRUE);
    readreal.AddTree("Trigger");
    readreal.AddTree("Camera");
    readreal.AddTree("Drive");
    readreal.AddTree("CC");
    readreal.AddTree("Currents");

    //    MReadMarsFile read("Events");
    //    read.DisableAutoScheme();
    MRawFileRead rawread(NULL);
    if (IsDataCheck())
        rawread.AddFiles(*fRuns);
    else
      {
        //        read.AddFiles(*fRuns);
        readreal.AddFiles(*fRuns);
      }

    // Check for interleaved events
    MTriggerPatternDecode decode;

    MGeomApply             apply; // Only necessary to create geometry
    apply.SetGeometry(fGeometry);
    MBadPixelsMerge        merge(&fBadPixels);

    MPedestalCam pedcam;

    pedcamab.SetName("MPedestalFundamental");
    pedcam2.SetName("MPedestalFromExtractorRndm");
    pedcam.SetName("MPedestalFromExtractor");
    plist.AddToList(&pedcam);
    plist.AddToList(&pedcam2);
    plist.AddToList(&pedcamab);

    // Do signal and pedestal calculation
    MPedCalcFromLoGain     pedlo1("MPedCalcFundamental");
    pedlo1.SetPedestalUpdate(kTRUE);
    pedlo1.SetNamePedestalCamOut("MPedestalFundamental");

    MPedCalcFromLoGain     pedlo2("MPedCalcWithExtractorRndm");
    pedlo2.SetPedestalUpdate(kTRUE);
    pedlo2.SetRandomCalculation(kTRUE);
    pedlo2.SetNamePedestalCamIn("MPedestalFundamental");
    pedlo2.SetNamePedestalCamOut("MPedestalFromExtractorRndm");

    MPedCalcFromLoGain     pedlo3("MPedCalcWithExtractor");
    pedlo3.SetPedestalUpdate(kTRUE);
    pedlo3.SetRandomCalculation(kFALSE);
    pedlo3.SetNamePedestalCamIn("MPedestalFundamental");
    pedlo3.SetNamePedestalCamOut("MPedestalFromExtractor");

    if (fExtractor)
    {
        fExtractor->SetPedestals(&pedcamab);

        if (fExtractor->InheritsFrom("MExtractTimeAndCharge"))
        {
            pedlo2.SetExtractor((MExtractTimeAndCharge*)fExtractor);
            pedlo3.SetExtractor((MExtractTimeAndCharge*)fExtractor);
            pedlo1.SetExtractWindow(15, 0/*obsolete*/);
            pedlo2.SetExtractWindow(15, 0/*obsolete*/);
            pedlo3.SetExtractWindow(15, 0/*obsolete*/);
        }
        else
        {
            // FIXME: How to get the fixed value 15 automatically?
            const Int_t f = (Int_t)(15.5+fExtractor->GetHiGainFirst());
            const Int_t n = (Int_t)(15.5+fExtractor->GetNumHiGainSamples());
            pedlo1.SetExtractWindow(f, n);
            pedlo2.SetExtractWindow(f, n);
            pedlo3.SetExtractWindow(f, n);
        }
    }

    MMcPedestalCopy        pcopy;
    MTaskEnv taskenv1("ExtractSignal");
    taskenv1.SetDefault(fExtractor);
    MCalibrateData         calib;
    if (filetype==3) // MC file
    {
        calib.SetCalibrationMode(MCalibrateData::kFfactor);
        calib.SetPedestalFlag(MCalibrateData::kRun);
        calib.AddPedestal("MPedestalCam", "MPedPhotFundamental");
        calib.AddPedestal("MPedestalCam", "MPedPhotFromExtractor");
        calib.AddPedestal("MPedestalCam", "MPedPhotFromExtractorRndm");
    }
    else
    {
        calib.AddPedestal("Fundamental");
        calib.AddPedestal("FromExtractor");
        calib.AddPedestal("FromExtractorRndm");
        calib.SetPedestalFlag(MCalibrateData::kEvent);
        calib.SetSignalType(MCalibrateData::kPhe);
    }
    
    MCalibrateRelTimes     caltm;
    MBadPixelsCalc         bpcal;
    MBadPixelsTreat        treat;

    MHCamEvent evt0(0, "PedFLG",      "Pedestal from Lo Gain;;P [fadc/sl]");
    MHCamEvent evt1(2, "PedRmsFLG",   "Pedestal RMS from Lo Gain;;\\sigma_{p} [fadc/sl]");
    MHCamEvent evt2(0, "Extra'd",     "Extracted Signal;;S [fadc/sl]");
    MHCamEvent evt3(0, "PedPhot",     "Calibrated Pedestal;;P [\\gamma]");
    MHCamEvent evt4(1, "PedRMS",      "Calibrated Pedestal RMS;;\\sigma_{p} [\\gamma]");
    MHCamEvent evt5(0, "Interp'd",    "Interpolated Signal;;S [\\gamma]");
    MHCamEvent evt6(2, "Unsuitable",  "Unsuitable event ratio;;%");
    MHCamEvent evt7(0, "Times",       "Arrival Time;;T [slice]");
    MHCamEvent evt8(0, "HiLoConv",    "Ratio Hi-Lo Gain Signal;;Ratio [1]");

    evt0.EnableVariance();
    evt1.EnableVariance();
    evt2.EnableVariance();
    evt3.EnableVariance();
    evt4.EnableVariance();
    evt5.EnableVariance();
    evt7.EnableVariance();
    evt8.EnableVariance();

    MFillH fill0(&evt0, "MPedestalFundamental",   "FillPedFLG");
    MFillH fill1(&evt1, "MPedestalFromExtractorRndm", "FillPedRmsFLG");
    MFillH fill2(&evt2, "MExtractedSignalCam",    "FillExtracted");
    MFillH fill3(&evt3, "MPedPhotFundamental",    "FillPedPhot");
    MFillH fill4(&evt4, "MPedPhotFromExtractorRndm","FillPedRMS");
    MFillH fill5(&evt5, "MCerPhotEvt",            "FillInterpolated");
    MFillH fill6(&evt6, "MBadPixelsCam",       "FillUnsuitable");
    MFillH fill7(&evt7, "MArrivalTime",        "FillTimes");
    MFillH fill8(&evt8, &hilocam,              "FillHiLo");

    MWriteRootFile write(2, Form("%s{s/_D_/_Y_}", fPathOut.Data()), fOverwrite);
    // Run Header
    write.AddContainer("MRawRunHeader",       "RunHeaders");
    write.AddContainer("MBadPixelsCam",       "RunHeaders");
    write.AddContainer("MGeomCam",            "RunHeaders");
    // Monte Carlo Headers
    write.AddContainer("MMcTrigHeader",       "RunHeaders", kFALSE);
    write.AddContainer("MMcConfigRunHeader",  "RunHeaders", kFALSE);
    write.AddContainer("MMcCorsikaRunHeader", "RunHeaders", kFALSE);
    // Monte Carlo
    write.AddContainer("MMcEvt",              "Events", kFALSE);
    write.AddContainer("MMcTrig",             "Events", kFALSE);
    // Data
    write.AddContainer("MCerPhotEvt",         "Events");
    write.AddContainer("MPedPhotCam",         "Events");
    write.AddContainer("MTime",               "Events", kFALSE);
    write.AddContainer("MRawEvtHeader",       "Events");
    write.AddContainer("MArrivalTime",        "Events");
    // Slow-Control: Current
    write.AddContainer("MTimeCurrents",       "Currents", kFALSE);
    write.AddContainer("MCameraDC",           "Currents", kFALSE);
    write.AddContainer("MReportCurrents",     "Currents", kFALSE);
    // Slow-Control: Camera
    write.AddContainer("MReportCamera",       "Camera", kFALSE);
    write.AddContainer("MTimeCamera",         "Camera", kFALSE);
    write.AddContainer("MCameraAUX",          "Camera", kFALSE);
    write.AddContainer("MCameraCalibration",  "Camera", kFALSE);
    write.AddContainer("MCameraCooling",      "Camera", kFALSE);
    write.AddContainer("MCameraHV",           "Camera", kFALSE);
    write.AddContainer("MCameraLV",           "Camera", kFALSE);
    write.AddContainer("MCameraLids",         "Camera", kFALSE);
    // Slow-Control: Trigger
    write.AddContainer("MReportTrigger",      "Trigger", kFALSE);
    write.AddContainer("MTimeTrigger",        "Trigger", kFALSE);
    // Slow-Control: Drive
    write.AddContainer("MPointingPos",        "Drive", kFALSE);
    write.AddContainer("MReportDrive",        "Drive", kFALSE);
    write.AddContainer("MTimeDrive",          "Drive", kFALSE);
    // Slow-Control: Central Control
    write.AddContainer("MReportCC",           "CC", kFALSE);
    write.AddContainer("MTimeCC",             "CC", kFALSE);

    // Now setup tasklist for events
    MTaskList tlist2;
    tlist2.AddToList(&apply);
    tlist2.AddToList(&merge);
    tlist2.AddToList(&decode);
    if (filetype==3)
        tlist2.AddToList(&pcopy);
    else
    {
        tlist2.AddToList(&pedlo1);
        tlist2.AddToList(&pedlo2);
    }
    MExtractPINDiode        pinext;
    MExtractBlindPixel      blindext;

    MFTriggerPattern fcalib;
    
    MCalibrationChargeCalc  chargecalc;
    MCalibrationRelTimeCalc relcalc;

    MCalibCalcFromPast      pastcalc;

    pinext.SetPedestals(&pedcamab);
    blindext.SetPedestals(&pedcamab);
    chargecalc.SetPedestals(&pedcam2);

    pastcalc.SetChargeCalc(&chargecalc);
    pastcalc.SetRelTimeCalc(&relcalc);
    pastcalc.SetCalibrate(&calib);

    tlist2.AddToList(&fill0);
    tlist2.AddToList(&fill1);
    tlist2.AddToList(&taskenv1);
    tlist2.AddToList(&fill2);
    tlist2.AddToList(&pinext);
    tlist2.AddToList(&blindext);
    //
    // Calibration histogramming
    //
    MFillH fillpin("MHCalibrationChargePINDiode", "MExtractedSignalPINDiode",   "FillPINDiode");
    MFillH fillbnd("MHCalibrationChargeBlindCam", "MExtractedSignalBlindPixel", "FillBlindCam");
    MFillH fillcam("MHCalibrationChargeCam",      "MExtractedSignalCam",        "FillChargeCam");
    MFillH filltme("MHCalibrationRelTimeCam",     "MArrivalTimeCam",            "FillRelTime");
    fillpin.SetBit(MFillH::kDoNotDisplay);
    fillbnd.SetBit(MFillH::kDoNotDisplay);
    fillcam.SetBit(MFillH::kDoNotDisplay);
    filltme.SetBit(MFillH::kDoNotDisplay);

    //
    // Calibration Results containers
    //
    if (fInterlaced)
      {
        fcalib.DenyCalibration();
        fcalib.SetInverted();

        chargecalc.SetFilter(&fcalib);
        pastcalc.SetFilter(  &fcalib);
        fillcam.SetFilter(&fcalib);
        filltme.SetFilter(&fcalib);
        fillbnd.SetFilter(&fcalib);
        fillpin.SetFilter(&fcalib);

        tlist2.AddToList(&fcalib);
        tlist2.AddToList(&pastcalc);
        tlist2.AddToList(&fillcam);
        tlist2.AddToList(&filltme);
        tlist2.AddToList(&fillbnd);
        tlist2.AddToList(&fillpin);
        tlist2.AddToList(&chargecalc);
        tlist2.AddToList(&relcalc);
      }
    
    tlist2.AddToList(&calib);
    if (&relcam)
      tlist2.AddToList(&caltm);
    tlist2.AddToList(&fillhilo);
    tlist2.AddToList(&fill3);
    tlist2.AddToList(&bpcal);
    tlist2.AddToList(&treat);
    tlist2.AddToList(&fill4);
    tlist2.AddToList(&fill5);
    tlist2.AddToList(&fill6);
    tlist2.AddToList(&fill7);
    tlist2.AddToList(&fill8);

    // Setup List for Drive-tree
    MPointingPosCalc pcalc;

    // Now setup main tasklist
    tlist.AddToList(IsDataCheck() ? (MTask*)&rawread : (MTask*)&readreal);
    tlist.AddToList(&tlist2, IsDataCheck()?"All":"Events");

    //    if (!IsDataCheck())
    //        tlist.AddToList(&pcalc, "Drive");

    tlist.AddToList(&write);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);
    if (!SetupEnv(evtloop))
        return kFALSE;

    // Execute first analysis
    if (!evtloop.Eventloop(fMaxEvents))
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    
    if (fInterlaced)
      {
        TObjArray cont;
        cont.Add(&intensqe);
        cont.Add(&intenscharge);
        cont.Add(&intensblind);
        //        cont.Add(&intensCalibrationPINDiode);
        cont.Add(&intensreltime);
        cont.Add(&intensbad);
        WriteContainer(cont);
      }

    if (!WriteResult())
        return kFALSE;

    *fLog << all << GetDescriptor() << ": Done." << endl;
    *fLog << endl << endl;

    return kTRUE;
}
