#ifndef MARS_MPointing
#define MARS_MPointing

#ifndef ROOT_TArrayD
#include <TArrayD.h>
#endif

#ifndef ROOT_TVector3
#include <TVector3.h>
#endif

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif

// ---------------------------------------------------
// FIXME: Replace coord.h completely with this!
#ifndef __MARS__
#include "coord.h"
#else
#include <TVector2.h>
class AltAz : public TVector2
{
public:
    AltAz(Double_t alt, Double_t az) : TVector2(alt, az)
    {
    }
    Double_t Alt() const { return fX; }
    Double_t Az() const  { return fY; }
    ClassDef(AltAz, 1)
};

class ZdAz : public TVector2
{
public:
    ZdAz(Double_t zd, Double_t az) : TVector2(zd, az)
    {
    }
    Double_t Zd() const { return fX; }
    Double_t Az() const { return fY; }
    ClassDef(ZdAz, 1)
};
#endif
// ---------------------------------------------------

class TMinuit;

class MPointing : public MParContainer
{
private:
    static const Int_t fNumPar;

    Double_t fIe   ; // [rad] Index Error in Elevation
    Double_t fIa   ; // [rad] Index Error in Azimuth
    Double_t fFlop ; // [rad] Vertical Sag
    Double_t fNpae ; // [rad] Az-El Nonperpendicularity
    Double_t fCa   ; // [rad] Left-Right Collimation Error
    Double_t fAn   ; // [rad] Azimuth Axis Misalignment (N-S)
    Double_t fAw   ; // [rad] Azimuth Axis Misalignment (E-W)
    Double_t fTf   ; // [rad] Tube fluxture (sin)
    Double_t fTx   ; // [rad] Tube fluxture (tan)
    Double_t fNrx  ; // [rad] Nasmyth rotator displacement, horizontal
    Double_t fNry  ; // [rad] Nasmyth rotator displacement, vertical
    Double_t fCrx  ; // [rad] Alt/Az Coude Displacement (N-S)
    Double_t fCry  ; // [rad] Alt/Az Coude Displacement (E-W)
    Double_t fEces ; // [rad] Elevation Centering Error (sin)
    Double_t fAces ; // [rad] Azimuth Centering Error (sin)
    Double_t fEcec ; // [rad] Elevation Centering Error (cos)
    Double_t fAcec ; // [rad] Azimuth Centering Error (cos)
    Double_t fMagic1; // [rad] Magic Term (what is it?)
    Double_t fMagic2; // [rad] Magic Term (what is it?)

    Double_t **fCoeff; //!
    TString   *fNames; //!
    TString   *fDescr; //!

    TArrayD   fError;

    void Init(const char *name=0, const char *title=0);

    void Clear(Option_t *o="")
    {
        for (int i=0; i<fNumPar; i++)
        {
            *fCoeff[i] = 0;
            fError[i] = 0;
        }
    }

    static Double_t Sign(Double_t val, Double_t alt);
    AltAz CalcAnAw(const AltAz &p, Int_t sign) const;

public:
    MPointing() { fError.Set(fNumPar); Init(); Clear(); }
    MPointing(const char *name) { fError.Set(fNumPar); Init(); Clear(); Load(name); }
    virtual ~MPointing() { delete fNames; delete fCoeff; delete fDescr; }

    void Load(const char *name);
    void Save(const char *name);

    void Reset();

    ZdAz     Correct(const ZdAz &zdaz) const;
    AltAz    Correct(const AltAz &aaz) const;
    TVector3 Correct(const TVector3 &v) const;

    ZdAz     CorrectBack(const ZdAz &zdaz) const;
    AltAz    CorrectBack(const AltAz &aaz) const;
    TVector3 CorrectBack(const TVector3 &v) const;

    ZdAz     operator()(const ZdAz &zdaz)  const { return Correct(zdaz); }
    AltAz    operator()(const AltAz &aaz)  const { return Correct(aaz); }
    TVector3 operator()(const TVector3 &v) const { return Correct(v); }

    ZdAz operator()(const ZdAz &zdaz, void (*fcn)(ZdAz &zdaz, Double_t *par)) const
    {
        Double_t par[fNumPar];
        GetParameters(par);
        ZdAz za = zdaz;
        fcn(za, par);
        return za;
    }

    AltAz operator()(const AltAz &aaz, void (*fcn)(AltAz &aaz, Double_t *par)) const
    {
        Double_t par[fNumPar];
        GetParameters(par);
        AltAz aa = aaz;
        fcn(aa, par);
        return aa;
    }

    TVector3 operator()(const TVector3 &aaz, void (*fcn)(TVector3 &aaz, Double_t *par)) const
    {
        Double_t par[fNumPar];
        GetParameters(par);
        TVector3 v = aaz;
        fcn(v, par);
        return v;
    }

    AltAz  AddOffsets(const AltAz &aa) const;
    ZdAz   AddOffsets(const ZdAz &zdaz) const
    {
        AltAz p(TMath::Pi()/2-zdaz.Zd(), zdaz.Az());
        AltAz c = AddOffsets(p);
        return ZdAz(TMath::Pi()/2-c.Alt(), c.Az());
    }
    TVector3 AddOffsets(const TVector3 &v) const
    {
        AltAz p(TMath::Pi()/2-v.Theta(), v.Phi());
        AltAz c = AddOffsets(p);
        TVector3 rc;
        rc.SetMagThetaPhi(1, TMath::Pi()/2-c.Alt(), c.Az());
        return rc;
    }

    AltAz  SubtractOffsets(const AltAz &aa) const;
    ZdAz   SubtractOffsets(const ZdAz &zdaz) const
    {
        AltAz p(TMath::Pi()/2-zdaz.Zd(), zdaz.Az());
        AltAz c = SubtractOffsets(p);
        return ZdAz(TMath::Pi()/2-c.Alt(), c.Az());
    }
    TVector3 SubtractOffsets(const TVector3 &v) const
    {
        AltAz p(TMath::Pi()/2-v.Theta(), v.Phi());
        AltAz c = SubtractOffsets(p);
        TVector3 rc;
        rc.SetMagThetaPhi(1, TMath::Pi()/2-c.Alt(), c.Az());
        return rc;
    }

    void SetParameters(const Double_t *par, Int_t n=fNumPar);
    void GetParameters(Double_t *par, Int_t n=fNumPar) const;

    void SetParameters(const TArrayD &par)
    {
        SetParameters(par.GetArray(), par.GetSize());
    }
    void GetParameters(TArrayD &par) const
    {
        par.Set(fNumPar);
        GetParameters(par.GetArray());
    }
    void GetError(TArrayD &par) const
    {
        par = fError;
        for (int i=0; i<fNumPar; i++)
            par[i] *= TMath::RadToDeg();
    }

    void SetMinuitParameters(TMinuit &m, Int_t n=-1) const;
    void GetMinuitParameters(TMinuit &m, Int_t n=-1);
    void PrintMinuitParameters(TMinuit &m, Int_t n=-1) const;

    const TString &GetVarName(int i) const { return fNames[i]; }
    const TString &GetDescription(int i) const { return fDescr[i]; }

    /*
     Double_t GetIe() const { return fIe; }
     Double_t GetIa() const { return fIa; }
     Double_t GetCa() const { return fCa; }
     Double_t GetAn() const { return fAn; }
     Double_t GetAw() const { return fAw; }
     Double_t GetNrx() const { return fNrx; }
     Double_t GetNry() const { return fNry; }
     Double_t GetCrx() const { return fNrx; }
     Double_t GetCry() const { return fNry; }
     Double_t GetEces() const { return fEces; }
     Double_t GetEcec() const { return fEcec; }
     Double_t GetAces() const { return fAces; }
     Double_t GetAcec() const { return fAcec; }
     Double_t GetNpae() const { return fNpae; }
     */

    static const Int_t GetNumPar() { return fNumPar; }

    ClassDef(MPointing, 1) // Pointing Model for MAGIC
};

#endif
