/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   11/2003 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
/////////////////////////////////////////////////////////////////////////////
//                                                               
// MBadPixelsIntensityCam                                               
//                                                               
// Base class for intensity calibration results 
//
// Contains TClonesArrays for the following objects:
// - fCams:  Array of classes derived from MBadPixelsCam, one entry 
//           per calibration camera result. Has to be created
//
// See also: MCalibrationIntensityChargeCam, MCalibrationIntensityQECam,
//           MCalibrationIntensityRelTimeCam,
//           MCalibrationCam, MCalibrationPix, 
//           MCalibrationQECam, MCalibrationQEPix,
//           MHCalibrationChargePix, MHCalibrationChargeCam              
//           MCalibrationChargeBlindPix, MCalibrationChargePINDiode
//
//
/////////////////////////////////////////////////////////////////////////////
#include "MBadPixelsIntensityCam.h"

#include <TOrdCollection.h>

#include "MGeomCam.h"

ClassImp(MBadPixelsIntensityCam);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Initializes and sets owner of:
// - fCams
// - Initializes fCams to entry 
//
MBadPixelsIntensityCam::MBadPixelsIntensityCam(const char *name, const char *title)
{

  fName  = name  ? name  : "MBadPixelsIntensityCam";
  fTitle = title ? title : "Base container for the Intensity Calibration";
  
  fCams = new TOrdCollection;
  fCams->SetOwner();

  InitSize(1);
}

// --------------------------------------------------------------------------
//
// Deletes the cameras if they exist
//
MBadPixelsIntensityCam::~MBadPixelsIntensityCam()
{
  if (fCams)
    delete fCams;
}

// --------------------------------------------------------------------------
//
// Add a new MBadPixelsCam to fCams, give it the name "name" and initialize
// it with geom.
//
void MBadPixelsIntensityCam::AddToList( const char* name, const MGeomCam &geom) 
{
  InitSize(GetSize()+1);
  GetCam()->SetName(name);
  GetCam()->Init(geom);
}



// --------------------------------------------------------------------------
//
// Copy 'constructor'
//
void MBadPixelsIntensityCam::Copy(TObject& object) const
{
  
  MBadPixelsIntensityCam &calib = (MBadPixelsIntensityCam&)object;
  
  MParContainer::Copy(calib);
  
  const UInt_t n = GetSize();
  if (n != 0)
    {
      calib.InitSize(n);
      for (UInt_t i=0; i<n; i++)
        GetCam(i)->Copy(*(calib.GetCam(i)));
    }
  
}

// -----------------------------------------------------
//
// Calls Clear() for all entries fCams
//
void MBadPixelsIntensityCam::Clear(Option_t *o)
{

  fCams->ForEach(MBadPixelsCam, Clear)(); 

  return;
}

// -----------------------------------------------------
//
// Calls Print(o) for all entries fCams
//
void MBadPixelsIntensityCam::Print(Option_t *o) const
{
  fCams->ForEach(MBadPixelsCam, Print)(o); 
}


// -------------------------------------------------------------------
//
// Initialize the objects inside the TOrdCollection using the 
// function Add().
//
// InitSize can only increase the size, but not shrink. 
// 
// It can be called more than one time. New Containers are 
// added only from the current size to the argument i.
//
void MBadPixelsIntensityCam::InitSize(const UInt_t i)
{

  const UInt_t save = GetSize();

  if (i==save)
    return;
  
  if (i>save)
    Add(save,i);
}

// -------------------------------------------------------------------
//
// Add MBadPixelsCams in the ranges from - to. In order to initialize
// from MBadPixelsCam derived containers, overwrite this function
//
void MBadPixelsIntensityCam::Add(const UInt_t from, const UInt_t to)
{
  for (UInt_t i=from; i<to; i++)
    fCams->AddAt(new MBadPixelsCam,i);
}


// -------------------------------------------------------------------
//
// If size is still 0, Intialize a first Cam.
// Calls Init(geom) for all fCams
//
void MBadPixelsIntensityCam::Init(const MGeomCam &geom)
{
  if (GetSize() == 0)
    InitSize(1);

  fCams->ForEach(MBadPixelsCam,Init)(geom);
}


// --------------------------------------------------------------------------
//
// Returns the current size of the TOrdCollection fCams 
// independently if the MBadPixelsCam is filled with values or not.
//
Int_t MBadPixelsIntensityCam::GetSize() const 
{
  return fCams->GetSize();
}

// --------------------------------------------------------------------------
//
// Get i-th pixel from current camera
//
MBadPixelsPix &MBadPixelsIntensityCam::operator[](Int_t i)
{
  return (*GetCam())[i];
}

// --------------------------------------------------------------------------
//
// Get i-th pixel from current camera
//
const MBadPixelsPix &MBadPixelsIntensityCam::operator[](Int_t i) const 
{
  return (*GetCam())[i];
}


// --------------------------------------------------------------------------
//
// Get i-th camera 
//
MBadPixelsCam *MBadPixelsIntensityCam::GetCam(Int_t i)
{
  return static_cast<MBadPixelsCam*>(i==-1 ? fCams->Last() : fCams->At(i));
}

// --------------------------------------------------------------------------
//
// Get i-th camera 
//
const MBadPixelsCam *MBadPixelsIntensityCam::GetCam(Int_t i) const 
{
  return static_cast<MBadPixelsCam*>(i==-1 ? fCams->Last() : fCams->At(i));
}

// --------------------------------------------------------------------------
//
// Get camera with name 'name' 
//
MBadPixelsCam *MBadPixelsIntensityCam::GetCam(const char *name )
{
  return static_cast<MBadPixelsCam*>(fCams->FindObject(name));
}

// --------------------------------------------------------------------------
//
// Get camera with name 'name' 
//
const MBadPixelsCam *MBadPixelsIntensityCam::GetCam(const char *name ) const 
{
  return static_cast<MBadPixelsCam*>(fCams->FindObject(name));
}

// --------------------------------------------------------------------------
//
// Calls GetPixelContent for the current entry in fCams
//
Bool_t MBadPixelsIntensityCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{
  return GetCam()->GetPixelContent(val,idx,cam,type);
}

// --------------------------------------------------------------------------
//
// Calls DrawPixelContent for the current entry in fCams
//
void MBadPixelsIntensityCam::DrawPixelContent( Int_t num ) const
{
  return GetCam()->DrawPixelContent(num);
}

