#ifndef MARS_MCalibrateData
#define MARS_MCalibrateData

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MCalibrateData                                                          //
//                                                                         //
// Integrates the desired ADC time slices of one pixel and apply           //
// calibration constants                                                   //
//                                                                         //
// Differences between MCalibrateData and MCalibrate :                     //
// in MCalibrateData                                                       //
// - in ReInit the MPedPhot container is filled using                      //
//      - the pedstals/slice from MPedestalCam                             //
//      - the number of used FADC slices from MExtractedSignalCam          //
//      - the photon/ADC conversion factor from MCalibrationCam            //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#ifndef MARS_MTask
#include "MTask.h"
#endif

#ifndef MARS_MArrayF
#include "MArrayF.h"
#endif

class MGeomCam;
class MBadPixelsCam;
class MPedestalCam;
class MCalibrationChargeCam;
class MCalibrationQECam;
class MExtractedSignalCam;

class MPedPhotCam;
class MCerPhotEvt;
class MCalibConstCam;
class MCalibrateData : public MTask
{
private:

  static const Float_t  fgCalibConvMinLimit; //! Minimum limit for conv. factor
  static const Float_t  fgCalibConvMaxLimit; //! Maximum limit for conv. factor

  Float_t  fCalibConvMinLimit;           // Minimum limit for conv. factor
  Float_t  fCalibConvMaxLimit;           // Maximum limit for conv. factor
    
  MGeomCam              *fGeomCam;       //! Camera geometry container
  MBadPixelsCam         *fBadPixels;     //! Bad Pixels information
  MCalibrationChargeCam *fCalibrations;  //! Calibration constants
  MCalibrationQECam     *fQEs;           //! Quantum efficiencies
  MExtractedSignalCam   *fSignals;       //! Integrated charge in FADCs counts
  MCerPhotEvt           *fCerPhotEvt;    //! Cerenkov Photon Event used for calculation
  MCalibConstCam        *fCalibConstCam; //! Temporary calib consts storage
  
  UShort_t fCalibrationMode;             // Flag defining the calibration mode (CalibrationMode_t)
  Byte_t   fPedestalFlag;                // Flags defining to calibrate the pedestal each event or each run
  Byte_t   fSignalType;                  // Flag defining the signal type (kphot or kphe)
  
  Float_t  fRenormFactor;                // Possible renormalization factor for signals (-> phes)
  
  TList   fNamesPedestal;                // Names of input and output pedestal conatainer
  TList   fPedestalCams;                 //! List of pointers to input MPedestalCam
  TList   fPedPhotCams;                  //! List of pointers to corresponding output MPedPhotCam

  MArrayF fCalibConsts;                  //! Array of calibration constants for each pixel, calculated only once!
  MArrayF fCalibFFactors;                //! Array of calibration F-Factors for each pixel, calculated only once!  
  MArrayF fHiLoConv;                     //! Array of calibration constants for each pixel, calculated only once!
  MArrayF fHiLoConvErr;                  //! Array of calibration F-Factors for each pixel, calculated only once!  
  
  Int_t  Calibrate(Bool_t data, Bool_t pedestal) const;
  
  Int_t  PreProcess(MParList *pList);
  Bool_t ReInit(MParList *pList);
  Int_t  Process();
  Int_t  ReadEnv(const TEnv &env, TString prefix, Bool_t print);
  void   StreamPrimitive(ofstream &out) const;
  
public:

    enum CalibrationMode_t
    {
        kSkip       =  0,
        kNone       =  1,
        kFlatCharge =  2,
        kBlindPixel =  3,
        kFfactor    =  4,
        kPinDiode   =  5,
        kCombined   =  6,
        kDummy      =  7
    };

  static const CalibrationMode_t kDefault = kFfactor;
  
  enum PedestalType_t
  {
      kNo    = BIT(0),
      kRun   = BIT(1),
      kEvent = BIT(2)
  };
  
  enum SignalType_t
  {
      kPhe,
      kPhot
  };
  
  MCalibrateData(CalibrationMode_t calmode=kDefault,
                 const char *name=NULL, const char *title=NULL);
  
  void   AddPedestal(const char *name="Cam");
  void   AddPedestal(const char *pedestal, const char *pedphot);

  void   EnablePedestalType(PedestalType_t i)     { fPedestalFlag |=  i;      }

  Int_t  GetSize() const { return fCalibConsts.GetSize(); }

  void   Print(Option_t *o="") const;

  void   SetPedestalFlag(PedestalType_t i=kRun)   { fPedestalFlag  =  i;      }
  Bool_t TestPedestalFlag(PedestalType_t i) const { return fPedestalFlag&i ? kTRUE : kFALSE; }
  
  void   SetCalibrationMode ( CalibrationMode_t calmode=kDefault ) { fCalibrationMode=calmode; }
  void   SetSignalType      ( SignalType_t      sigtype=kPhe    ) { fSignalType     =sigtype; }  

  void   SetCalibConvMinLimit( const Float_t f=fgCalibConvMinLimit ) { fCalibConvMinLimit = f; }
  void   SetCalibConvMaxLimit( const Float_t f=fgCalibConvMaxLimit ) { fCalibConvMaxLimit = f; }
    
  Bool_t UpdateConversionFactors( const MCalibrationChargeCam *updatecam=NULL);

  ClassDef(MCalibrateData, 1)   // Task to calibrate FADC counts into photons or photo-electrons
};

#endif   /* MCalibrateData */






