#ifndef STARCATALOG_H
#define STARCATALOG_H

#ifndef ROOT_TROOT
#include <TROOT.h>
#endif
#ifndef SAOFILE_H
#include "SaoFile.h"
#endif

#include "coord.h"

class StarCatalog
{
private:
    SaoFile *fSao;
    sort_t  *fSrt;
    int      fEntries;

    double   fPhi;      // location of observatory
    double   fElong;

    double   fPixSize;  // [rad/pix] size of one pixel
    double   fWidth;    // size of display
    double   fHeight;   //

    float    fLimitMag; // [M] limiting magnitude for display

    AltAz    fAltAz;    // [rad]
    char     fAz0[360];
    int      fAltMin;
    int      fAltMax;
    int      fAzCnt;

    RaDec    fRaDec;    // [rad]
    char     fRa0[360];
    int      fRaCnt;
    int      fDecMin;
    int      fDecMax;

    double   fAlpha;
    double   fMjd;
    double   fAmprms[21];
    double   fAoprms[14];

    void DrawCross(char *img, const int x, const int y);
    void DrawCircle(int color, char *img, int xx, int yy, int size);

    Bool_t DrawAltAz(const int color, char *img, double alt, double az, int size=0);
    Bool_t DrawRaDec(const int color, char *img, double ra,  double dec, int size=0);

    Bool_t Draw(const int color, char *img, const AltAz &altaz);
    Bool_t Draw(const int color, char *img, const RaDec &radec);
    Bool_t Draw(const int color, char *img, const SaoFile *sao);

    void   CalcImg(char *);

    void   SetAltAz(const AltAz &altaz);
    void   SetMjd(const double mjd);
    void   DrawSCAltAz(char *img, const int color);
    void   DrawSCRaDec(char *img, const int color);
    void   SetRaDec(const RaDec &radec, const RaDec &rdpm);

    void   CalcRaDecRange();
    void   CalcAltAzRange();

    RaDec  AltAz2RaDec(const AltAz &altaz) const;
    AltAz  RaDec2AltAz(const RaDec &radec, const RaDec &rdpm) const;

public:
    StarCatalog();
    virtual ~StarCatalog();

    void   GetImg(char *img, char *cimg, const double utc,
                  const RaDec &radec);
    void   GetImg(char *img, char *cimg, const double utc,
                  const AltAz &altaz);

    const AltAz GetAltAz() const { return fAltAz*360/D2PI; }
    const ZdAz  GetZdAz() const  { return ZdAz(DPI/2-fAltAz.Alt(), fAltAz.Az())*360/D2PI; }
    const RaDec GetRaDec() const { return fRaDec*360/D2PI; }

    void  SetPixSize(const double pixsize);
    void  SetLimitMag(const float mag) { fLimitMag = mag; };
};

#endif
