/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 11/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
// MReportFileReadCC
//
// This is a report file reader which implements the CC header checking.
// Because diffrent subsystem are writing different headers it is not
// easily possible to have one Reader for all files. Because of this
// you must know to which subsystem the file belongs before you can
// instantiate your reader if you need the header or want to check the
// header.
//
// If you want to restrict reading to 'single run report files' you can
// call SetRunNumber(12345). In this case Checking the Header will fail
// if no run information is available or the runnumber in the header
// doesn't match. To request a 'all run' file use SetRunNumber(0).
// To allow both file types to be read use SetRunNumber(-1) <default>
//
//////////////////////////////////////////////////////////////////////////////
#include "MReportFileReadCC.h"

#include <fstream>
#include <stdlib.h> // atoi on gcc 2.95.3

#include <TRegexp.h>

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MReportFileReadCC);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. It tries to open the given file and creates a
// THashTable which allows faster access to the MReport* objects.
//
MReportFileReadCC::MReportFileReadCC(const char *fname, const char *name, const char *title)
    : MReportFileRead(fname, name, title), fRunNumber(-1)
{
    fName  = name  ? name  : "MReportFileReadCC";
    fTitle = title ? title : "Read task to read Central Control report files";
}

// --------------------------------------------------------------------------
//
// Check whether the file header corresponds to a central control file
// header and check for the existance of a correct version number.
// The version number may later be used to be able to read different
// file versions
//
Int_t MReportFileReadCC::CheckFileHeader()
{
    Int_t line = 0;

    TString str;
    str.ReadLine(*fIn);   // Read to EOF or newline
    if (!*fIn)
        return line;

    line++;

    Int_t run = -1;

    // whole night report file
    if (str==TString("[CC Report File]"))
        run = 0;

    // report file matching a single run
    if (!str(TRegexp("^[CC Run [0-9]+ Control File]$")).IsNull())
        run = atoi(str(TRegexp(" [0-9]+")).Data());

    if (run<0)
    {
        *fLog << err << "ERROR - First line doesn't match '[CC Report File]' ";
        *fLog << "nor '^[CC Run [0-9]+ Control File]$'" << endl;
        return line;
    }

    if (fRunNumber!=-1 && fRunNumber!=run)
    {
        *fLog << err << "ERROR - Requested run #" << fRunNumber << " doesn't ";
        *fLog << "match found run #" << run << endl;
        return line;
    }

    str.ReadLine(*fIn);   // Read to EOF or newline
    if (!*fIn)
        return line;

    if (str(TRegexp("^Arehucas Version Number [0-9]+-[0-9]$")).IsNull())
    {
        *fLog << err << "ERROR - Version '^Arehucas Version Number [0-9]+-[0-9]$' ";
        *fLog << "not found in second line." << endl;
        return line;
    }
    line++;

    str = str(TRegexp("[0-9]+-[0-9]"));
    str.Prepend("20");
    str.ReplaceAll("-", "");

    const Int_t ver = atoi(str.Data());

    *fLog << all << "Report File version: <" << ver << ">" << endl;

    SetVersion(ver);

    return line;
}
