#ifndef MARS_MRawRunHeader
#define MARS_MRawRunHeader
///////////////////////////////////////////////////////////////////////
//                                                                   //
// MRunHeader                                                        //
//                                                                   //
///////////////////////////////////////////////////////////////////////

#ifndef MARS_MTime
#include "MTime.h"
#endif

class TBuffer;
class MArrayS;

class MRawRunHeader : public MParContainer
{
public:
    //
    // enum for the Run Type. Monte Carlo Runs have
    // to have a value greater than 255 (>0xff)
    //
    enum {
        kRTData        = 0x0000,
        kRTPedestal    = 0x0001,
        kRTCalibration = 0x0002,
        kRTPointRun    = 0x0007,
        kRTMonteCarlo  = 0x0100,
        kRTNone        = 0xffff
    };

    //
    // Magic number to detect the magic file type
    //
    static const UShort_t kMagicNumber;
    static const Byte_t   kMaxFormatVersion;

private:
    /* ---- Run Header Informations ---- */
    UShort_t  fMagicNumber;
    UShort_t  fFormatVersion;
    UShort_t  fSoftVersion;
    UShort_t  fCameraVersion;
    UShort_t  fTelescopeNumber;
    UShort_t  fRunType;
    UInt_t    fRunNumber;
    Char_t    fProjectName[101];//23];
    Char_t    fSourceName[81];  //13];
    Char_t    fObservationMode[61];
    Char_t    fSourceEpochChar[2];
    UShort_t  fSourceEpochDate;
    UShort_t  fNumCrates;
    UShort_t  fNumPixInCrate;
    UShort_t  fNumSamplesLoGain;
    UShort_t  fNumSamplesHiGain;
    UInt_t    fNumEvents;
    MTime     fRunStart;
    MTime     fRunStop;
    MArrayS  *fPixAssignment;

    Bool_t SwapAssignment(Short_t id0, Short_t id1);
    Bool_t FixAssignment();

public:
    MRawRunHeader(const char *name=NULL, const char *title=NULL);
    ~MRawRunHeader();

    // This is to be used in the MC chain only!
    void SetMagicNumber(UShort_t a)       { fMagicNumber=a; }
    void SetFormatVersion(UShort_t a)     { fFormatVersion=a; }
    void SetSoftVersion(UShort_t a)       { fSoftVersion=a; }
    void SetRunType(UShort_t a)           { fRunType=a; }
    void SetRunNumber(UInt_t a)           { fRunNumber=a; }
    void SetNumEvents(UInt_t a)           { fNumEvents=a; }
    void SetNumSamples(UShort_t low, UShort_t high)
    {
        fNumSamplesLoGain=low;
        fNumSamplesHiGain=high;
    }
    void SetNumCrates(UShort_t a)         { fNumCrates=a; }
    void SetNumPixInCrate(UShort_t a)     { fNumPixInCrate=a; }
    void SetRunTime(Float_t start, Float_t stop)
    { fRunStart.SetMjd(start); fRunStop.SetMjd(stop); }

    // This is to get the numbers...
    UShort_t GetMagicNumber() const       { return fMagicNumber; }
    UShort_t GetFormatVersion() const     { return fFormatVersion; }
    UShort_t GetSoftVersion() const       { return fSoftVersion; }
    UInt_t   GetRunNumber() const         { return fRunNumber; }
    UShort_t GetRunType() const           { return fRunType; }
    const Char_t *GetRunTypeStr() const;
    const Char_t *GetProjectName() const { return fProjectName; }
    const Char_t *GetSourceName() const  { return fSourceName; }
    const Char_t *GetSourceEpocheChar() const { return fSourceEpochChar; }
    const Char_t *GetObservationMode() const { return fObservationMode; }
    UShort_t GetSourceEpocheDate() const  { return fSourceEpochDate; }
    UShort_t GetNumCrates() const         { return fNumCrates; }
    UShort_t GetNumPixInCrate() const     { return fNumPixInCrate; }
    UShort_t GetNumSamplesLoGain() const  { return fNumSamplesLoGain; }
    UShort_t GetNumSamplesHiGain() const  { return fNumSamplesHiGain; }
    UInt_t   GetNumEvents() const         { return fNumEvents; }
    const MTime &GetRunStart() const      { return fRunStart; }
    const MTime &GetRunEnd() const        { return fRunStop; }
    Short_t GetPixAssignment(UShort_t i) const;
    UShort_t GetMaxPixId() const;
    UShort_t GetMinPixId() const;
    UShort_t GetNumConnectedPixels() const;
    UShort_t GetNumNormalPixels() const;
    UShort_t GetNumSpecialPixels() const;

    UInt_t GetNumSamplesPerCrate() const
    {
        return fNumPixInCrate*(fNumSamplesLoGain+fNumSamplesHiGain);
    }

    UShort_t GetNumPixel() const;
    Int_t GetNumTotalBytes() const;

    Bool_t IsValid() const { return fMagicNumber==0xc0c0 || fMagicNumber==0xc0c1; }
    Bool_t IsMonteCarloRun() const { return fRunType>0x00ff; }

    void Print(Option_t *t=NULL) const;

    Bool_t ReadEvt(istream& fin);

    ClassDef(MRawRunHeader, 4)	// storage container for general info
};
#endif
