//=//////////////////////////////////////////////////////////////////////
//=
//= srreadparam            
//=
//= @file        srreadparam.cxx
//= @desc        Reading of parameters file
//= @author      O Blanch Bigas
//= @email       blanch@ifae.es
//= @date        Wed Feb 21 17:43:07 CET 2001
//=
//=----------------------------------------------------------------------
//=
//= Created: Wed Feb 21 17:43:07 CET 2001
//= Author:  Oscar Blanch Bigas
//= Purpose: Program for star response  simulation
//= Notes:   See files README for details
//=    
//=----------------------------------------------------------------------
//=
//= $RCSfile: srreadparam.cxx,v $
//= $Revision: 1.7 $
//= $Author: moralejo $ 
//= $Date: 2004-11-16 11:09:11 $
//=
//=//////////////////////////////////////////////////////////////////////

// @T \newpage

//!@section Source code of |srreadparam.cxx|.

/*!@"
  
  This section describes briefly the source code for the file
  |srreadparam.cxx|. This file is very closely related to the file
  |readparams.cxx| from the |reflector| program. Actually, this later
  file was the ancestror of the file you are looking at.

  All the defines it uses are located in the file |srreadparam.h|. In
  the first one we can see the definitions of the commands available
  for the parameters file. We describe these commands in a later
  section.
  
  @"*/

//!@subsection Includes and Global variables definition.

/*!@"
  
  All the defines are located in the file {\tt srreadparam.h}.

  @"*/
 
//!@{

using namespace std;

#include "srreadparam.h"

//!@}

//!@subsection Definition of global variables.

/*!@"

  Here we define the global variables where the values from the
  parameters file are stored.
  
  @"*/

//!@{

static char Database_path[PATH_MAX_LENGTH];  //@< path to store the database
static float Simulated_Phe_l = 0.0;          //@< lower limit for phe loop
static float Simulated_Phe_u = 50.0;         //@< higher limit for phe loop
static float Simulated_Phe_p = 0.1;          //@< precision for phe loop
static int   FADC_Shape=0;
static float FADC_Integ = MFADC_RESPONSE_INTEGRAL;
static float FADC_FWHM = MFADC_RESPONSE_FWHM;
static float FADC_slices_per_ns = FADC_SLICES_PER_NSEC;
static int   Trig_Shape = 0;
static float Trig_Ampl = 1.0;
static float Trig_FWHM = 2.0;
static int   Write_Root = 0;
static int   gain_fluctuations = 1; // Is 0 if PMT gain fluctuations are disabled

//!@}

//!@subsection The function |readparam()|.

//!-----------------------------------------------------------
// @name  srreadparam
//                                                
// @desc  read parameters from the stdin / parameters file
//
// @var   *filename  Name of the parameters file (NULL->STDIN)
//
// @date Wed Feb 21 17:43:07 CET 2001
//------------------------------------------------------------
// @function

//!@{ 
void 
readparam(char * filename)
{
  char line[LINE_MAX_LENGTH];    //@< line to get from the stdin
  char token[ITEM_MAX_LENGTH];   //@< a single token
  int i;                         //@< dummy counters
  ifstream ifile;

  ifile.open( filename );

  // loop till the "end" directive is reached
  int is_end = FALSE;
  while (! is_end) {          

    // get line from file or stdin
    if ( filename != NULL )
      ifile.getline(line, LINE_MAX_LENGTH);
    else
      cin.getline(line, LINE_MAX_LENGTH);

    // skip comments (start with '#')
    if (line[0] == '#')
      continue;

    // show user comments (start with '>')
    if (line[0] == '>') {
      cout << line << endl << flush;
      continue;
    }

    // look for each item at the beginning of the line
    for (i=0; i<=end_file; i++) 
      if (strstr(line, ITEM_NAMES[i]) == line)
        break;
        
    // if it is not a valid line, just ignore it
    if (i == end_file+1) {
      cerr << "Skipping unknown token in [" << line << "]\n";
      continue;
    }

    // case block for each directive
    switch ( i ) {

    case database_path:          //@< name of the output path
          
      // get the path for the outcoming database 
      sscanf(line, "%s %s", token, Database_path);

      break;

    case simulated_phe:       //@< limits for the phe loop
          
      sscanf(line, "%s %f %f %f", token, &Simulated_Phe_l, &Simulated_Phe_u,
	     &Simulated_Phe_p);

      break;

    case trig_properties:       //@< shape of trigger response
          
      sscanf(line, "%s %d %f %f", token, &Trig_Shape, &Trig_Ampl,
	     &Trig_FWHM);

      break;

    case fadc_properties:       //@< shape of fadc response
          
      sscanf(line, "%s %d %f %f", token, &FADC_Shape, &FADC_Integ,
	     &FADC_FWHM);

      break;

    case fadc_GHz:  //  Get FADC sampling frequency in GHz

      sscanf(line, "%s %f", token, &FADC_slices_per_ns);

      break;

    case gain_fluctuations_off:

      gain_fluctuations = 0;  // Disable PMT gain fluctuations

      break;

    case write_root:          //@< Write histogram

      Write_Root = 1;

    case end_file:            //@< end of the parameters file

      // show a short message
      is_end = TRUE;

      break;

    } // switch ( i ) 

  } // while (! is_end)

  // after the loop is finished, return to the main function
  return;
}
//!@}


//!-----------------------------------------------------------
// @name get_database_path
//                                                
// @desc get name of the ouput path
//
// @return   Name of the output path
//
// @date Wed Feb 21 17:57:19 CET 2001
//------------------------------------------------------------
// @function 

//!@{
char *
get_database_path(void)
{
  return (Database_path);
}
//!@}

//!-----------------------------------------------------------
// @name get_simulated_phe 
//
// @desc return limits for the phe loop
//
// @var *lphe  Lower limit in the phe loop
// @var *uphe  Higher limit in the phe loop
// @var *pphe  Precision in the phe loop
// @return  void
//
// @date Wed Feb 21 18:04:03 CET 2001
//------------------------------------------------------------
// @function 

//!@{
void
get_simulated_phe(float *lphe, float *uphe, float *pphe)
{
  *lphe =  Simulated_Phe_l;
  *uphe =  Simulated_Phe_u;
  *pphe =  Simulated_Phe_p;
}
//!@}

//!-----------------------------------------------------------
// @name get_trig_properties 
//
// @desc return shape of the single phe trigger response
//
// @var *shape number to indentify the shape (0 ->gaussian)
// @var *ampl  Amplitud of the gaussian response
// @var *fwhm  FWHM of the gaussian response
// @return  void
//
// @date Wed Feb 21 18:04:03 CET 2001
//------------------------------------------------------------
// @function 

//!@{
void
get_trig_properties(int *shape, float *ampl, float *fwhm)
{
  *shape =  Trig_Shape;
  *ampl =  Trig_Ampl;
  *fwhm =  Trig_FWHM;
}
//!@}

//!-----------------------------------------------------------
// @name get_fadc_properties 
//
// @desc return shape of the single phe FADC response
//
// @var *shape number to indentify the shape (0 ->gaussian)
// @var *ampl  Amplitud of the gaussian response
// @var *fwhm  FWHM of the gaussian response
// @return  void
//
// @date Wed Feb 21 18:04:03 CET 2001
//------------------------------------------------------------
// @function 

//!@{
void
get_fadc_properties(int *shape, float *integ, float *fwhm, float *fadc_spns)
{
  *shape =  FADC_Shape;
  *integ =  FADC_Integ;
  *fwhm  =  FADC_FWHM;
  *fadc_spns = FADC_slices_per_ns;
}
//!@}

//------------------------------------------------------------
// Returns gain_fluctuations, which is 0 if the PMT gain fluctuations 
// have been disabled in the input card.
//
int apply_gain_fluctuations()
{
  return gain_fluctuations;
}

//!-----------------------------------------------------------
// @name get_write_root
//                                                
// @desc get boolean to write root files
//
// @return   0 (false) or 1 (true)
//
// @date Fri Mar  2 16:17:26 CET 2001
//------------------------------------------------------------
// @function 

//!@{
int 
get_write_root(void)
{
  return (Write_Root);
}
//!@}

//=------------------------------------------------------------
//!@subsection Log of this file.

//!@{
//
// $Log: not supported by cvs2svn $
// Revision 1.6  2004/10/12 13:41:09  moralejo
// *** empty log message ***
//
// Revision 1.5  2004/09/17 13:51:02  moralejo
//
// Adapted headers to current c++ style, removed -Wno-deprecated from
// compilation options.
//
// Revision 1.4  2004/09/16 16:20:13  moralejo
// *** empty log message ***
//
// Revision 1.3  2004/01/30 10:34:59  blanch
// Change in variables name to clarify the meaning.
//
// Revision 1.2  2001/03/05 11:01:44  blanch
// Commads to enter the FADC and trigger properties have been added.
// There is also a new command that allows to writte the root files or not
//
// Revision 1.1  2001/02/23 10:13:44  magicsol
// It read form an input card (defeult=starresponse.par) the parameters that
// are needed for the starresponse program.
//
//!@}

//=EOF
