/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Abelardo Moralejo 06/2005 <mailto:moralejo@pd.infn.it>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MGeomCamMagic1183
//
// This class stores the geometry information of a possible Magic-2 camera,
// which only has small pixels.
// All information are copied from tables, see source code.
//
////////////////////////////////////////////////////////////////////////////
#include "MGeomCamMagic1183.h"

#include "MGeomPix.h"

ClassImp(MGeomCamMagic1183);

// --------------------------------------------------------------------------
//
//  This Magic camera has 1183 pixels. For geometry and Next Neighbor info see
//  CreateCam and CreateNN
//
MGeomCamMagic1183::MGeomCamMagic1183(const char *name)
    : MGeomCam(1183, 17, name, 
	       "Geometry information of Magic Camera with only small pixels")
{
    CreateCam();
    CreateNN();
    InitGeometry();
}

// --------------------------------------------------------------------------
//
//  This fills the geometry information from a table into the pixel objects.
//
void MGeomCamMagic1183::CreateCam()
{
    //
    //   fill the geometry class with the coordinates of the MAGIC camera
    //
    //*fLog << inf << " Creating Magic geometry " << endl ;

    //
    //   here define the hardwire things of the magic telescope
    //

  const float xtemp[1183] = {
    0, 30, 15, -15, -30, -15, 15, 60, 45, 30, 
    0, -30, -45, -60, -45, -30, 0, 30, 45, 90, 
    75, 60, 45, 15, -15, -45, -60, -75, -90, -75, 
    -60, -45, -15, 15, 45, 60, 75, 120, 105, 90, 
    75, 60, 30, 0, -30, -60, -75, -90, -105, -120, 
    -105, -90, -75, -60, -30, 0, 30, 60, 75, 90, 
    105, 150, 135, 120, 105, 90, 75, 45, 15, -15, 
    -45, -75, -90, -105, -120, -135, -150, -135, -120, -105, 
    -90, -75, -45, -15, 15, 45, 75, 90, 105, 120, 
    135, 180, 165, 150, 135, 120, 105, 90, 60, 30, 
    0, -30, -60, -90, -105, -120, -135, -150, -165, -180, 
    -165, -150, -135, -120, -105, -90, -60, -30, 0, 30, 
    60, 90, 105, 120, 135, 150, 165, 210, 195, 180, 
    165, 150, 135, 120, 105, 75, 45, 15, -15, -45, 
    -75, -105, -120, -135, -150, -165, -180, -195, -210, -195, 
    -180, -165, -150, -135, -120, -105, -75, -45, -15, 15, 
    45, 75, 105, 120, 135, 150, 165, 180, 195, 240, 
    225, 210, 195, 180, 165, 150, 135, 120, 90, 60, 
    30, 0, -30, -60, -90, -120, -135, -150, -165, -180, 
    -195, -210, -225, -240, -225, -210, -195, -180, -165, -150, 
    -135, -120, -90, -60, -30, 0, 30, 60, 90, 120, 
    135, 150, 165, 180, 195, 210, 225, 270, 255, 240, 
    225, 210, 195, 180, 165, 150, 135, 105, 75, 45, 
    15, -15, -45, -75, -105, -135, -150, -165, -180, -195, 
    -210, -225, -240, -255, -270, -255, -240, -225, -210, -195, 
    -180, -165, -150, -135, -105, -75, -45, -15, 15, 45, 
    75, 105, 135, 150, 165, 180, 195, 210, 225, 240, 
    255, 300, 285, 270, 255, 240, 225, 210, 195, 180, 
    165, 150, 120, 90, 60, 30, 0, -30, -60, -90, 
    -120, -150, -165, -180, -195, -210, -225, -240, -255, -270, 
    -285, -300, -285, -270, -255, -240, -225, -210, -195, -180, 
    -165, -150, -120, -90, -60, -30, 0, 30, 60, 90, 
    120, 150, 165, 180, 195, 210, 225, 240, 255, 270, 
    285, 330, 315, 300, 285, 270, 255, 240, 225, 210, 
    195, 180, 165, 135, 105, 75, 45, 15, -15, -45, 
    -75, -105, -135, -165, -180, -195, -210, -225, -240, -255, 
    -270, -285, -300, -315, -330, -315, -300, -285, -270, -255, 
    -240, -225, -210, -195, -180, -165, -135, -105, -75, -45, 
    -15, 15, 45, 75, 105, 135, 165, 180, 195, 210, 
    225, 240, 255, 270, 285, 300, 315, 360, 345, 330, 
    315, 300, 285, 270, 255, 240, 225, 210, 195, 180, 
    150, 120, 90, 60, 30, 0, -30, -60, -90, -120, 
    -150, -180, -195, -210, -225, -240, -255, -270, -285, -300, 
    -315, -330, -345, -360, -345, -330, -315, -300, -285, -270, 
    -255, -240, -225, -210, -195, -180, -150, -120, -90, -60, 
    -30, 0, 30, 60, 90, 120, 150, 180, 195, 210, 
    225, 240, 255, 270, 285, 300, 315, 330, 345, 390, 
    375, 360, 345, 330, 315, 300, 285, 270, 255, 240, 
    225, 210, 195, 165, 135, 105, 75, 45, 15, -15, 
    -45, -75, -105, -135, -165, -195, -210, -225, -240, -255, 
    -270, -285, -300, -315, -330, -345, -360, -375, -390, -375, 
    -360, -345, -330, -315, -300, -285, -270, -255, -240, -225, 
    -210, -195, -165, -135, -105, -75, -45, -15, 15, 45, 
    75, 105, 135, 165, 195, 210, 225, 240, 255, 270, 
    285, 300, 315, 330, 345, 360, 375, 420, 405, 390, 
    375, 360, 345, 330, 315, 300, 285, 270, 255, 240, 
    225, 210, 180, 150, 120, 90, 60, 30, 0, -30, 
    -60, -90, -120, -150, -180, -210, -225, -240, -255, -270, 
    -285, -300, -315, -330, -345, -360, -375, -390, -405, -420, 
    -405, -390, -375, -360, -345, -330, -315, -300, -285, -270, 
    -255, -240, -225, -210, -180, -150, -120, -90, -60, -30, 
    0, 30, 60, 90, 120, 150, 180, 210, 225, 240, 
    255, 270, 285, 300, 315, 330, 345, 360, 375, 390, 
    405, 450, 435, 420, 405, 390, 375, 360, 345, 330, 
    315, 300, 285, 270, 255, 240, 225, 195, 165, 135, 
    105, 75, 45, 15, -15, -45, -75, -105, -135, -165, 
    -195, -225, -240, -255, -270, -285, -300, -315, -330, -345, 
    -360, -375, -390, -405, -420, -435, -450, -435, -420, -405, 
    -390, -375, -360, -345, -330, -315, -300, -285, -270, -255, 
    -240, -225, -195, -165, -135, -105, -75, -45, -15, 15, 
    45, 75, 105, 135, 165, 195, 225, 240, 255, 270, 
    285, 300, 315, 330, 345, 360, 375, 390, 405, 420, 
    435, 480, 465, 450, 435, 420, 405, 390, 375, 360, 
    345, 330, 315, 300, 285, 270, 255, 240, 210, 180, 
    150, 120, 90, 60, 30, 0, -30, -60, -90, -120, 
    -150, -180, -210, -240, -255, -270, -285, -300, -315, -330, 
    -345, -360, -375, -390, -405, -420, -435, -450, -465, -480, 
    -465, -450, -435, -420, -405, -390, -375, -360, -345, -330, 
    -315, -300, -285, -270, -255, -240, -210, -180, -150, -120, 
    -90, -60, -30, 0, 30, 60, 90, 120, 150, 180, 
    210, 240, 255, 270, 285, 300, 315, 330, 345, 360, 
    375, 390, 405, 420, 435, 450, 465, 510, 495, 480, 
    465, 450, 435, 420, 405, 390, 375, 360, 345, 330, 
    315, 300, 285, 270, 255, 225, 195, 165, 135, 105, 
    75, 45, 15, -15, -45, -75, -105, -135, -165, -195, 
    -225, -255, -270, -285, -300, -315, -330, -345, -360, -375, 
    -390, -405, -420, -435, -450, -465, -480, -495, -510, -495, 
    -480, -465, -450, -435, -420, -405, -390, -375, -360, -345, 
    -330, -315, -300, -285, -270, -255, -225, -195, -165, -135, 
    -105, -75, -45, -15, 15, 45, 75, 105, 135, 165, 
    195, 225, 255, 270, 285, 300, 315, 330, 345, 360, 
    375, 390, 405, 420, 435, 450, 465, 480, 495, 510, 
    495, 480, 465, 450, 435, 420, 405, 390, 375, 360, 
    345, 330, 315, 300, 285, 210, 180, 150, 120, 90, 
    60, 30, 0, -30, -60, -90, -120, -150, -180, -210, 
    -240, -300, -315, -330, -345, -360, -375, -390, -405, -420, 
    -435, -450, -465, -480, -495, -510, -525, -510, -495, -480, 
    -465, -450, -435, -420, -405, -390, -375, -360, -345, -330, 
    -315, -300, -285, -210, -180, -150, -120, -90, -60, -30, 
    0, 30, 60, 90, 120, 150, 180, 210, 240, 300, 
    315, 330, 345, 360, 375, 390, 405, 420, 435, 450, 
    465, 480, 495, 510, 525, 495, 480, 465, 450, 435, 
    420, 405, 390, 375, 360, 345, 330, 135, 105, 75, 
    45, 15, -15, -45, -75, -105, -135, -165, -195, -360, 
    -375, -390, -405, -420, -435, -450, -465, -480, -495, -510, 
    -525, -495, -480, -465, -450, -435, -420, -405, -390, -375, 
    -360, -345, -330, -135, -105, -75, -45, -15, 15, 45, 
    75, 105, 135, 165, 195, 360, 375, 390, 405, 420, 
    435, 450, 465, 480, 495, 510, 525, 480, 465, 450, 
    435, 420, 405, 390, 375, 360, 60, 30, 0, -30, 
    -60, -90, -120, -150, -180, -420, -435, -450, -465, -480, 
    -495, -510, -525, -540, -480, -465, -450, -435, -420, -405, 
    -390, -375, -360, -60, -30, 0, 30, 60, 90, 120, 
    150, 180, 420, 435, 450, 465, 480, 495, 510, 525, 
    540, 465, 450, 435, 420, 405, -15, -45, -75, -105, 
    -135, -480, -495, -510, -525, -540, -465, -450, -435, -420, 
    -405, 15, 45, 75, 105, 135, 480, 495, 510, 525, 
    540, 450, 435, -90, -120, -540, -555, -450, -435, 90, 
    120, 540, 555
  };

  const float ytemp[1183] = {
    0, 0, 25.9808, 25.9808, 0, 
    -25.9808, -25.9808, 0, 25.9808, 51.9615, 
    51.9615, 51.9615, 25.9808, 0, -25.9808, 
    -51.9615, -51.9615, -51.9615, -25.9808, 0, 
    25.9808, 51.9615, 77.9423, 77.9423, 77.9423, 
    77.9423, 51.9615, 25.9808, 0, -25.9808, 
    -51.9615, -77.9423, -77.9423, -77.9423, -77.9423, 
    -51.9615, -25.9808, 0, 25.9808, 51.9615, 
    77.9423, 103.923, 103.923, 103.923, 103.923, 
    103.923, 77.9423, 51.9615, 25.9808, 0, 
    -25.9808, -51.9615, -77.9423, -103.923, -103.923, 
    -103.923, -103.923, -103.923, -77.9423, -51.9615, 
    -25.9808, 0, 25.9808, 51.9615, 77.9423, 
    103.923, 129.904, 129.904, 129.904, 129.904, 
    129.904, 129.904, 103.923, 77.9423, 51.9615, 
    25.9808, 0, -25.9808, -51.9615, -77.9423, 
    -103.923, -129.904, -129.904, -129.904, -129.904, 
    -129.904, -129.904, -103.923, -77.9423, -51.9615, 
    -25.9808, 0, 25.9808, 51.9615, 77.9423, 
    103.923, 129.904, 155.885, 155.885, 155.885, 
    155.885, 155.885, 155.885, 155.885, 129.904, 
    103.923, 77.9423, 51.9615, 25.9808, 0, 
    -25.9808, -51.9615, -77.9423, -103.923, -129.904, 
    -155.885, -155.885, -155.885, -155.885, -155.885, 
    -155.885, -155.885, -129.904, -103.923, -77.9423, 
    -51.9615, -25.9808, 0, 25.9808, 51.9615, 
    77.9423, 103.923, 129.904, 155.885, 181.865, 
    181.865, 181.865, 181.865, 181.865, 181.865, 
    181.865, 181.865, 155.885, 129.904, 103.923, 
    77.9423, 51.9615, 25.9808, 0, -25.9808, 
    -51.9615, -77.9423, -103.923, -129.904, -155.885, 
    -181.865, -181.865, -181.865, -181.865, -181.865, 
    -181.865, -181.865, -181.865, -155.885, -129.904, 
    -103.923, -77.9423, -51.9615, -25.9808, 0, 
    25.9808, 51.9615, 77.9423, 103.923, 129.904, 
    155.885, 181.865, 207.846, 207.846, 207.846, 
    207.846, 207.846, 207.846, 207.846, 207.846, 
    207.846, 181.865, 155.885, 129.904, 103.923, 
    77.9423, 51.9615, 25.9808, 0, -25.9808, 
    -51.9615, -77.9423, -103.923, -129.904, -155.885, 
    -181.865, -207.846, -207.846, -207.846, -207.846, 
    -207.846, -207.846, -207.846, -207.846, -207.846, 
    -181.865, -155.885, -129.904, -103.923, -77.9423, 
    -51.9615, -25.9808, 0, 25.9808, 51.9615, 
    77.9423, 103.923, 129.904, 155.885, 181.865, 
    207.846, 233.827, 233.827, 233.827, 233.827, 
    233.827, 233.827, 233.827, 233.827, 233.827, 
    233.827, 207.846, 181.865, 155.885, 129.904, 
    103.923, 77.9423, 51.9615, 25.9808, 0, 
    -25.9808, -51.9615, -77.9423, -103.923, -129.904, 
    -155.885, -181.865, -207.846, -233.827, -233.827, 
    -233.827, -233.827, -233.827, -233.827, -233.827, 
    -233.827, -233.827, -233.827, -207.846, -181.865, 
    -155.885, -129.904, -103.923, -77.9423, -51.9615, 
    -25.9808, 0, 25.9808, 51.9615, 77.9423, 
    103.923, 129.904, 155.885, 181.865, 207.846, 
    233.827, 259.808, 259.808, 259.808, 259.808, 
    259.808, 259.808, 259.808, 259.808, 259.808, 
    259.808, 259.808, 233.827, 207.846, 181.865, 
    155.885, 129.904, 103.923, 77.9423, 51.9615, 
    25.9808, 0, -25.9808, -51.9615, -77.9423, 
    -103.923, -129.904, -155.885, -181.865, -207.846, 
    -233.827, -259.808, -259.808, -259.808, -259.808, 
    -259.808, -259.808, -259.808, -259.808, -259.808, 
    -259.808, -259.808, -233.827, -207.846, -181.865, 
    -155.885, -129.904, -103.923, -77.9423, -51.9615, 
    -25.9808, 0, 25.9808, 51.9615, 77.9423, 
    103.923, 129.904, 155.885, 181.865, 207.846, 
    233.827, 259.808, 285.788, 285.788, 285.788, 
    285.788, 285.788, 285.788, 285.788, 285.788, 
    285.788, 285.788, 285.788, 285.788, 259.808, 
    233.827, 207.846, 181.865, 155.885, 129.904, 
    103.923, 77.9423, 51.9615, 25.9808, 0, 
    -25.9808, -51.9615, -77.9423, -103.923, -129.904, 
    -155.885, -181.865, -207.846, -233.827, -259.808, 
    -285.788, -285.788, -285.788, -285.788, -285.788, 
    -285.788, -285.788, -285.788, -285.788, -285.788, 
    -285.788, -285.788, -259.808, -233.827, -207.846, 
    -181.865, -155.885, -129.904, -103.923, -77.9423, 
    -51.9615, -25.9808, 0, 25.9808, 51.9615, 
    77.9423, 103.923, 129.904, 155.885, 181.865, 
    207.846, 233.827, 259.808, 285.788, 311.769, 
    311.769, 311.769, 311.769, 311.769, 311.769, 
    311.769, 311.769, 311.769, 311.769, 311.769, 
    311.769, 311.769, 285.788, 259.808, 233.827, 
    207.846, 181.865, 155.885, 129.904, 103.923, 
    77.9423, 51.9615, 25.9808, 0, -25.9808, 
    -51.9615, -77.9423, -103.923, -129.904, -155.885, 
    -181.865, -207.846, -233.827, -259.808, -285.788, 
    -311.769, -311.769, -311.769, -311.769, -311.769, 
    -311.769, -311.769, -311.769, -311.769, -311.769, 
    -311.769, -311.769, -311.769, -285.788, -259.808, 
    -233.827, -207.846, -181.865, -155.885, -129.904, 
    -103.923, -77.9423, -51.9615, -25.9808, 0, 
    25.9808, 51.9615, 77.9423, 103.923, 129.904, 
    155.885, 181.865, 207.846, 233.827, 259.808, 
    285.788, 311.769, 337.75, 337.75, 337.75, 
    337.75, 337.75, 337.75, 337.75, 337.75, 
    337.75, 337.75, 337.75, 337.75, 337.75, 
    337.75, 311.769, 285.788, 259.808, 233.827, 
    207.846, 181.865, 155.885, 129.904, 103.923, 
    77.9423, 51.9615, 25.9808, 0, -25.9808, 
    -51.9615, -77.9423, -103.923, -129.904, -155.885, 
    -181.865, -207.846, -233.827, -259.808, -285.788, 
    -311.769, -337.75, -337.75, -337.75, -337.75, 
    -337.75, -337.75, -337.75, -337.75, -337.75, 
    -337.75, -337.75, -337.75, -337.75, -337.75, 
    -311.769, -285.788, -259.808, -233.827, -207.846, 
    -181.865, -155.885, -129.904, -103.923, -77.9423, 
    -51.9615, -25.9808, 0, 25.9808, 51.9615, 
    77.9423, 103.923, 129.904, 155.885, 181.865, 
    207.846, 233.827, 259.808, 285.788, 311.769, 
    337.75, 363.731, 363.731, 363.731, 363.731, 
    363.731, 363.731, 363.731, 363.731, 363.731, 
    363.731, 363.731, 363.731, 363.731, 363.731, 
    363.731, 337.75, 311.769, 285.788, 259.808, 
    233.827, 207.846, 181.865, 155.885, 129.904, 
    103.923, 77.9423, 51.9615, 25.9808, 0, 
    -25.9808, -51.9615, -77.9423, -103.923, -129.904, 
    -155.885, -181.865, -207.846, -233.827, -259.808, 
    -285.788, -311.769, -337.75, -363.731, -363.731, 
    -363.731, -363.731, -363.731, -363.731, -363.731, 
    -363.731, -363.731, -363.731, -363.731, -363.731, 
    -363.731, -363.731, -363.731, -337.75, -311.769, 
    -285.788, -259.808, -233.827, -207.846, -181.865, 
    -155.885, -129.904, -103.923, -77.9423, -51.9615, 
    -25.9808, 0, 25.9808, 51.9615, 77.9423, 
    103.923, 129.904, 155.885, 181.865, 207.846, 
    233.827, 259.808, 285.788, 311.769, 337.75, 
    363.731, 389.711, 389.711, 389.711, 389.711, 
    389.711, 389.711, 389.711, 389.711, 389.711, 
    389.711, 389.711, 389.711, 389.711, 389.711, 
    389.711, 389.711, 363.731, 337.75, 311.769, 
    285.788, 259.808, 233.827, 207.846, 181.865, 
    155.885, 129.904, 103.923, 77.9423, 51.9615, 
    25.9808, 0, -25.9808, -51.9615, -77.9423, 
    -103.923, -129.904, -155.885, -181.865, -207.846, 
    -233.827, -259.808, -285.788, -311.769, -337.75, 
    -363.731, -389.711, -389.711, -389.711, -389.711, 
    -389.711, -389.711, -389.711, -389.711, -389.711, 
    -389.711, -389.711, -389.711, -389.711, -389.711, 
    -389.711, -389.711, -363.731, -337.75, -311.769, 
    -285.788, -259.808, -233.827, -207.846, -181.865, 
    -155.885, -129.904, -103.923, -77.9423, -51.9615, 
    -25.9808, 0, 25.9808, 51.9615, 77.9423, 
    103.923, 129.904, 155.885, 181.865, 207.846, 
    233.827, 259.808, 285.788, 311.769, 337.75, 
    363.731, 389.711, 415.692, 415.692, 415.692, 
    415.692, 415.692, 415.692, 415.692, 415.692, 
    415.692, 415.692, 415.692, 415.692, 415.692, 
    415.692, 415.692, 415.692, 415.692, 389.711, 
    363.731, 337.75, 311.769, 285.788, 259.808, 
    233.827, 207.846, 181.865, 155.885, 129.904, 
    103.923, 77.9423, 51.9615, 25.9808, 0, 
    -25.9808, -51.9615, -77.9423, -103.923, -129.904, 
    -155.885, -181.865, -207.846, -233.827, -259.808, 
    -285.788, -311.769, -337.75, -363.731, -389.711, 
    -415.692, -415.692, -415.692, -415.692, -415.692, 
    -415.692, -415.692, -415.692, -415.692, -415.692, 
    -415.692, -415.692, -415.692, -415.692, -415.692, 
    -415.692, -415.692, -389.711, -363.731, -337.75, 
    -311.769, -285.788, -259.808, -233.827, -207.846, 
    -181.865, -155.885, -129.904, -103.923, -77.9423, 
    -51.9615, -25.9808, 0, 25.9808, 51.9615, 
    77.9423, 103.923, 129.904, 155.885, 181.865, 
    207.846, 233.827, 259.808, 285.788, 311.769, 
    337.75, 363.731, 389.711, 415.692, 441.673, 
    441.673, 441.673, 441.673, 441.673, 441.673, 
    441.673, 441.673, 441.673, 441.673, 441.673, 
    441.673, 441.673, 441.673, 441.673, 441.673, 
    441.673, 441.673, 415.692, 389.711, 363.731, 
    337.75, 311.769, 285.788, 259.808, 233.827, 
    207.846, 181.865, 155.885, 129.904, 103.923, 
    77.9423, 51.9615, 25.9808, 0, -25.9808, 
    -51.9615, -77.9423, -103.923, -129.904, -155.885, 
    -181.865, -207.846, -233.827, -259.808, -285.788, 
    -311.769, -337.75, -363.731, -389.711, -415.692, 
    -441.673, -441.673, -441.673, -441.673, -441.673, 
    -441.673, -441.673, -441.673, -441.673, -441.673, 
    -441.673, -441.673, -441.673, -441.673, -441.673, 
    -441.673, -441.673, -441.673, -415.692, -389.711, 
    -363.731, -337.75, -311.769, -285.788, -259.808, 
    -233.827, -207.846, -181.865, -155.885, -129.904, 
    -103.923, -77.9423, -51.9615, -25.9808, 51.9615, 
    77.9423, 103.923, 129.904, 155.885, 181.865, 
    207.846, 233.827, 259.808, 285.788, 311.769, 
    337.75, 363.731, 389.711, 415.692, 441.673, 
    467.654, 467.654, 467.654, 467.654, 467.654, 
    467.654, 467.654, 467.654, 467.654, 467.654, 
    467.654, 467.654, 467.654, 467.654, 467.654, 
    467.654, 415.692, 389.711, 363.731, 337.75, 
    311.769, 285.788, 259.808, 233.827, 207.846, 
    181.865, 155.885, 129.904, 103.923, 77.9423, 
    51.9615, 25.9808, -51.9615, -77.9423, -103.923, 
    -129.904, -155.885, -181.865, -207.846, -233.827, 
    -259.808, -285.788, -311.769, -337.75, -363.731, 
    -389.711, -415.692, -441.673, -467.654, -467.654, 
    -467.654, -467.654, -467.654, -467.654, -467.654, 
    -467.654, -467.654, -467.654, -467.654, -467.654, 
    -467.654, -467.654, -467.654, -467.654, -415.692, 
    -389.711, -363.731, -337.75, -311.769, -285.788, 
    -259.808, -233.827, -207.846, -181.865, -155.885, 
    -129.904, -103.923, -77.9423, -51.9615, -25.9808, 
    129.904, 155.885, 181.865, 207.846, 233.827, 
    259.808, 285.788, 311.769, 337.75, 363.731, 
    389.711, 415.692, 493.634, 493.634, 493.634, 
    493.634, 493.634, 493.634, 493.634, 493.634, 
    493.634, 493.634, 493.634, 493.634, 363.731, 
    337.75, 311.769, 285.788, 259.808, 233.827, 
    207.846, 181.865, 155.885, 129.904, 103.923, 
    77.9423, -129.904, -155.885, -181.865, -207.846, 
    -233.827, -259.808, -285.788, -311.769, -337.75, 
    -363.731, -389.711, -415.692, -493.634, -493.634, 
    -493.634, -493.634, -493.634, -493.634, -493.634, 
    -493.634, -493.634, -493.634, -493.634, -493.634, 
    -363.731, -337.75, -311.769, -285.788, -259.808, 
    -233.827, -207.846, -181.865, -155.885, -129.904, 
    -103.923, -77.9423, 207.846, 233.827, 259.808, 
    285.788, 311.769, 337.75, 363.731, 389.711, 
    415.692, 519.615, 519.615, 519.615, 519.615, 
    519.615, 519.615, 519.615, 519.615, 519.615, 
    311.769, 285.788, 259.808, 233.827, 207.846, 
    181.865, 155.885, 129.904, 103.923, -207.846, 
    -233.827, -259.808, -285.788, -311.769, -337.75, 
    -363.731, -389.711, -415.692, -519.615, -519.615, 
    -519.615, -519.615, -519.615, -519.615, -519.615, 
    -519.615, -519.615, -311.769, -285.788, -259.808, 
    -233.827, -207.846, -181.865, -155.885, -129.904, 
    -103.923, 285.788, 311.769, 337.75, 363.731, 
    389.711, 545.596, 545.596, 545.596, 545.596, 
    545.596, 259.808, 233.827, 207.846, 181.865, 
    155.885, -285.788, -311.769, -337.75, -363.731, 
    -389.711, -545.596, -545.596, -545.596, -545.596, 
    -545.596, -259.808, -233.827, -207.846, -181.865, 
    -155.885, 363.731, 389.711, 571.577, 571.577, 
    207.846, 181.865, -363.731, -389.711, -571.577, 
    -571.577, -207.846, -181.865
  };

  const float rtemp[1183] = {
	30, 30, 30, 30, 30, 30, 30, 30, // 0
	30, 30, 30, 30, 30, 30, 30, 30, // 8
	30, 30, 30, 30, 30, 30, 30, 30, // 16
	30, 30, 30, 30, 30, 30, 30, 30, // 24
	30, 30, 30, 30, 30, 30, 30, 30, // 32
	30, 30, 30, 30, 30, 30, 30, 30, // 40
	30, 30, 30, 30, 30, 30, 30, 30, // 48
	30, 30, 30, 30, 30, 30, 30, 30, // 56
	30, 30, 30, 30, 30, 30, 30, 30, // 64
	30, 30, 30, 30, 30, 30, 30, 30, // 72
	30, 30, 30, 30, 30, 30, 30, 30, // 80
	30, 30, 30, 30, 30, 30, 30, 30, // 88
	30, 30, 30, 30, 30, 30, 30, 30, // 96
	30, 30, 30, 30, 30, 30, 30, 30, // 104
	30, 30, 30, 30, 30, 30, 30, 30, // 112
	30, 30, 30, 30, 30, 30, 30, 30, // 120
	30, 30, 30, 30, 30, 30, 30, 30, // 128
	30, 30, 30, 30, 30, 30, 30, 30, // 136
	30, 30, 30, 30, 30, 30, 30, 30, // 144
	30, 30, 30, 30, 30, 30, 30, 30, // 152
	30, 30, 30, 30, 30, 30, 30, 30, // 160
	30, 30, 30, 30, 30, 30, 30, 30, // 168
	30, 30, 30, 30, 30, 30, 30, 30, // 176
	30, 30, 30, 30, 30, 30, 30, 30, // 184
	30, 30, 30, 30, 30, 30, 30, 30, // 192
	30, 30, 30, 30, 30, 30, 30, 30, // 200
	30, 30, 30, 30, 30, 30, 30, 30, // 208
	30, 30, 30, 30, 30, 30, 30, 30, // 216
	30, 30, 30, 30, 30, 30, 30, 30, // 224
	30, 30, 30, 30, 30, 30, 30, 30, // 232
	30, 30, 30, 30, 30, 30, 30, 30, // 240
	30, 30, 30, 30, 30, 30, 30, 30, // 248
	30, 30, 30, 30, 30, 30, 30, 30, // 256
	30, 30, 30, 30, 30, 30, 30, 30, // 264
	30, 30, 30, 30, 30, 30, 30, 30, // 272
	30, 30, 30, 30, 30, 30, 30, 30, // 280
	30, 30, 30, 30, 30, 30, 30, 30, // 288
	30, 30, 30, 30, 30, 30, 30, 30, // 296
	30, 30, 30, 30, 30, 30, 30, 30, // 304
	30, 30, 30, 30, 30, 30, 30, 30, // 312
	30, 30, 30, 30, 30, 30, 30, 30, // 320
	30, 30, 30, 30, 30, 30, 30, 30, // 328
	30, 30, 30, 30, 30, 30, 30, 30, // 336
	30, 30, 30, 30, 30, 30, 30, 30, // 344
	30, 30, 30, 30, 30, 30, 30, 30, // 352
	30, 30, 30, 30, 30, 30, 30, 30, // 360
	30, 30, 30, 30, 30, 30, 30, 30, // 368
	30, 30, 30, 30, 30, 30, 30, 30, // 376
	30, 30, 30, 30, 30, 30, 30, 30, // 384
	30, 30, 30, 30, 30, 30, 30, 30, // 392
	30, 30, 30, 30, 30, 30, 30, 30, // 400
	30, 30, 30, 30, 30, 30, 30, 30, // 408
	30, 30, 30, 30, 30, 30, 30, 30, // 416
	30, 30, 30, 30, 30, 30, 30, 30, // 424
	30, 30, 30, 30, 30, 30, 30, 30, // 432
	30, 30, 30, 30, 30, 30, 30, 30, // 440
	30, 30, 30, 30, 30, 30, 30, 30, // 448
	30, 30, 30, 30, 30, 30, 30, 30, // 456
	30, 30, 30, 30, 30, 30, 30, 30, // 464
	30, 30, 30, 30, 30, 30, 30, 30, // 472
	30, 30, 30, 30, 30, 30, 30, 30, // 480
	30, 30, 30, 30, 30, 30, 30, 30, // 488
	30, 30, 30, 30, 30, 30, 30, 30, // 496
	30, 30, 30, 30, 30, 30, 30, 30, // 504
	30, 30, 30, 30, 30, 30, 30, 30, // 512
	30, 30, 30, 30, 30, 30, 30, 30, // 520
	30, 30, 30, 30, 30, 30, 30, 30, // 528
	30, 30, 30, 30, 30, 30, 30, 30, // 536
	30, 30, 30, 30, 30, 30, 30, 30, // 544
	30, 30, 30, 30, 30, 30, 30, 30, // 552
	30, 30, 30, 30, 30, 30, 30, 30, // 560
	30, 30, 30, 30, 30, 30, 30, 30, // 568
	30, 30, 30, 30, 30, 30, 30, 30, // 576
	30, 30, 30, 30, 30, 30, 30, 30, // 584
	30, 30, 30, 30, 30, 30, 30, 30, // 592
	30, 30, 30, 30, 30, 30, 30, 30, // 600
	30, 30, 30, 30, 30, 30, 30, 30, // 608
	30, 30, 30, 30, 30, 30, 30, 30, // 616
	30, 30, 30, 30, 30, 30, 30, 30, // 624
	30, 30, 30, 30, 30, 30, 30, 30, // 632
	30, 30, 30, 30, 30, 30, 30, 30, // 640
	30, 30, 30, 30, 30, 30, 30, 30, // 648
	30, 30, 30, 30, 30, 30, 30, 30, // 656
	30, 30, 30, 30, 30, 30, 30, 30, // 664
	30, 30, 30, 30, 30, 30, 30, 30, // 672
	30, 30, 30, 30, 30, 30, 30, 30, // 680
	30, 30, 30, 30, 30, 30, 30, 30, // 688
	30, 30, 30, 30, 30, 30, 30, 30, // 696
	30, 30, 30, 30, 30, 30, 30, 30, // 704
	30, 30, 30, 30, 30, 30, 30, 30, // 712
	30, 30, 30, 30, 30, 30, 30, 30, // 720
	30, 30, 30, 30, 30, 30, 30, 30, // 728
	30, 30, 30, 30, 30, 30, 30, 30, // 736
	30, 30, 30, 30, 30, 30, 30, 30, // 744
	30, 30, 30, 30, 30, 30, 30, 30, // 752
	30, 30, 30, 30, 30, 30, 30, 30, // 760
	30, 30, 30, 30, 30, 30, 30, 30, // 768
	30, 30, 30, 30, 30, 30, 30, 30, // 776
	30, 30, 30, 30, 30, 30, 30, 30, // 784
	30, 30, 30, 30, 30, 30, 30, 30, // 792
	30, 30, 30, 30, 30, 30, 30, 30, // 800
	30, 30, 30, 30, 30, 30, 30, 30, // 808
	30, 30, 30, 30, 30, 30, 30, 30, // 816
	30, 30, 30, 30, 30, 30, 30, 30, // 824
	30, 30, 30, 30, 30, 30, 30, 30, // 832
	30, 30, 30, 30, 30, 30, 30, 30, // 840
	30, 30, 30, 30, 30, 30, 30, 30, // 848
	30, 30, 30, 30, 30, 30, 30, 30, // 856
	30, 30, 30, 30, 30, 30, 30, 30, // 864
	30, 30, 30, 30, 30, 30, 30, 30, // 872
	30, 30, 30, 30, 30, 30, 30, 30, // 880
	30, 30, 30, 30, 30, 30, 30, 30, // 888
	30, 30, 30, 30, 30, 30, 30, 30, // 896
	30, 30, 30, 30, 30, 30, 30, 30, // 904
	30, 30, 30, 30, 30, 30, 30, 30, // 912
	30, 30, 30, 30, 30, 30, 30, 30, // 920
	30, 30, 30, 30, 30, 30, 30, 30, // 928
	30, 30, 30, 30, 30, 30, 30, 30, // 936
	30, 30, 30, 30, 30, 30, 30, 30, // 944
	30, 30, 30, 30, 30, 30, 30, 30, // 952
	30, 30, 30, 30, 30, 30, 30, 30, // 960
	30, 30, 30, 30, 30, 30, 30, 30, // 968
	30, 30, 30, 30, 30, 30, 30, 30, // 976
	30, 30, 30, 30, 30, 30, 30, 30, // 984
	30, 30, 30, 30, 30, 30, 30, 30, // 992
	30, 30, 30, 30, 30, 30, 30, 30, // 1000
	30, 30, 30, 30, 30, 30, 30, 30, // 1008
	30, 30, 30, 30, 30, 30, 30, 30, // 1016
	30, 30, 30, 30, 30, 30, 30, 30, // 1032
	30, 30, 30, 30, 30, 30, 30, 30, // 1040
	30, 30, 30, 30, 30, 30, 30, 30, // 1048
	30, 30, 30, 30, 30, 30, 30, 30, // 1056
	30, 30, 30, 30, 30, 30, 30, 30, // 1064
	30, 30, 30, 30, 30, 30, 30, 30, // 1072
	30, 30, 30, 30, 30, 30, 30, 30, // 1080
	30, 30, 30, 30, 30, 30, 30, 30, // 1088
	30, 30, 30, 30, 30, 30, 30, 30, // 1096
	30, 30, 30, 30, 30, 30, 30, 30, // 1104
	30, 30, 30, 30, 30, 30, 30, 30, // 1112
	30, 30, 30, 30, 30, 30, 30, 30, // 1120
	30, 30, 30, 30, 30, 30, 30, 30, // 1128
	30, 30, 30, 30, 30, 30, 30, 30, // 1136
	30, 30, 30, 30, 30, 30, 30, 30, // 1144
	30, 30, 30, 30, 30, 30, 30, 30, // 1152
	30, 30, 30, 30, 30, 30, 30, 30, // 1160
	30, 30, 30, 30, 30, 30, 30, 30, // 1168
	30, 30, 30, 30, 30, 30, 30, 30, // 1176
	30, 30, 30, 30, 30, 30, 30 // 1183
};

    //
    //   fill the pixels list with this data
    //

    for (UInt_t i=0; i<GetNumPixels(); i++)
        (*this)[i].Set(xtemp[i], ytemp[i], rtemp[i]) ;
}

// --------------------------------------------------------------------------
//
//  This fills the next neighbor information from a table into the pixel
//  objects.
//
void MGeomCamMagic1183::CreateNN()
{
  const Short_t nn[1183][6] = {       // Neighbours of [pixid=0-1182]
    { 1, 2, 3, 4, 5, 6 },
    { 0, 2, 6, 7, 8, 18 },
    { 0, 1, 3, 8, 9, 10 },
    { 0, 2, 4, 10, 11, 12 },
    { 0, 3, 5, 12, 13, 14 },
    { 0, 4, 6, 14, 15, 16 },
    { 0, 1, 5, 16, 17, 18 },
    { 1, 8, 18, 19, 20, 36 },
    { 1, 2, 7, 9, 20, 21 },
    { 2, 8, 10, 21, 22, 23 },
    { 2, 3, 9, 11, 23, 24 },
    { 3, 10, 12, 24, 25, 26 },
    { 3, 4, 11, 13, 26, 27 },
    { 4, 12, 14, 27, 28, 29 },
    { 4, 5, 13, 15, 29, 30 },
    { 5, 14, 16, 30, 31, 32 },
    { 5, 6, 15, 17, 32, 33 },
    { 6, 16, 18, 33, 34, 35 },
    { 1, 6, 7, 17, 35, 36 },
    { 7, 20, 36, 37, 38, 60 },
    { 7, 8, 19, 21, 38, 39 },
    { 8, 9, 20, 22, 39, 40 },
    { 9, 21, 23, 40, 41, 42 },
    { 9, 10, 22, 24, 42, 43 },
    { 10, 11, 23, 25, 43, 44 },
    { 11, 24, 26, 44, 45, 46 },
    { 11, 12, 25, 27, 46, 47 },
    { 12, 13, 26, 28, 47, 48 },
    { 13, 27, 29, 48, 49, 50 },
    { 13, 14, 28, 30, 50, 51 },
    { 14, 15, 29, 31, 51, 52 },
    { 15, 30, 32, 52, 53, 54 },
    { 15, 16, 31, 33, 54, 55 },
    { 16, 17, 32, 34, 55, 56 },
    { 17, 33, 35, 56, 57, 58 },
    { 17, 18, 34, 36, 58, 59 },
    { 7, 18, 19, 35, 59, 60 },
    { 19, 38, 60, 61, 62, 90 },
    { 19, 20, 37, 39, 62, 63 },
    { 20, 21, 38, 40, 63, 64 },
    { 21, 22, 39, 41, 64, 65 },
    { 22, 40, 42, 65, 66, 67 },
    { 22, 23, 41, 43, 67, 68 },
    { 23, 24, 42, 44, 68, 69 },
    { 24, 25, 43, 45, 69, 70 },
    { 25, 44, 46, 70, 71, 72 },
    { 25, 26, 45, 47, 72, 73 },
    { 26, 27, 46, 48, 73, 74 },
    { 27, 28, 47, 49, 74, 75 },
    { 28, 48, 50, 75, 76, 77 },
    { 28, 29, 49, 51, 77, 78 },
    { 29, 30, 50, 52, 78, 79 },
    { 30, 31, 51, 53, 79, 80 },
    { 31, 52, 54, 80, 81, 82 },
    { 31, 32, 53, 55, 82, 83 },
    { 32, 33, 54, 56, 83, 84 },
    { 33, 34, 55, 57, 84, 85 },
    { 34, 56, 58, 85, 86, 87 },
    { 34, 35, 57, 59, 87, 88 },
    { 35, 36, 58, 60, 88, 89 },
    { 19, 36, 37, 59, 89, 90 },
    { 37, 62, 90, 91, 92, 126 },
    { 37, 38, 61, 63, 92, 93 },
    { 38, 39, 62, 64, 93, 94 },
    { 39, 40, 63, 65, 94, 95 },
    { 40, 41, 64, 66, 95, 96 },
    { 41, 65, 67, 96, 97, 98 },
    { 41, 42, 66, 68, 98, 99 },
    { 42, 43, 67, 69, 99, 100 },
    { 43, 44, 68, 70, 100, 101 },
    { 44, 45, 69, 71, 101, 102 },
    { 45, 70, 72, 102, 103, 104 },
    { 45, 46, 71, 73, 104, 105 },
    { 46, 47, 72, 74, 105, 106 },
    { 47, 48, 73, 75, 106, 107 },
    { 48, 49, 74, 76, 107, 108 },
    { 49, 75, 77, 108, 109, 110 },
    { 49, 50, 76, 78, 110, 111 },
    { 50, 51, 77, 79, 111, 112 },
    { 51, 52, 78, 80, 112, 113 },
    { 52, 53, 79, 81, 113, 114 },
    { 53, 80, 82, 114, 115, 116 },
    { 53, 54, 81, 83, 116, 117 },
    { 54, 55, 82, 84, 117, 118 },
    { 55, 56, 83, 85, 118, 119 },
    { 56, 57, 84, 86, 119, 120 },
    { 57, 85, 87, 120, 121, 122 },
    { 57, 58, 86, 88, 122, 123 },
    { 58, 59, 87, 89, 123, 124 },
    { 59, 60, 88, 90, 124, 125 },
    { 37, 60, 61, 89, 125, 126 },
    { 61, 92, 126, 127, 128, 168 },
    { 61, 62, 91, 93, 128, 129 },
    { 62, 63, 92, 94, 129, 130 },
    { 63, 64, 93, 95, 130, 131 },
    { 64, 65, 94, 96, 131, 132 },
    { 65, 66, 95, 97, 132, 133 },
    { 66, 96, 98, 133, 134, 135 },
    { 66, 67, 97, 99, 135, 136 },
    { 67, 68, 98, 100, 136, 137 },
    { 68, 69, 99, 101, 137, 138 },
    { 69, 70, 100, 102, 138, 139 },
    { 70, 71, 101, 103, 139, 140 },
    { 71, 102, 104, 140, 141, 142 },
    { 71, 72, 103, 105, 142, 143 },
    { 72, 73, 104, 106, 143, 144 },
    { 73, 74, 105, 107, 144, 145 },
    { 74, 75, 106, 108, 145, 146 },
    { 75, 76, 107, 109, 146, 147 },
    { 76, 108, 110, 147, 148, 149 },
    { 76, 77, 109, 111, 149, 150 },
    { 77, 78, 110, 112, 150, 151 },
    { 78, 79, 111, 113, 151, 152 },
    { 79, 80, 112, 114, 152, 153 },
    { 80, 81, 113, 115, 153, 154 },
    { 81, 114, 116, 154, 155, 156 },
    { 81, 82, 115, 117, 156, 157 },
    { 82, 83, 116, 118, 157, 158 },
    { 83, 84, 117, 119, 158, 159 },
    { 84, 85, 118, 120, 159, 160 },
    { 85, 86, 119, 121, 160, 161 },
    { 86, 120, 122, 161, 162, 163 },
    { 86, 87, 121, 123, 163, 164 },
    { 87, 88, 122, 124, 164, 165 },
    { 88, 89, 123, 125, 165, 166 },
    { 89, 90, 124, 126, 166, 167 },
    { 61, 90, 91, 125, 167, 168 },
    { 91, 128, 168, 169, 170, 216 },
    { 91, 92, 127, 129, 170, 171 },
    { 92, 93, 128, 130, 171, 172 },
    { 93, 94, 129, 131, 172, 173 },
    { 94, 95, 130, 132, 173, 174 },
    { 95, 96, 131, 133, 174, 175 },
    { 96, 97, 132, 134, 175, 176 },
    { 97, 133, 135, 176, 177, 178 },
    { 97, 98, 134, 136, 178, 179 },
    { 98, 99, 135, 137, 179, 180 },
    { 99, 100, 136, 138, 180, 181 },
    { 100, 101, 137, 139, 181, 182 },
    { 101, 102, 138, 140, 182, 183 },
    { 102, 103, 139, 141, 183, 184 },
    { 103, 140, 142, 184, 185, 186 },
    { 103, 104, 141, 143, 186, 187 },
    { 104, 105, 142, 144, 187, 188 },
    { 105, 106, 143, 145, 188, 189 },
    { 106, 107, 144, 146, 189, 190 },
    { 107, 108, 145, 147, 190, 191 },
    { 108, 109, 146, 148, 191, 192 },
    { 109, 147, 149, 192, 193, 194 },
    { 109, 110, 148, 150, 194, 195 },
    { 110, 111, 149, 151, 195, 196 },
    { 111, 112, 150, 152, 196, 197 },
    { 112, 113, 151, 153, 197, 198 },
    { 113, 114, 152, 154, 198, 199 },
    { 114, 115, 153, 155, 199, 200 },
    { 115, 154, 156, 200, 201, 202 },
    { 115, 116, 155, 157, 202, 203 },
    { 116, 117, 156, 158, 203, 204 },
    { 117, 118, 157, 159, 204, 205 },
    { 118, 119, 158, 160, 205, 206 },
    { 119, 120, 159, 161, 206, 207 },
    { 120, 121, 160, 162, 207, 208 },
    { 121, 161, 163, 208, 209, 210 },
    { 121, 122, 162, 164, 210, 211 },
    { 122, 123, 163, 165, 211, 212 },
    { 123, 124, 164, 166, 212, 213 },
    { 124, 125, 165, 167, 213, 214 },
    { 125, 126, 166, 168, 214, 215 },
    { 91, 126, 127, 167, 215, 216 },
    { 127, 170, 216, 217, 218, 270 },
    { 127, 128, 169, 171, 218, 219 },
    { 128, 129, 170, 172, 219, 220 },
    { 129, 130, 171, 173, 220, 221 },
    { 130, 131, 172, 174, 221, 222 },
    { 131, 132, 173, 175, 222, 223 },
    { 132, 133, 174, 176, 223, 224 },
    { 133, 134, 175, 177, 224, 225 },
    { 134, 176, 178, 225, 226, 227 },
    { 134, 135, 177, 179, 227, 228 },
    { 135, 136, 178, 180, 228, 229 },
    { 136, 137, 179, 181, 229, 230 },
    { 137, 138, 180, 182, 230, 231 },
    { 138, 139, 181, 183, 231, 232 },
    { 139, 140, 182, 184, 232, 233 },
    { 140, 141, 183, 185, 233, 234 },
    { 141, 184, 186, 234, 235, 236 },
    { 141, 142, 185, 187, 236, 237 },
    { 142, 143, 186, 188, 237, 238 },
    { 143, 144, 187, 189, 238, 239 },
    { 144, 145, 188, 190, 239, 240 },
    { 145, 146, 189, 191, 240, 241 },
    { 146, 147, 190, 192, 241, 242 },
    { 147, 148, 191, 193, 242, 243 },
    { 148, 192, 194, 243, 244, 245 },
    { 148, 149, 193, 195, 245, 246 },
    { 149, 150, 194, 196, 246, 247 },
    { 150, 151, 195, 197, 247, 248 },
    { 151, 152, 196, 198, 248, 249 },
    { 152, 153, 197, 199, 249, 250 },
    { 153, 154, 198, 200, 250, 251 },
    { 154, 155, 199, 201, 251, 252 },
    { 155, 200, 202, 252, 253, 254 },
    { 155, 156, 201, 203, 254, 255 },
    { 156, 157, 202, 204, 255, 256 },
    { 157, 158, 203, 205, 256, 257 },
    { 158, 159, 204, 206, 257, 258 },
    { 159, 160, 205, 207, 258, 259 },
    { 160, 161, 206, 208, 259, 260 },
    { 161, 162, 207, 209, 260, 261 },
    { 162, 208, 210, 261, 262, 263 },
    { 162, 163, 209, 211, 263, 264 },
    { 163, 164, 210, 212, 264, 265 },
    { 164, 165, 211, 213, 265, 266 },
    { 165, 166, 212, 214, 266, 267 },
    { 166, 167, 213, 215, 267, 268 },
    { 167, 168, 214, 216, 268, 269 },
    { 127, 168, 169, 215, 269, 270 },
    { 169, 218, 270, 271, 272, 330 },
    { 169, 170, 217, 219, 272, 273 },
    { 170, 171, 218, 220, 273, 274 },
    { 171, 172, 219, 221, 274, 275 },
    { 172, 173, 220, 222, 275, 276 },
    { 173, 174, 221, 223, 276, 277 },
    { 174, 175, 222, 224, 277, 278 },
    { 175, 176, 223, 225, 278, 279 },
    { 176, 177, 224, 226, 279, 280 },
    { 177, 225, 227, 280, 281, 282 },
    { 177, 178, 226, 228, 282, 283 },
    { 178, 179, 227, 229, 283, 284 },
    { 179, 180, 228, 230, 284, 285 },
    { 180, 181, 229, 231, 285, 286 },
    { 181, 182, 230, 232, 286, 287 },
    { 182, 183, 231, 233, 287, 288 },
    { 183, 184, 232, 234, 288, 289 },
    { 184, 185, 233, 235, 289, 290 },
    { 185, 234, 236, 290, 291, 292 },
    { 185, 186, 235, 237, 292, 293 },
    { 186, 187, 236, 238, 293, 294 },
    { 187, 188, 237, 239, 294, 295 },
    { 188, 189, 238, 240, 295, 296 },
    { 189, 190, 239, 241, 296, 297 },
    { 190, 191, 240, 242, 297, 298 },
    { 191, 192, 241, 243, 298, 299 },
    { 192, 193, 242, 244, 299, 300 },
    { 193, 243, 245, 300, 301, 302 },
    { 193, 194, 244, 246, 302, 303 },
    { 194, 195, 245, 247, 303, 304 },
    { 195, 196, 246, 248, 304, 305 },
    { 196, 197, 247, 249, 305, 306 },
    { 197, 198, 248, 250, 306, 307 },
    { 198, 199, 249, 251, 307, 308 },
    { 199, 200, 250, 252, 308, 309 },
    { 200, 201, 251, 253, 309, 310 },
    { 201, 252, 254, 310, 311, 312 },
    { 201, 202, 253, 255, 312, 313 },
    { 202, 203, 254, 256, 313, 314 },
    { 203, 204, 255, 257, 314, 315 },
    { 204, 205, 256, 258, 315, 316 },
    { 205, 206, 257, 259, 316, 317 },
    { 206, 207, 258, 260, 317, 318 },
    { 207, 208, 259, 261, 318, 319 },
    { 208, 209, 260, 262, 319, 320 },
    { 209, 261, 263, 320, 321, 322 },
    { 209, 210, 262, 264, 322, 323 },
    { 210, 211, 263, 265, 323, 324 },
    { 211, 212, 264, 266, 324, 325 },
    { 212, 213, 265, 267, 325, 326 },
    { 213, 214, 266, 268, 326, 327 },
    { 214, 215, 267, 269, 327, 328 },
    { 215, 216, 268, 270, 328, 329 },
    { 169, 216, 217, 269, 329, 330 },
    { 217, 272, 330, 331, 332, 396 },
    { 217, 218, 271, 273, 332, 333 },
    { 218, 219, 272, 274, 333, 334 },
    { 219, 220, 273, 275, 334, 335 },
    { 220, 221, 274, 276, 335, 336 },
    { 221, 222, 275, 277, 336, 337 },
    { 222, 223, 276, 278, 337, 338 },
    { 223, 224, 277, 279, 338, 339 },
    { 224, 225, 278, 280, 339, 340 },
    { 225, 226, 279, 281, 340, 341 },
    { 226, 280, 282, 341, 342, 343 },
    { 226, 227, 281, 283, 343, 344 },
    { 227, 228, 282, 284, 344, 345 },
    { 228, 229, 283, 285, 345, 346 },
    { 229, 230, 284, 286, 346, 347 },
    { 230, 231, 285, 287, 347, 348 },
    { 231, 232, 286, 288, 348, 349 },
    { 232, 233, 287, 289, 349, 350 },
    { 233, 234, 288, 290, 350, 351 },
    { 234, 235, 289, 291, 351, 352 },
    { 235, 290, 292, 352, 353, 354 },
    { 235, 236, 291, 293, 354, 355 },
    { 236, 237, 292, 294, 355, 356 },
    { 237, 238, 293, 295, 356, 357 },
    { 238, 239, 294, 296, 357, 358 },
    { 239, 240, 295, 297, 358, 359 },
    { 240, 241, 296, 298, 359, 360 },
    { 241, 242, 297, 299, 360, 361 },
    { 242, 243, 298, 300, 361, 362 },
    { 243, 244, 299, 301, 362, 363 },
    { 244, 300, 302, 363, 364, 365 },
    { 244, 245, 301, 303, 365, 366 },
    { 245, 246, 302, 304, 366, 367 },
    { 246, 247, 303, 305, 367, 368 },
    { 247, 248, 304, 306, 368, 369 },
    { 248, 249, 305, 307, 369, 370 },
    { 249, 250, 306, 308, 370, 371 },
    { 250, 251, 307, 309, 371, 372 },
    { 251, 252, 308, 310, 372, 373 },
    { 252, 253, 309, 311, 373, 374 },
    { 253, 310, 312, 374, 375, 376 },
    { 253, 254, 311, 313, 376, 377 },
    { 254, 255, 312, 314, 377, 378 },
    { 255, 256, 313, 315, 378, 379 },
    { 256, 257, 314, 316, 379, 380 },
    { 257, 258, 315, 317, 380, 381 },
    { 258, 259, 316, 318, 381, 382 },
    { 259, 260, 317, 319, 382, 383 },
    { 260, 261, 318, 320, 383, 384 },
    { 261, 262, 319, 321, 384, 385 },
    { 262, 320, 322, 385, 386, 387 },
    { 262, 263, 321, 323, 387, 388 },
    { 263, 264, 322, 324, 388, 389 },
    { 264, 265, 323, 325, 389, 390 },
    { 265, 266, 324, 326, 390, 391 },
    { 266, 267, 325, 327, 391, 392 },
    { 267, 268, 326, 328, 392, 393 },
    { 268, 269, 327, 329, 393, 394 },
    { 269, 270, 328, 330, 394, 395 },
    { 217, 270, 271, 329, 395, 396 },
    { 271, 332, 396, 397, 398, 468 },
    { 271, 272, 331, 333, 398, 399 },
    { 272, 273, 332, 334, 399, 400 },
    { 273, 274, 333, 335, 400, 401 },
    { 274, 275, 334, 336, 401, 402 },
    { 275, 276, 335, 337, 402, 403 },
    { 276, 277, 336, 338, 403, 404 },
    { 277, 278, 337, 339, 404, 405 },
    { 278, 279, 338, 340, 405, 406 },
    { 279, 280, 339, 341, 406, 407 },
    { 280, 281, 340, 342, 407, 408 },
    { 281, 341, 343, 408, 409, 410 },
    { 281, 282, 342, 344, 410, 411 },
    { 282, 283, 343, 345, 411, 412 },
    { 283, 284, 344, 346, 412, 413 },
    { 284, 285, 345, 347, 413, 414 },
    { 285, 286, 346, 348, 414, 415 },
    { 286, 287, 347, 349, 415, 416 },
    { 287, 288, 348, 350, 416, 417 },
    { 288, 289, 349, 351, 417, 418 },
    { 289, 290, 350, 352, 418, 419 },
    { 290, 291, 351, 353, 419, 420 },
    { 291, 352, 354, 420, 421, 422 },
    { 291, 292, 353, 355, 422, 423 },
    { 292, 293, 354, 356, 423, 424 },
    { 293, 294, 355, 357, 424, 425 },
    { 294, 295, 356, 358, 425, 426 },
    { 295, 296, 357, 359, 426, 427 },
    { 296, 297, 358, 360, 427, 428 },
    { 297, 298, 359, 361, 428, 429 },
    { 298, 299, 360, 362, 429, 430 },
    { 299, 300, 361, 363, 430, 431 },
    { 300, 301, 362, 364, 431, 432 },
    { 301, 363, 365, 432, 433, 434 },
    { 301, 302, 364, 366, 434, 435 },
    { 302, 303, 365, 367, 435, 436 },
    { 303, 304, 366, 368, 436, 437 },
    { 304, 305, 367, 369, 437, 438 },
    { 305, 306, 368, 370, 438, 439 },
    { 306, 307, 369, 371, 439, 440 },
    { 307, 308, 370, 372, 440, 441 },
    { 308, 309, 371, 373, 441, 442 },
    { 309, 310, 372, 374, 442, 443 },
    { 310, 311, 373, 375, 443, 444 },
    { 311, 374, 376, 444, 445, 446 },
    { 311, 312, 375, 377, 446, 447 },
    { 312, 313, 376, 378, 447, 448 },
    { 313, 314, 377, 379, 448, 449 },
    { 314, 315, 378, 380, 449, 450 },
    { 315, 316, 379, 381, 450, 451 },
    { 316, 317, 380, 382, 451, 452 },
    { 317, 318, 381, 383, 452, 453 },
    { 318, 319, 382, 384, 453, 454 },
    { 319, 320, 383, 385, 454, 455 },
    { 320, 321, 384, 386, 455, 456 },
    { 321, 385, 387, 456, 457, 458 },
    { 321, 322, 386, 388, 458, 459 },
    { 322, 323, 387, 389, 459, 460 },
    { 323, 324, 388, 390, 460, 461 },
    { 324, 325, 389, 391, 461, 462 },
    { 325, 326, 390, 392, 462, 463 },
    { 326, 327, 391, 393, 463, 464 },
    { 327, 328, 392, 394, 464, 465 },
    { 328, 329, 393, 395, 465, 466 },
    { 329, 330, 394, 396, 466, 467 },
    { 271, 330, 331, 395, 467, 468 },
    { 331, 398, 468, 469, 470, 546 },
    { 331, 332, 397, 399, 470, 471 },
    { 332, 333, 398, 400, 471, 472 },
    { 333, 334, 399, 401, 472, 473 },
    { 334, 335, 400, 402, 473, 474 },
    { 335, 336, 401, 403, 474, 475 },
    { 336, 337, 402, 404, 475, 476 },
    { 337, 338, 403, 405, 476, 477 },
    { 338, 339, 404, 406, 477, 478 },
    { 339, 340, 405, 407, 478, 479 },
    { 340, 341, 406, 408, 479, 480 },
    { 341, 342, 407, 409, 480, 481 },
    { 342, 408, 410, 481, 482, 483 },
    { 342, 343, 409, 411, 483, 484 },
    { 343, 344, 410, 412, 484, 485 },
    { 344, 345, 411, 413, 485, 486 },
    { 345, 346, 412, 414, 486, 487 },
    { 346, 347, 413, 415, 487, 488 },
    { 347, 348, 414, 416, 488, 489 },
    { 348, 349, 415, 417, 489, 490 },
    { 349, 350, 416, 418, 490, 491 },
    { 350, 351, 417, 419, 491, 492 },
    { 351, 352, 418, 420, 492, 493 },
    { 352, 353, 419, 421, 493, 494 },
    { 353, 420, 422, 494, 495, 496 },
    { 353, 354, 421, 423, 496, 497 },
    { 354, 355, 422, 424, 497, 498 },
    { 355, 356, 423, 425, 498, 499 },
    { 356, 357, 424, 426, 499, 500 },
    { 357, 358, 425, 427, 500, 501 },
    { 358, 359, 426, 428, 501, 502 },
    { 359, 360, 427, 429, 502, 503 },
    { 360, 361, 428, 430, 503, 504 },
    { 361, 362, 429, 431, 504, 505 },
    { 362, 363, 430, 432, 505, 506 },
    { 363, 364, 431, 433, 506, 507 },
    { 364, 432, 434, 507, 508, 509 },
    { 364, 365, 433, 435, 509, 510 },
    { 365, 366, 434, 436, 510, 511 },
    { 366, 367, 435, 437, 511, 512 },
    { 367, 368, 436, 438, 512, 513 },
    { 368, 369, 437, 439, 513, 514 },
    { 369, 370, 438, 440, 514, 515 },
    { 370, 371, 439, 441, 515, 516 },
    { 371, 372, 440, 442, 516, 517 },
    { 372, 373, 441, 443, 517, 518 },
    { 373, 374, 442, 444, 518, 519 },
    { 374, 375, 443, 445, 519, 520 },
    { 375, 444, 446, 520, 521, 522 },
    { 375, 376, 445, 447, 522, 523 },
    { 376, 377, 446, 448, 523, 524 },
    { 377, 378, 447, 449, 524, 525 },
    { 378, 379, 448, 450, 525, 526 },
    { 379, 380, 449, 451, 526, 527 },
    { 380, 381, 450, 452, 527, 528 },
    { 381, 382, 451, 453, 528, 529 },
    { 382, 383, 452, 454, 529, 530 },
    { 383, 384, 453, 455, 530, 531 },
    { 384, 385, 454, 456, 531, 532 },
    { 385, 386, 455, 457, 532, 533 },
    { 386, 456, 458, 533, 534, 535 },
    { 386, 387, 457, 459, 535, 536 },
    { 387, 388, 458, 460, 536, 537 },
    { 388, 389, 459, 461, 537, 538 },
    { 389, 390, 460, 462, 538, 539 },
    { 390, 391, 461, 463, 539, 540 },
    { 391, 392, 462, 464, 540, 541 },
    { 392, 393, 463, 465, 541, 542 },
    { 393, 394, 464, 466, 542, 543 },
    { 394, 395, 465, 467, 543, 544 },
    { 395, 396, 466, 468, 544, 545 },
    { 331, 396, 397, 467, 545, 546 },
    { 397, 470, 546, 547, 548, 630 },
    { 397, 398, 469, 471, 548, 549 },
    { 398, 399, 470, 472, 549, 550 },
    { 399, 400, 471, 473, 550, 551 },
    { 400, 401, 472, 474, 551, 552 },
    { 401, 402, 473, 475, 552, 553 },
    { 402, 403, 474, 476, 553, 554 },
    { 403, 404, 475, 477, 554, 555 },
    { 404, 405, 476, 478, 555, 556 },
    { 405, 406, 477, 479, 556, 557 },
    { 406, 407, 478, 480, 557, 558 },
    { 407, 408, 479, 481, 558, 559 },
    { 408, 409, 480, 482, 559, 560 },
    { 409, 481, 483, 560, 561, 562 },
    { 409, 410, 482, 484, 562, 563 },
    { 410, 411, 483, 485, 563, 564 },
    { 411, 412, 484, 486, 564, 565 },
    { 412, 413, 485, 487, 565, 566 },
    { 413, 414, 486, 488, 566, 567 },
    { 414, 415, 487, 489, 567, 568 },
    { 415, 416, 488, 490, 568, 569 },
    { 416, 417, 489, 491, 569, 570 },
    { 417, 418, 490, 492, 570, 571 },
    { 418, 419, 491, 493, 571, 572 },
    { 419, 420, 492, 494, 572, 573 },
    { 420, 421, 493, 495, 573, 574 },
    { 421, 494, 496, 574, 575, 576 },
    { 421, 422, 495, 497, 576, 577 },
    { 422, 423, 496, 498, 577, 578 },
    { 423, 424, 497, 499, 578, 579 },
    { 424, 425, 498, 500, 579, 580 },
    { 425, 426, 499, 501, 580, 581 },
    { 426, 427, 500, 502, 581, 582 },
    { 427, 428, 501, 503, 582, 583 },
    { 428, 429, 502, 504, 583, 584 },
    { 429, 430, 503, 505, 584, 585 },
    { 430, 431, 504, 506, 585, 586 },
    { 431, 432, 505, 507, 586, 587 },
    { 432, 433, 506, 508, 587, 588 },
    { 433, 507, 509, 588, 589, 590 },
    { 433, 434, 508, 510, 590, 591 },
    { 434, 435, 509, 511, 591, 592 },
    { 435, 436, 510, 512, 592, 593 },
    { 436, 437, 511, 513, 593, 594 },
    { 437, 438, 512, 514, 594, 595 },
    { 438, 439, 513, 515, 595, 596 },
    { 439, 440, 514, 516, 596, 597 },
    { 440, 441, 515, 517, 597, 598 },
    { 441, 442, 516, 518, 598, 599 },
    { 442, 443, 517, 519, 599, 600 },
    { 443, 444, 518, 520, 600, 601 },
    { 444, 445, 519, 521, 601, 602 },
    { 445, 520, 522, 602, 603, 604 },
    { 445, 446, 521, 523, 604, 605 },
    { 446, 447, 522, 524, 605, 606 },
    { 447, 448, 523, 525, 606, 607 },
    { 448, 449, 524, 526, 607, 608 },
    { 449, 450, 525, 527, 608, 609 },
    { 450, 451, 526, 528, 609, 610 },
    { 451, 452, 527, 529, 610, 611 },
    { 452, 453, 528, 530, 611, 612 },
    { 453, 454, 529, 531, 612, 613 },
    { 454, 455, 530, 532, 613, 614 },
    { 455, 456, 531, 533, 614, 615 },
    { 456, 457, 532, 534, 615, 616 },
    { 457, 533, 535, 616, 617, 618 },
    { 457, 458, 534, 536, 618, 619 },
    { 458, 459, 535, 537, 619, 620 },
    { 459, 460, 536, 538, 620, 621 },
    { 460, 461, 537, 539, 621, 622 },
    { 461, 462, 538, 540, 622, 623 },
    { 462, 463, 539, 541, 623, 624 },
    { 463, 464, 540, 542, 624, 625 },
    { 464, 465, 541, 543, 625, 626 },
    { 465, 466, 542, 544, 626, 627 },
    { 466, 467, 543, 545, 627, 628 },
    { 467, 468, 544, 546, 628, 629 },
    { 397, 468, 469, 545, 629, 630 },
    { 469, 548, 630, 631, 632, 720 },
    { 469, 470, 547, 549, 632, 633 },
    { 470, 471, 548, 550, 633, 634 },
    { 471, 472, 549, 551, 634, 635 },
    { 472, 473, 550, 552, 635, 636 },
    { 473, 474, 551, 553, 636, 637 },
    { 474, 475, 552, 554, 637, 638 },
    { 475, 476, 553, 555, 638, 639 },
    { 476, 477, 554, 556, 639, 640 },
    { 477, 478, 555, 557, 640, 641 },
    { 478, 479, 556, 558, 641, 642 },
    { 479, 480, 557, 559, 642, 643 },
    { 480, 481, 558, 560, 643, 644 },
    { 481, 482, 559, 561, 644, 645 },
    { 482, 560, 562, 645, 646, 647 },
    { 482, 483, 561, 563, 647, 648 },
    { 483, 484, 562, 564, 648, 649 },
    { 484, 485, 563, 565, 649, 650 },
    { 485, 486, 564, 566, 650, 651 },
    { 486, 487, 565, 567, 651, 652 },
    { 487, 488, 566, 568, 652, 653 },
    { 488, 489, 567, 569, 653, 654 },
    { 489, 490, 568, 570, 654, 655 },
    { 490, 491, 569, 571, 655, 656 },
    { 491, 492, 570, 572, 656, 657 },
    { 492, 493, 571, 573, 657, 658 },
    { 493, 494, 572, 574, 658, 659 },
    { 494, 495, 573, 575, 659, 660 },
    { 495, 574, 576, 660, 661, 662 },
    { 495, 496, 575, 577, 662, 663 },
    { 496, 497, 576, 578, 663, 664 },
    { 497, 498, 577, 579, 664, 665 },
    { 498, 499, 578, 580, 665, 666 },
    { 499, 500, 579, 581, 666, 667 },
    { 500, 501, 580, 582, 667, 668 },
    { 501, 502, 581, 583, 668, 669 },
    { 502, 503, 582, 584, 669, 670 },
    { 503, 504, 583, 585, 670, 671 },
    { 504, 505, 584, 586, 671, 672 },
    { 505, 506, 585, 587, 672, 673 },
    { 506, 507, 586, 588, 673, 674 },
    { 507, 508, 587, 589, 674, 675 },
    { 508, 588, 590, 675, 676, 677 },
    { 508, 509, 589, 591, 677, 678 },
    { 509, 510, 590, 592, 678, 679 },
    { 510, 511, 591, 593, 679, 680 },
    { 511, 512, 592, 594, 680, 681 },
    { 512, 513, 593, 595, 681, 682 },
    { 513, 514, 594, 596, 682, 683 },
    { 514, 515, 595, 597, 683, 684 },
    { 515, 516, 596, 598, 684, 685 },
    { 516, 517, 597, 599, 685, 686 },
    { 517, 518, 598, 600, 686, 687 },
    { 518, 519, 599, 601, 687, 688 },
    { 519, 520, 600, 602, 688, 689 },
    { 520, 521, 601, 603, 689, 690 },
    { 521, 602, 604, 690, 691, 692 },
    { 521, 522, 603, 605, 692, 693 },
    { 522, 523, 604, 606, 693, 694 },
    { 523, 524, 605, 607, 694, 695 },
    { 524, 525, 606, 608, 695, 696 },
    { 525, 526, 607, 609, 696, 697 },
    { 526, 527, 608, 610, 697, 698 },
    { 527, 528, 609, 611, 698, 699 },
    { 528, 529, 610, 612, 699, 700 },
    { 529, 530, 611, 613, 700, 701 },
    { 530, 531, 612, 614, 701, 702 },
    { 531, 532, 613, 615, 702, 703 },
    { 532, 533, 614, 616, 703, 704 },
    { 533, 534, 615, 617, 704, 705 },
    { 534, 616, 618, 705, 706, 707 },
    { 534, 535, 617, 619, 707, 708 },
    { 535, 536, 618, 620, 708, 709 },
    { 536, 537, 619, 621, 709, 710 },
    { 537, 538, 620, 622, 710, 711 },
    { 538, 539, 621, 623, 711, 712 },
    { 539, 540, 622, 624, 712, 713 },
    { 540, 541, 623, 625, 713, 714 },
    { 541, 542, 624, 626, 714, 715 },
    { 542, 543, 625, 627, 715, 716 },
    { 543, 544, 626, 628, 716, 717 },
    { 544, 545, 627, 629, 717, 718 },
    { 545, 546, 628, 630, 718, 719 },
    { 469, 546, 547, 629, 719, 720 },
    { 547, 632, 720, 721, 722, 816 },
    { 547, 548, 631, 633, 722, 723 },
    { 548, 549, 632, 634, 723, 724 },
    { 549, 550, 633, 635, 724, 725 },
    { 550, 551, 634, 636, 725, 726 },
    { 551, 552, 635, 637, 726, 727 },
    { 552, 553, 636, 638, 727, 728 },
    { 553, 554, 637, 639, 728, 729 },
    { 554, 555, 638, 640, 729, 730 },
    { 555, 556, 639, 641, 730, 731 },
    { 556, 557, 640, 642, 731, 732 },
    { 557, 558, 641, 643, 732, 733 },
    { 558, 559, 642, 644, 733, 734 },
    { 559, 560, 643, 645, 734, 735 },
    { 560, 561, 644, 646, 735, 736 },
    { 561, 645, 647, 736, 737, 738 },
    { 561, 562, 646, 648, 738, 739 },
    { 562, 563, 647, 649, 739, 740 },
    { 563, 564, 648, 650, 740, 741 },
    { 564, 565, 649, 651, 741, 742 },
    { 565, 566, 650, 652, 742, 743 },
    { 566, 567, 651, 653, 743, 744 },
    { 567, 568, 652, 654, 744, 745 },
    { 568, 569, 653, 655, 745, 746 },
    { 569, 570, 654, 656, 746, 747 },
    { 570, 571, 655, 657, 747, 748 },
    { 571, 572, 656, 658, 748, 749 },
    { 572, 573, 657, 659, 749, 750 },
    { 573, 574, 658, 660, 750, 751 },
    { 574, 575, 659, 661, 751, 752 },
    { 575, 660, 662, 752, 753, 754 },
    { 575, 576, 661, 663, 754, 755 },
    { 576, 577, 662, 664, 755, 756 },
    { 577, 578, 663, 665, 756, 757 },
    { 578, 579, 664, 666, 757, 758 },
    { 579, 580, 665, 667, 758, 759 },
    { 580, 581, 666, 668, 759, 760 },
    { 581, 582, 667, 669, 760, 761 },
    { 582, 583, 668, 670, 761, 762 },
    { 583, 584, 669, 671, 762, 763 },
    { 584, 585, 670, 672, 763, 764 },
    { 585, 586, 671, 673, 764, 765 },
    { 586, 587, 672, 674, 765, 766 },
    { 587, 588, 673, 675, 766, 767 },
    { 588, 589, 674, 676, 767, 768 },
    { 589, 675, 677, 768, 769, 770 },
    { 589, 590, 676, 678, 770, 771 },
    { 590, 591, 677, 679, 771, 772 },
    { 591, 592, 678, 680, 772, 773 },
    { 592, 593, 679, 681, 773, 774 },
    { 593, 594, 680, 682, 774, 775 },
    { 594, 595, 681, 683, 775, 776 },
    { 595, 596, 682, 684, 776, 777 },
    { 596, 597, 683, 685, 777, 778 },
    { 597, 598, 684, 686, 778, 779 },
    { 598, 599, 685, 687, 779, 780 },
    { 599, 600, 686, 688, 780, 781 },
    { 600, 601, 687, 689, 781, 782 },
    { 601, 602, 688, 690, 782, 783 },
    { 602, 603, 689, 691, 783, 784 },
    { 603, 690, 692, 784, 785, 786 },
    { 603, 604, 691, 693, 786, 787 },
    { 604, 605, 692, 694, 787, 788 },
    { 605, 606, 693, 695, 788, 789 },
    { 606, 607, 694, 696, 789, 790 },
    { 607, 608, 695, 697, 790, 791 },
    { 608, 609, 696, 698, 791, 792 },
    { 609, 610, 697, 699, 792, 793 },
    { 610, 611, 698, 700, 793, 794 },
    { 611, 612, 699, 701, 794, 795 },
    { 612, 613, 700, 702, 795, 796 },
    { 613, 614, 701, 703, 796, 797 },
    { 614, 615, 702, 704, 797, 798 },
    { 615, 616, 703, 705, 798, 799 },
    { 616, 617, 704, 706, 799, 800 },
    { 617, 705, 707, 800, 801, 802 },
    { 617, 618, 706, 708, 802, 803 },
    { 618, 619, 707, 709, 803, 804 },
    { 619, 620, 708, 710, 804, 805 },
    { 620, 621, 709, 711, 805, 806 },
    { 621, 622, 710, 712, 806, 807 },
    { 622, 623, 711, 713, 807, 808 },
    { 623, 624, 712, 714, 808, 809 },
    { 624, 625, 713, 715, 809, 810 },
    { 625, 626, 714, 716, 810, 811 },
    { 626, 627, 715, 717, 811, 812 },
    { 627, 628, 716, 718, 812, 813 },
    { 628, 629, 717, 719, 813, 814 },
    { 629, 630, 718, 720, 814, 815 },
    { 547, 630, 631, 719, 815, 816 },
    { 631, 722, 816, 817, 818, 918 },
    { 631, 632, 721, 723, 818, 819 },
    { 632, 633, 722, 724, 819, 820 },
    { 633, 634, 723, 725, 820, 821 },
    { 634, 635, 724, 726, 821, 822 },
    { 635, 636, 725, 727, 822, 823 },
    { 636, 637, 726, 728, 823, 824 },
    { 637, 638, 727, 729, 824, 825 },
    { 638, 639, 728, 730, 825, 826 },
    { 639, 640, 729, 731, 826, 827 },
    { 640, 641, 730, 732, 827, 828 },
    { 641, 642, 731, 733, 828, 829 },
    { 642, 643, 732, 734, 829, 830 },
    { 643, 644, 733, 735, 830, 831 },
    { 644, 645, 734, 736, 831, 832 },
    { 645, 646, 735, 737, 832, 833 },
    { 646, 736, 738, 833, 834, 835 },
    { 646, 647, 737, 739, 835, 836 },
    { 647, 648, 738, 740, 836, 837 },
    { 648, 649, 739, 741, 837, 838 },
    { 649, 650, 740, 742, 838, 839 },
    { 650, 651, 741, 743, 839, 840 },
    { 651, 652, 742, 744, 840, 841 },
    { 652, 653, 743, 745, 841, 842 },
    { 653, 654, 744, 746, 842, 843 },
    { 654, 655, 745, 747, 843, 844 },
    { 655, 656, 746, 748, 844, 845 },
    { 656, 657, 747, 749, 845, 846 },
    { 657, 658, 748, 750, 846, 847 },
    { 658, 659, 749, 751, 847, 848 },
    { 659, 660, 750, 752, 848, 849 },
    { 660, 661, 751, 753, 849, 850 },
    { 661, 752, 754, 850, 851, 852 },
    { 661, 662, 753, 755, 852, 853 },
    { 662, 663, 754, 756, 853, 854 },
    { 663, 664, 755, 757, 854, 855 },
    { 664, 665, 756, 758, 855, 856 },
    { 665, 666, 757, 759, 856, 857 },
    { 666, 667, 758, 760, 857, 858 },
    { 667, 668, 759, 761, 858, 859 },
    { 668, 669, 760, 762, 859, 860 },
    { 669, 670, 761, 763, 860, 861 },
    { 670, 671, 762, 764, 861, 862 },
    { 671, 672, 763, 765, 862, 863 },
    { 672, 673, 764, 766, 863, 864 },
    { 673, 674, 765, 767, 864, 865 },
    { 674, 675, 766, 768, 865, 866 },
    { 675, 676, 767, 769, 866, 867 },
    { 676, 768, 770, 867, 868, 869 },
    { 676, 677, 769, 771, 869, 870 },
    { 677, 678, 770, 772, 870, 871 },
    { 678, 679, 771, 773, 871, 872 },
    { 679, 680, 772, 774, 872, 873 },
    { 680, 681, 773, 775, 873, 874 },
    { 681, 682, 774, 776, 874, 875 },
    { 682, 683, 775, 777, 875, 876 },
    { 683, 684, 776, 778, 876, 877 },
    { 684, 685, 777, 779, 877, 878 },
    { 685, 686, 778, 780, 878, 879 },
    { 686, 687, 779, 781, 879, 880 },
    { 687, 688, 780, 782, 880, 881 },
    { 688, 689, 781, 783, 881, 882 },
    { 689, 690, 782, 784, 882, 883 },
    { 690, 691, 783, 785, 883, 884 },
    { 691, 784, 786, 884, 885, 886 },
    { 691, 692, 785, 787, 886, 887 },
    { 692, 693, 786, 788, 887, 888 },
    { 693, 694, 787, 789, 888, 889 },
    { 694, 695, 788, 790, 889, 890 },
    { 695, 696, 789, 791, 890, 891 },
    { 696, 697, 790, 792, 891, 892 },
    { 697, 698, 791, 793, 892, 893 },
    { 698, 699, 792, 794, 893, 894 },
    { 699, 700, 793, 795, 894, 895 },
    { 700, 701, 794, 796, 895, 896 },
    { 701, 702, 795, 797, 896, 897 },
    { 702, 703, 796, 798, 897, 898 },
    { 703, 704, 797, 799, 898, 899 },
    { 704, 705, 798, 800, 899, 900 },
    { 705, 706, 799, 801, 900, 901 },
    { 706, 800, 802, 901, 902, 903 },
    { 706, 707, 801, 803, 903, 904 },
    { 707, 708, 802, 804, 904, 905 },
    { 708, 709, 803, 805, 905, 906 },
    { 709, 710, 804, 806, 906, 907 },
    { 710, 711, 805, 807, 907, 908 },
    { 711, 712, 806, 808, 908, 909 },
    { 712, 713, 807, 809, 909, 910 },
    { 713, 714, 808, 810, 910, 911 },
    { 714, 715, 809, 811, 911, 912 },
    { 715, 716, 810, 812, 912, 913 },
    { 716, 717, 811, 813, 913, 914 },
    { 717, 718, 812, 814, 914, 915 },
    { 718, 719, 813, 815, 915, 916 },
    { 719, 720, 814, 816, 916, 917 },
    { 631, 720, 721, 815, 917, 918 },
    { 721, 818, 918, 1014, -1, -1 },
    { 721, 722, 817, 819, 919, -1 },
    { 722, 723, 818, 820, 919, 920 },
    { 723, 724, 819, 821, 920, 921 },
    { 724, 725, 820, 822, 921, 922 },
    { 725, 726, 821, 823, 922, 923 },
    { 726, 727, 822, 824, 923, 924 },
    { 727, 728, 823, 825, 924, 925 },
    { 728, 729, 824, 826, 925, 926 },
    { 729, 730, 825, 827, 926, 927 },
    { 730, 731, 826, 828, 927, 928 },
    { 731, 732, 827, 829, 928, 929 },
    { 732, 733, 828, 830, 929, 930 },
    { 733, 734, 829, 831, 930, 931 },
    { 734, 735, 830, 832, 931, 932 },
    { 735, 736, 831, 833, 932, 933 },
    { 736, 737, 832, 834, 933, 934 },
    { 737, 833, 835, 934, -1, -1 },
    { 737, 738, 834, 836, 935, -1 },
    { 738, 739, 835, 837, 935, 936 },
    { 739, 740, 836, 838, 936, 937 },
    { 740, 741, 837, 839, 937, 938 },
    { 741, 742, 838, 840, 938, 939 },
    { 742, 743, 839, 841, 939, 940 },
    { 743, 744, 840, 842, 940, 941 },
    { 744, 745, 841, 843, 941, 942 },
    { 745, 746, 842, 844, 942, 943 },
    { 746, 747, 843, 845, 943, 944 },
    { 747, 748, 844, 846, 944, 945 },
    { 748, 749, 845, 847, 945, 946 },
    { 749, 750, 846, 848, 946, 947 },
    { 750, 751, 847, 849, 947, 948 },
    { 751, 752, 848, 850, 948, 949 },
    { 752, 753, 849, 851, 949, 950 },
    { 753, 850, 852, 950, -1, -1 },
    { 753, 754, 851, 853, 951, -1 },
    { 754, 755, 852, 854, 951, 952 },
    { 755, 756, 853, 855, 952, 953 },
    { 756, 757, 854, 856, 953, 954 },
    { 757, 758, 855, 857, 954, 955 },
    { 758, 759, 856, 858, 955, 956 },
    { 759, 760, 857, 859, 956, 957 },
    { 760, 761, 858, 860, 957, 958 },
    { 761, 762, 859, 861, 958, 959 },
    { 762, 763, 860, 862, 959, 960 },
    { 763, 764, 861, 863, 960, 961 },
    { 764, 765, 862, 864, 961, 962 },
    { 765, 766, 863, 865, 962, 963 },
    { 766, 767, 864, 866, 963, 964 },
    { 767, 768, 865, 867, 964, 965 },
    { 768, 769, 866, 868, 965, 966 },
    { 769, 867, 869, 966, -1, -1 },
    { 769, 770, 868, 870, 967, -1 },
    { 770, 771, 869, 871, 967, 968 },
    { 771, 772, 870, 872, 968, 969 },
    { 772, 773, 871, 873, 969, 970 },
    { 773, 774, 872, 874, 970, 971 },
    { 774, 775, 873, 875, 971, 972 },
    { 775, 776, 874, 876, 972, 973 },
    { 776, 777, 875, 877, 973, 974 },
    { 777, 778, 876, 878, 974, 975 },
    { 778, 779, 877, 879, 975, 976 },
    { 779, 780, 878, 880, 976, 977 },
    { 780, 781, 879, 881, 977, 978 },
    { 781, 782, 880, 882, 978, 979 },
    { 782, 783, 881, 883, 979, 980 },
    { 783, 784, 882, 884, 980, 981 },
    { 784, 785, 883, 885, 981, 982 },
    { 785, 884, 886, 982, -1, -1 },
    { 785, 786, 885, 887, 983, -1 },
    { 786, 787, 886, 888, 983, 984 },
    { 787, 788, 887, 889, 984, 985 },
    { 788, 789, 888, 890, 985, 986 },
    { 789, 790, 889, 891, 986, 987 },
    { 790, 791, 890, 892, 987, 988 },
    { 791, 792, 891, 893, 988, 989 },
    { 792, 793, 892, 894, 989, 990 },
    { 793, 794, 893, 895, 990, 991 },
    { 794, 795, 894, 896, 991, 992 },
    { 795, 796, 895, 897, 992, 993 },
    { 796, 797, 896, 898, 993, 994 },
    { 797, 798, 897, 899, 994, 995 },
    { 798, 799, 898, 900, 995, 996 },
    { 799, 800, 899, 901, 996, 997 },
    { 800, 801, 900, 902, 997, 998 },
    { 801, 901, 903, 998, -1, -1 },
    { 801, 802, 902, 904, 999, -1 },
    { 802, 803, 903, 905, 999, 1000 },
    { 803, 804, 904, 906, 1000, 1001 },
    { 804, 805, 905, 907, 1001, 1002 },
    { 805, 806, 906, 908, 1002, 1003 },
    { 806, 807, 907, 909, 1003, 1004 },
    { 807, 808, 908, 910, 1004, 1005 },
    { 808, 809, 909, 911, 1005, 1006 },
    { 809, 810, 910, 912, 1006, 1007 },
    { 810, 811, 911, 913, 1007, 1008 },
    { 811, 812, 912, 914, 1008, 1009 },
    { 812, 813, 913, 915, 1009, 1010 },
    { 813, 814, 914, 916, 1010, 1011 },
    { 814, 815, 915, 917, 1011, 1012 },
    { 815, 816, 916, 918, 1012, 1013 },
    { 721, 816, 817, 917, 1013, 1014 },
    { 818, 819, 920, -1, -1, -1 },
    { 819, 820, 919, 921, -1, -1 },
    { 820, 821, 920, 922, 1015, -1 },
    { 821, 822, 921, 923, 1015, 1016 },
    { 822, 823, 922, 924, 1016, 1017 },
    { 823, 824, 923, 925, 1017, 1018 },
    { 824, 825, 924, 926, 1018, 1019 },
    { 825, 826, 925, 927, 1019, 1020 },
    { 826, 827, 926, 928, 1020, 1021 },
    { 827, 828, 927, 929, 1021, 1022 },
    { 828, 829, 928, 930, 1022, 1023 },
    { 829, 830, 929, 931, 1023, 1024 },
    { 830, 831, 930, 932, 1024, 1025 },
    { 831, 832, 931, 933, 1025, 1026 },
    { 832, 833, 932, 934, 1026, -1 },
    { 833, 834, 933, -1, -1, -1 },
    { 835, 836, 936, -1, -1, -1 },
    { 836, 837, 935, 937, -1, -1 },
    { 837, 838, 936, 938, 1027, -1 },
    { 838, 839, 937, 939, 1027, 1028 },
    { 839, 840, 938, 940, 1028, 1029 },
    { 840, 841, 939, 941, 1029, 1030 },
    { 841, 842, 940, 942, 1030, 1031 },
    { 842, 843, 941, 943, 1031, 1032 },
    { 843, 844, 942, 944, 1032, 1033 },
    { 844, 845, 943, 945, 1033, 1034 },
    { 845, 846, 944, 946, 1034, 1035 },
    { 846, 847, 945, 947, 1035, 1036 },
    { 847, 848, 946, 948, 1036, 1037 },
    { 848, 849, 947, 949, 1037, 1038 },
    { 849, 850, 948, 950, 1038, -1 },
    { 850, 851, 949, -1, -1, -1 },
    { 852, 853, 952, -1, -1, -1 },
    { 853, 854, 951, 953, -1, -1 },
    { 854, 855, 952, 954, 1039, -1 },
    { 855, 856, 953, 955, 1039, 1040 },
    { 856, 857, 954, 956, 1040, 1041 },
    { 857, 858, 955, 957, 1041, 1042 },
    { 858, 859, 956, 958, 1042, 1043 },
    { 859, 860, 957, 959, 1043, 1044 },
    { 860, 861, 958, 960, 1044, 1045 },
    { 861, 862, 959, 961, 1045, 1046 },
    { 862, 863, 960, 962, 1046, 1047 },
    { 863, 864, 961, 963, 1047, 1048 },
    { 864, 865, 962, 964, 1048, 1049 },
    { 865, 866, 963, 965, 1049, 1050 },
    { 866, 867, 964, 966, 1050, -1 },
    { 867, 868, 965, -1, -1, -1 },
    { 869, 870, 968, -1, -1, -1 },
    { 870, 871, 967, 969, -1, -1 },
    { 871, 872, 968, 970, 1051, -1 },
    { 872, 873, 969, 971, 1051, 1052 },
    { 873, 874, 970, 972, 1052, 1053 },
    { 874, 875, 971, 973, 1053, 1054 },
    { 875, 876, 972, 974, 1054, 1055 },
    { 876, 877, 973, 975, 1055, 1056 },
    { 877, 878, 974, 976, 1056, 1057 },
    { 878, 879, 975, 977, 1057, 1058 },
    { 879, 880, 976, 978, 1058, 1059 },
    { 880, 881, 977, 979, 1059, 1060 },
    { 881, 882, 978, 980, 1060, 1061 },
    { 882, 883, 979, 981, 1061, 1062 },
    { 883, 884, 980, 982, 1062, -1 },
    { 884, 885, 981, -1, -1, -1 },
    { 886, 887, 984, -1, -1, -1 },
    { 887, 888, 983, 985, -1, -1 },
    { 888, 889, 984, 986, 1063, -1 },
    { 889, 890, 985, 987, 1063, 1064 },
    { 890, 891, 986, 988, 1064, 1065 },
    { 891, 892, 987, 989, 1065, 1066 },
    { 892, 893, 988, 990, 1066, 1067 },
    { 893, 894, 989, 991, 1067, 1068 },
    { 894, 895, 990, 992, 1068, 1069 },
    { 895, 896, 991, 993, 1069, 1070 },
    { 896, 897, 992, 994, 1070, 1071 },
    { 897, 898, 993, 995, 1071, 1072 },
    { 898, 899, 994, 996, 1072, 1073 },
    { 899, 900, 995, 997, 1073, 1074 },
    { 900, 901, 996, 998, 1074, -1 },
    { 901, 902, 997, -1, -1, -1 },
    { 903, 904, 1000, -1, -1, -1 },
    { 904, 905, 999, 1001, -1, -1 },
    { 905, 906, 1000, 1002, 1075, -1 },
    { 906, 907, 1001, 1003, 1075, 1076 },
    { 907, 908, 1002, 1004, 1076, 1077 },
    { 908, 909, 1003, 1005, 1077, 1078 },
    { 909, 910, 1004, 1006, 1078, 1079 },
    { 910, 911, 1005, 1007, 1079, 1080 },
    { 911, 912, 1006, 1008, 1080, 1081 },
    { 912, 913, 1007, 1009, 1081, 1082 },
    { 913, 914, 1008, 1010, 1082, 1083 },
    { 914, 915, 1009, 1011, 1083, 1084 },
    { 915, 916, 1010, 1012, 1084, 1085 },
    { 916, 917, 1011, 1013, 1085, 1086 },
    { 917, 918, 1012, 1014, 1086, -1 },
    { 817, 918, 1013, -1, -1, -1 },
    { 921, 922, 1016, -1, -1, -1 },
    { 922, 923, 1015, 1017, -1, -1 },
    { 923, 924, 1016, 1018, 1087, -1 },
    { 924, 925, 1017, 1019, 1087, 1088 },
    { 925, 926, 1018, 1020, 1088, 1089 },
    { 926, 927, 1019, 1021, 1089, 1090 },
    { 927, 928, 1020, 1022, 1090, 1091 },
    { 928, 929, 1021, 1023, 1091, 1092 },
    { 929, 930, 1022, 1024, 1092, 1093 },
    { 930, 931, 1023, 1025, 1093, 1094 },
    { 931, 932, 1024, 1026, 1094, 1095 },
    { 932, 933, 1025, 1095, -1, -1 },
    { 937, 938, 1028, -1, -1, -1 },
    { 938, 939, 1027, 1029, -1, -1 },
    { 939, 940, 1028, 1030, 1096, -1 },
    { 940, 941, 1029, 1031, 1096, 1097 },
    { 941, 942, 1030, 1032, 1097, 1098 },
    { 942, 943, 1031, 1033, 1098, 1099 },
    { 943, 944, 1032, 1034, 1099, 1100 },
    { 944, 945, 1033, 1035, 1100, 1101 },
    { 945, 946, 1034, 1036, 1101, 1102 },
    { 946, 947, 1035, 1037, 1102, 1103 },
    { 947, 948, 1036, 1038, 1103, 1104 },
    { 948, 949, 1037, 1104, -1, -1 },
    { 953, 954, 1040, -1, -1, -1 },
    { 954, 955, 1039, 1041, -1, -1 },
    { 955, 956, 1040, 1042, 1105, -1 },
    { 956, 957, 1041, 1043, 1105, 1106 },
    { 957, 958, 1042, 1044, 1106, 1107 },
    { 958, 959, 1043, 1045, 1107, 1108 },
    { 959, 960, 1044, 1046, 1108, 1109 },
    { 960, 961, 1045, 1047, 1109, 1110 },
    { 961, 962, 1046, 1048, 1110, 1111 },
    { 962, 963, 1047, 1049, 1111, 1112 },
    { 963, 964, 1048, 1050, 1112, 1113 },
    { 964, 965, 1049, 1113, -1, -1 },
    { 969, 970, 1052, -1, -1, -1 },
    { 970, 971, 1051, 1053, -1, -1 },
    { 971, 972, 1052, 1054, 1114, -1 },
    { 972, 973, 1053, 1055, 1114, 1115 },
    { 973, 974, 1054, 1056, 1115, 1116 },
    { 974, 975, 1055, 1057, 1116, 1117 },
    { 975, 976, 1056, 1058, 1117, 1118 },
    { 976, 977, 1057, 1059, 1118, 1119 },
    { 977, 978, 1058, 1060, 1119, 1120 },
    { 978, 979, 1059, 1061, 1120, 1121 },
    { 979, 980, 1060, 1062, 1121, 1122 },
    { 980, 981, 1061, 1122, -1, -1 },
    { 985, 986, 1064, -1, -1, -1 },
    { 986, 987, 1063, 1065, -1, -1 },
    { 987, 988, 1064, 1066, 1123, -1 },
    { 988, 989, 1065, 1067, 1123, 1124 },
    { 989, 990, 1066, 1068, 1124, 1125 },
    { 990, 991, 1067, 1069, 1125, 1126 },
    { 991, 992, 1068, 1070, 1126, 1127 },
    { 992, 993, 1069, 1071, 1127, 1128 },
    { 993, 994, 1070, 1072, 1128, 1129 },
    { 994, 995, 1071, 1073, 1129, 1130 },
    { 995, 996, 1072, 1074, 1130, 1131 },
    { 996, 997, 1073, 1131, -1, -1 },
    { 1001, 1002, 1076, -1, -1, -1 },
    { 1002, 1003, 1075, 1077, -1, -1 },
    { 1003, 1004, 1076, 1078, 1132, -1 },
    { 1004, 1005, 1077, 1079, 1132, 1133 },
    { 1005, 1006, 1078, 1080, 1133, 1134 },
    { 1006, 1007, 1079, 1081, 1134, 1135 },
    { 1007, 1008, 1080, 1082, 1135, 1136 },
    { 1008, 1009, 1081, 1083, 1136, 1137 },
    { 1009, 1010, 1082, 1084, 1137, 1138 },
    { 1010, 1011, 1083, 1085, 1138, 1139 },
    { 1011, 1012, 1084, 1086, 1139, 1140 },
    { 1012, 1013, 1085, 1140, -1, -1 },
    { 1017, 1018, 1088, -1, -1, -1 },
    { 1018, 1019, 1087, 1089, -1, -1 },
    { 1019, 1020, 1088, 1090, 1141, -1 },
    { 1020, 1021, 1089, 1091, 1141, 1142 },
    { 1021, 1022, 1090, 1092, 1142, 1143 },
    { 1022, 1023, 1091, 1093, 1143, 1144 },
    { 1023, 1024, 1092, 1094, 1144, 1145 },
    { 1024, 1025, 1093, 1095, 1145, -1 },
    { 1025, 1026, 1094, -1, -1, -1 },
    { 1029, 1030, 1097, -1, -1, -1 },
    { 1030, 1031, 1096, 1098, -1, -1 },
    { 1031, 1032, 1097, 1099, 1146, -1 },
    { 1032, 1033, 1098, 1100, 1146, 1147 },
    { 1033, 1034, 1099, 1101, 1147, 1148 },
    { 1034, 1035, 1100, 1102, 1148, 1149 },
    { 1035, 1036, 1101, 1103, 1149, 1150 },
    { 1036, 1037, 1102, 1104, 1150, -1 },
    { 1037, 1038, 1103, -1, -1, -1 },
    { 1041, 1042, 1106, -1, -1, -1 },
    { 1042, 1043, 1105, 1107, -1, -1 },
    { 1043, 1044, 1106, 1108, 1151, -1 },
    { 1044, 1045, 1107, 1109, 1151, 1152 },
    { 1045, 1046, 1108, 1110, 1152, 1153 },
    { 1046, 1047, 1109, 1111, 1153, 1154 },
    { 1047, 1048, 1110, 1112, 1154, 1155 },
    { 1048, 1049, 1111, 1113, 1155, -1 },
    { 1049, 1050, 1112, -1, -1, -1 },
    { 1053, 1054, 1115, -1, -1, -1 },
    { 1054, 1055, 1114, 1116, -1, -1 },
    { 1055, 1056, 1115, 1117, 1156, -1 },
    { 1056, 1057, 1116, 1118, 1156, 1157 },
    { 1057, 1058, 1117, 1119, 1157, 1158 },
    { 1058, 1059, 1118, 1120, 1158, 1159 },
    { 1059, 1060, 1119, 1121, 1159, 1160 },
    { 1060, 1061, 1120, 1122, 1160, -1 },
    { 1061, 1062, 1121, -1, -1, -1 },
    { 1065, 1066, 1124, -1, -1, -1 },
    { 1066, 1067, 1123, 1125, -1, -1 },
    { 1067, 1068, 1124, 1126, 1161, -1 },
    { 1068, 1069, 1125, 1127, 1161, 1162 },
    { 1069, 1070, 1126, 1128, 1162, 1163 },
    { 1070, 1071, 1127, 1129, 1163, 1164 },
    { 1071, 1072, 1128, 1130, 1164, 1165 },
    { 1072, 1073, 1129, 1131, 1165, -1 },
    { 1073, 1074, 1130, -1, -1, -1 },
    { 1077, 1078, 1133, -1, -1, -1 },
    { 1078, 1079, 1132, 1134, -1, -1 },
    { 1079, 1080, 1133, 1135, 1166, -1 },
    { 1080, 1081, 1134, 1136, 1166, 1167 },
    { 1081, 1082, 1135, 1137, 1167, 1168 },
    { 1082, 1083, 1136, 1138, 1168, 1169 },
    { 1083, 1084, 1137, 1139, 1169, 1170 },
    { 1084, 1085, 1138, 1140, 1170, -1 },
    { 1085, 1086, 1139, -1, -1, -1 },
    { 1089, 1090, 1142, -1, -1, -1 },
    { 1090, 1091, 1141, 1143, -1, -1 },
    { 1091, 1092, 1142, 1144, 1171, -1 },
    { 1092, 1093, 1143, 1145, 1171, 1172 },
    { 1093, 1094, 1144, 1172, -1, -1 },
    { 1098, 1099, 1147, -1, -1, -1 },
    { 1099, 1100, 1146, 1148, -1, -1 },
    { 1100, 1101, 1147, 1149, 1173, -1 },
    { 1101, 1102, 1148, 1150, 1173, 1174 },
    { 1102, 1103, 1149, 1174, -1, -1 },
    { 1107, 1108, 1152, -1, -1, -1 },
    { 1108, 1109, 1151, 1153, -1, -1 },
    { 1109, 1110, 1152, 1154, 1175, -1 },
    { 1110, 1111, 1153, 1155, 1175, 1176 },
    { 1111, 1112, 1154, 1176, -1, -1 },
    { 1116, 1117, 1157, -1, -1, -1 },
    { 1117, 1118, 1156, 1158, -1, -1 },
    { 1118, 1119, 1157, 1159, 1177, -1 },
    { 1119, 1120, 1158, 1160, 1177, 1178 },
    { 1120, 1121, 1159, 1178, -1, -1 },
    { 1125, 1126, 1162, -1, -1, -1 },
    { 1126, 1127, 1161, 1163, -1, -1 },
    { 1127, 1128, 1162, 1164, 1179, -1 },
    { 1128, 1129, 1163, 1165, 1179, 1180 },
    { 1129, 1130, 1164, 1180, -1, -1 },
    { 1134, 1135, 1167, -1, -1, -1 },
    { 1135, 1136, 1166, 1168, -1, -1 },
    { 1136, 1137, 1167, 1169, 1181, -1 },
    { 1137, 1138, 1168, 1170, 1181, 1182 },
    { 1138, 1139, 1169, 1182, -1, -1 },
    { 1143, 1144, 1172, -1, -1, -1 },
    { 1144, 1145, 1171, -1, -1, -1 },
    { 1148, 1149, 1174, -1, -1, -1 },
    { 1149, 1150, 1173, -1, -1, -1 },
    { 1153, 1154, 1176, -1, -1, -1 },
    { 1154, 1155, 1175, -1, -1, -1 },
    { 1158, 1159, 1178, -1, -1, -1 },
    { 1159, 1160, 1177, -1, -1, -1 },
    { 1163, 1164, 1180, -1, -1, -1 },
    { 1164, 1165, 1179, -1, -1, -1 },
    { 1168, 1169, 1182, -1, -1, -1 },
    { 1169, 1170, 1181, -1, -1, -1 }
  };

  for (UInt_t i=0; i<GetNumPixels(); i++)
      (*this)[i].SetNeighbors(nn[i][0], nn[i][1], nn[i][2],
                              nn[i][3], nn[i][4], nn[i][5]);
}

