/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Javier Lopez 05/2001 (jlopez@ifae.es)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

#include "MMcEnerHisto.h" 

#include <MLog.h>
#include <TH1.h> 
#include <TF1.h> 
#include <TPaveLabel.h> 

ClassImp(MMcEnerHisto)

MMcEnerHisto::MMcEnerHisto(const int index)
{ 
  //
  //   default constructor
  //

  //  - we initialize the histogram and the gaus function
  //  - we have to give diferent names for the diferent histograms because
  //    root don't allow us to have diferent histograms with the same name

  char aux[15];
  sprintf (aux,"hLogEner%i",index);
  hLogEner = new TH1F(aux,"",100,0.5,4.5);
  sprintf (aux,"fLogEner%i",index);
  fLogEner = new TF1(aux,"gaus",1.,3.);

} 

MMcEnerHisto::~MMcEnerHisto()
{ 
  // we can not delete the histogram because it dissappear
  // when we display it using root

  //  delete hLogEner ;
  delete fLogEner ;
} 

void MMcEnerHisto::SetBins(Int_t nbins, Float_t xmin, Float_t xmax)
{
  hLogEner->SetBins(nbins,xmin,xmax);
}

void MMcEnerHisto::Fill(Float_t log10E, Float_t w)
{ 
  hLogEner->Fill(log10E,w) ; 
} 

void MMcEnerHisto::Fit(const char *fname, Option_t *option, Option_t *goption, Axis_t xxmin, Axis_t xxmax)
{ 
  hLogEner->Fit(fname,option,goption,xxmin,xxmax);
} 

void MMcEnerHisto::Draw(Option_t* option) 
{ 
  char text[50];
  sprintf(text,"Energy Threshold = %4.1f +- %4.1f GeV",GetEnerThre(),GetEnerThreErr());

  TPaveLabel* label = new TPaveLabel(2.2,0.75*(hLogEner->GetMinimum()+hLogEner->GetMaximum()),4.4,0.90*(hLogEner->GetMinimum()+hLogEner->GetMaximum()),text);
  
  hLogEner->SetYTitle("dN/dE") ;
  hLogEner->SetXTitle("Log(E) [GeV]") ;
  hLogEner->Draw(option) ;
  label->SetFillColor(10);
  label->SetTextSize(0.3);
  label->Draw();
}
 
void MMcEnerHisto::Print()
{ 
  cout << "Energy Threshold: " << GetEnerThre() << " +- " << GetEnerThreErr() << endl;
} 

Float_t MMcEnerHisto::GetEnerThre()
{
  return pow(10,fLogEner->GetParameter(1));
}

Float_t MMcEnerHisto::GetEnerThreErr() 
{
  return pow(10,fLogEner->GetParameter(1))*log(10)*fLogEner->GetParError(1);
}

Float_t MMcEnerHisto::GetPeakAtLogE()
{
  return fLogEner->GetParameter(1);
}

Float_t MMcEnerHisto::GetSigmaAtLogE()
{
  return fLogEner->GetParameter(2);
}



