/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  9/2001 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

#include "MAnalysis.h"

#include <TGButton.h>  // TGTextButton

ClassImp(MAnalysis)

enum {
    M_BUTTON_HILLAS,
    M_CHECK_DISPLHIL
};

void MAnalysis::AddButtons()
{
    TGTextButton *hillas = new TGTextButton(fTop2, "Calculate Standard Hillas", M_BUTTON_HILLAS);

    hillas->Associate(this);

    fList->Add(hillas);

    TGLayoutHints *laybut = new TGLayoutHints(kLHintsTop|kLHintsLeft, 10, 10, 5, 5);
    fList->Add(laybut);

    fTop2->AddFrame(hillas, laybut);
}

void MAnalysis::AddSetupTab()
{
    TGCompositeFrame *frame = CreateNewTab("Setup");

    TGLayoutHints *laybut = new TGLayoutHints(kLHintsTop|kLHintsLeft, 10, 10, 5, 5);
    fList->Add(laybut);

    fCheckButton1 = new TGCheckButton(frame, "Display Hillas Histograms when finished", M_CHECK_DISPLHIL);
    fCheckButton2 = new TGCheckButton(frame, "Display Star Map Histogram when finished", M_CHECK_DISPLHIL);

    fList->Add(fCheckButton1);
    fList->Add(fCheckButton2);

    frame->AddFrame(fCheckButton1, laybut);
    frame->AddFrame(fCheckButton2, laybut);
}

MAnalysis::MAnalysis(const TGWindow *main, const TGWindow *p,
                     const UInt_t w, const UInt_t h)
: MBrowser(main, p, w, h)
{
    AddButtons();
    AddSetupTab();

    MapSubwindows();

    Layout();

    SetWindowName("Analysis Window");
    SetIconName("Analysis");

    MapWindow();
}

// ======================================================================

#include "MParList.h"
#include "MTaskList.h"
#include "MGeomCamMagic.h"
#include "MPedestalCam.h"
#include "MHHillas.h"
#include "MHStarMap.h"
#include "MReadTree.h"
#include "MCerPhotCalc.h"
#include "MImgCleanStd.h"
#include "MHillasCalc.h"
#include "MFillH.h"
#include "MEvtLoop.h"
#include "MHillas.h"

void MAnalysis::CalculateHillas() const
{
    //
    // This is a demonstration program which calculates the Hillas
    // parameter out of a Magic root file.

    const Bool_t displhillas  = fCheckButton1->GetState();
    const Bool_t displstarmap = fCheckButton2->GetState();

    //
    // Create a empty Parameter List and an empty Task List
    // The tasklist is identified in the eventloop by its name
    //
    MParList  plist;

    MTaskList tlist;
    plist.AddToList(&tlist);

    //
    // The geometry container must be created by yourself to make sure
    // that you don't choos a wrong geometry by chance
    //
    MGeomCamMagic geomcam;
    plist.AddToList(&geomcam);

    MPedestalCam pedest;
    plist.AddToList(&pedest);

    //
    // The Hillas histograms (MHHillas) could be created automatically
    // but to make sure, that they are not deleted when the macro is
    // finished you must create them yourself and add it to the list
    //
    MHillas hillas;
    plist.AddToList(&hillas);

    MHHillas *hists = new MHHillas;
    plist.AddToList(hists);

    MHStarMap *smap = new MHStarMap;
    plist.AddToList(smap);

    // FIXME: Where do we delete this two objects???

    //
    // Now setup the tasks and tasklist:
    //
    //  1) read in the data from a magic root file   MReadTree
    //  2) calculate number of cerenkov photons      MCerPhotCalc
    //  3) clean the image                           MImgCleanStd
    //  4) calculate hillas                          MHillasCalc
    //  5) fill the hillas into the histograms       MFillH
    //

    //
    // The first argument is the tree you want to read.
    //   Events:     Cosmic ray events
    //   PedEvents:  Pedestal Events
    //   CalEvents:  Calibration Events
    //
    MReadTree      read("Events", fInputFile);
    MCerPhotCalc   ncalc;
    MImgCleanStd   clean;
    MHillasCalc    hcalc;
    MFillH         hfill(&hillas, hists);
    MFillH         sfill(&hillas, smap);

    tlist.AddToList(&read);
    tlist.AddToList(&ncalc);
    tlist.AddToList(&clean);
    tlist.AddToList(&hcalc);
    tlist.AddToList(&hfill);
    tlist.AddToList(&sfill);

    //
    // Create and setup the eventloop
    //
    MEvtLoop evtloop;
    evtloop.SetParList(&plist);

    //
    // Execute your analysis
    //
    if (!evtloop.Eventloop())
        return;

    //
    // After the analysis is finished we can display the histograms
    //

    if (displhillas)
        hists->Draw();

    if (displstarmap)
        smap->Draw();
}

// ======================================================================

Bool_t MAnalysis::ProcessMessage(Long_t msg, Long_t parm1, Long_t parm2)
{
    // Process events generated by the buttons in the frame.

    if (GET_MSG(msg)!=kC_COMMAND || GET_SUBMSG(msg)!=kCM_BUTTON)
        return MBrowser::ProcessMessage(msg, parm1, parm2);

    switch (parm1)
    {
    case M_BUTTON_HILLAS:
        if (!InputFileSelected())
        {
            DisplError("No Input (root) File selected!");
            return kTRUE;
        }

        switch (parm1)
        {
        case M_BUTTON_HILLAS:
            CalculateHillas();
            return kTRUE;
        }
        return kTRUE;
    }

    return MBrowser::ProcessMessage(msg, parm1, parm2);
}
