/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Harald Kornmayer 1/2001 (harald@mppmu.mpg.de)
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//
//  The class MHexagon is needed for the Event Display of
//  MAGIC.
//
#include "MHexagon.h"

#include <iostream.h>

#include <TVirtualPad.h>  // gPad

#include "MGeomPix.h"     // GetX

ClassImp(MHexagon);

// ------------------------------------------------------------------------
//
//   default constructor for MHexagon
//
MHexagon::MHexagon()
{
}

// ------------------------------------------------------------------------
//
//    normal constructor for MHexagon
//
MHexagon::MHexagon(Float_t x, Float_t y, Float_t d)
: TAttFill(0, 1001), fX(x), fY(y), fD(d)
{
}

// ------------------------------------------------------------------------
//
//    normal constructor for MHexagon
//
MHexagon::MHexagon(MGeomPix &pix)
: TAttFill(0, 1001)
{
    fX = pix.GetX();
    fY = pix.GetY();
    fD = pix.GetR();
}

// ------------------------------------------------------------------------
//
//    copy constructor for MHexagon
//
MHexagon::MHexagon(const MHexagon &hexagon)
{
    ((MHexagon&) hexagon).Copy(*this);
} 

// ------------------------------------------------------------------------
//
//     default destructor for MHexagon
//
MHexagon::~MHexagon()
{
}

// ------------------------------------------------------------------------
//
//     copy this hexagon to hexagon
//
void MHexagon::Copy(TObject &obj)
{
    MHexagon &hex = (MHexagon&) obj;

    TObject::Copy(obj);
    TAttLine::Copy(hex);
    TAttFill::Copy(hex);

    hex.fX = fX;
    hex.fY = fY;
    hex.fD = fD;
}

// ------------------------------------------------------------------------
//
// compute the distance of a point (px,py) to the Hexagon
// this functions needed for graphical primitives, that
// means without this function you are not able to interact
// with the graphical primitive with the mouse!!!
//
// All calcutations are running in pixel coordinates
//
Int_t MHexagon::DistancetoPrimitive(Int_t px, Int_t py)
{
    //
    //  compute the distance of the Point to the center of the Hexagon
    //
    const Int_t pxhex = gPad->XtoAbsPixel(fX);
    const Int_t pyhex = gPad->YtoAbsPixel(fY);

    const Double_t disthex = TMath::Sqrt((Double_t)((pxhex-px)*(pxhex-px) + (pyhex-py)*(pyhex-py)));

    const Double_t cosa = TMath::Abs(px-pxhex) / disthex;
    const Double_t sina = TMath::Abs(py-pyhex) / disthex;

    //
    // comput the distance to pixel border
    //
    const Double_t dx = fD * cosa / 2;
    const Double_t dy = fD * sina / 2;

    const Double_t xborder = fX + dx;
    const Double_t yborder = fY + dy;

    const Int_t pxborder = gPad->XtoAbsPixel(xborder);
    const Int_t pyborder = gPad->YtoAbsPixel(yborder);

    const Double_t distborder = TMath::Sqrt((Double_t)((pxborder-pxhex)*(pxborder-pxhex)+(pyborder-pyhex)*(pyborder-pyhex)));

    //
    //  compute the distance from the border of Pixel
    //  here in the first implementation is just circle inside
    //
    return distborder < disthex ? 999999 : 0;
}

// ------------------------------------------------------------------------
//
//  Draw this ellipse with new coordinate
//
void MHexagon::DrawHexagon(Float_t x, Float_t y, Float_t d)
{ 
    MHexagon *newhexagon = new MHexagon(x, y, d);

    TAttLine::Copy(*newhexagon);
    TAttFill::Copy(*newhexagon);

    newhexagon->SetBit(kCanDelete);
    newhexagon->AppendPad();
}

// ------------------------------------------------------------------------
//
//  This is the first test of implementing a clickable interface
//  for one pixel
//
void MHexagon::ExecuteEvent(Int_t event, Int_t px, Int_t py)
{

    switch (event)
    {
    case kButton1Down:
        cout << endl << "kButton1Down" << endl;
        SetFillColor(2);
        gPad->Modified();
        break;

    case kButton1Double:
        SetFillColor(0);
        gPad->Modified();
        break;
        //  case kMouseEnter:
        //     printf ("\n Mouse inside object \n" ) ;
        //     break;
    }
}

// ------------------------------------------------------------------------
//
//  list this hexagon with its attributes
//
void MHexagon::ls(Option_t *)
{
    TROOT::IndentLevel();
    Print();
}

// ------------------------------------------------------------------------
//
//  paint this hexagon with its attribute
//
void MHexagon::Paint(Option_t *)
{
    PaintHexagon(fX, fY, fD);
}

// ------------------------------------------------------------------------
//
//  draw this hexagon with the coordinates
//
void MHexagon::PaintHexagon(Float_t inX, Float_t inY, Float_t inD)
{ 

    const Int_t np = 6;

    const Float_t dx[np+1] = { .5   , 0.    , -.5   , -.5   , 0.    ,  .5   , .5    };
    const Float_t dy[np+1] = { .2886,  .5772,  .2886, -.2886, -.5772, -.2886, .2886 };

    static Float_t x[np+1], y[np+1];

    TAttLine::Modify();    // Change line attributes only if neccessary
    TAttFill::Modify();    // Change fill attributes only if neccessary

    //  calculate the positions of the pixel corners

    for (Int_t i=0; i<=np; i++)
    {
        x[i] = inX + dx[i]* inD;
        y[i] = inY + dy[i]* inD;
    }

    //   paint the pixel (hopefully)

    if (GetFillColor())
        gPad->PaintFillArea(np, x, y);

    if (GetLineStyle())
        gPad->PaintPolyLine(np+1, x, y);

}

// ------------------------------------------------------------------------
//
//  print/dump this hexagon with its attributes
//
void MHexagon::Print(Option_t *)
{
    cout << GetName() << ": X=" << fX << " Y=" << fY << "R=" << fD << endl;

    cout << " Color="     << GetLineColor() << endl;
    cout << " Style="     << GetLineStyle() << endl;
    cout << " Width="     << GetLineWidth() << endl;
    cout << " FillColor=" << GetFillColor() << endl;
}
