/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Harald Kornmayer 1/2001 (harald@mppmu.mpg.de)
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

#include "MCerPhotEvt.h"

#include <math.h>
#include <fstream.h>

#include <TCanvas.h>
#include <TClonesArray.h>

#include "MLog.h"
#include "MHexagon.h"
#include "MCerPhotPix.h"

ClassImp(MCerPhotEvt);

// --------------------------------------------------------------------------
//
// Creates a MCerPhotPix object for each pixel in the event
//
MCerPhotEvt::MCerPhotEvt(const char *name, const char *title) : fNumPixels(0)
{

  *fName  = name  ? name  : "MCerPhotEvt";
  *fTitle = title ? title : "(Number of Photon)-Event Information";
  
  fPixels = new TClonesArray ("MCerPhotPix", 577) ;

  //
  // FIXME: is this really necessary?
  //
  fPixels->Clear();
}

// --------------------------------------------------------------------------
//
// This is not yet implemented like it should.
//
void MCerPhotEvt::Draw(Option_t* option) 
{
    //
    //   FIXME!!! Here the Draw function of the CamDisplay
    //   should be called to add the CamDisplay to the Pad.
    //   The drawing should be done in MCamDisplay::Paint
    //

    //    MGeomCam *geom = fType ? new MGeomCamMagic : new MGeomCamCT1;
    //    MCamDisplay *disp = new MCamDisplay(geom);
    //    delete geom;
    //    disp->DrawPhotNum(this);
}

// --------------------------------------------------------------------------
//
// add a new pixel to the list and increase the number
// of valid pixels in the list by one
//
void MCerPhotEvt::AddPixel(Int_t id, Float_t nph, Float_t err)
{
    // TClonesArray -> 'operator new with placement' should be used
    new ((*fPixels)[fNumPixels++]) MCerPhotPix(id, nph, err);
}

// --------------------------------------------------------------------------
//
// reset counter and delete netries in list.
//
void MCerPhotEvt::Clear(Option_t *)
{
    fNumPixels = 0;
    fPixels->Clear();
}

// --------------------------------------------------------------------------
//
//  Dump the cerenkov photon event to *fLog
//
void MCerPhotEvt::Print(Option_t *) const
{
    const Int_t entries = fPixels->GetEntries();

    *fLog << "MCerPhotEvt::Print()" << endl
        << "Number of Pixels: " << fNumPixels
        << "(" << entries << ")"
        << endl ;

    for (Int_t il=0; il<entries; il++ )
        (*this)[il].Print();
}

// --------------------------------------------------------------------------
//
// Checks if in the pixel list is an entry with pixel id
//
Bool_t MCerPhotEvt::IsPixelExisting(Int_t id) const
{
    const Int_t entries = fPixels->GetEntries();

    for (Int_t il=0; il<entries; il++)
    {
        if (id == (*this)[il].GetPixId())
            return kTRUE;
    }

    return kFALSE;
} 

// --------------------------------------------------------------------------
//
//   Checks if in the pixel list is an entry with pixel id
//
Bool_t MCerPhotEvt::IsPixelUsed(Int_t id) const
{
    const Int_t entries = fPixels->GetEntries();

    for (Int_t il=0; il<entries; il++)
    {
        MCerPhotPix &pix = (*this)[il];

        if (id == pix.GetPixId() && pix.IsPixelUsed())
            return kTRUE;
    }

    return kFALSE;
} 

// --------------------------------------------------------------------------
//
//   Checks if in the pixel list is an entry with pixel id
//
Bool_t MCerPhotEvt::IsPixelCore(Int_t id) const
{
    const Int_t entries = fPixels->GetEntries();

    for (Int_t il=0; il<entries; il++)
    {
        MCerPhotPix &pix = (*this)[il];

        if (id == pix.GetPixId() && pix.IsCorePixel())
            return kTRUE;
    } 

    return kFALSE;
} 

// --------------------------------------------------------------------------
//
// get the minimum number of photons of all valid pixels in the list
//
Float_t MCerPhotEvt::GetNumPhotonsMin() const
{
    if (fNumPixels <= 0)
        return -5.;

    Float_t minval = (*this)[0].GetNumPhotons();

    Float_t testval;
    for (UInt_t i=1; i<fNumPixels; i++)
    {
        testval = (*this)[i].GetNumPhotons();

        if (testval < minval)
            minval = testval;
    }

    return minval;
}

// --------------------------------------------------------------------------
//
// get the maximum number of photons of all valid pixels in the list
//
Float_t MCerPhotEvt::GetNumPhotonsMax() const
{
    if (fNumPixels <= 0)
        return 50.;

    Float_t maxval = (*this)[0].GetNumPhotons();

    Float_t testval;
    for (UInt_t i=1; i<fNumPixels; i++)
    {
        testval = (*this)[i].GetNumPhotons();

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

