/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// MParContainer                                                            //
//                                                                          //
// The MParContainer class is the base class for all MARS parameter         //
// containers. At the moment it is almost the same than ROOT's TNamed.      //
// A TNamed contains the essential elements (name, title)                   //
// to identify a derived object in lists like our MParList or MTaskList.    //
// The main difference is that the name and title isn't stored and read     //
// to and from root files ("//!")                                           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
#include "MParContainer.h"

#include <TClass.h>      // IsA
#include <TROOT.h>       // TROOT::Identlevel
#include <TVirtualPad.h> // gPad

#include "MLog.h"

ClassImp(MParContainer);

// --------------------------------------------------------------------------
//
//  MParContainer copy ctor
//
MParContainer::MParContainer(const MParContainer &named)
{
    *fName  = *(named.fName);
    *fTitle = *(named.fTitle);

    fLog = named.fLog;

    fReadyToSave = named.fReadyToSave;
}

// --------------------------------------------------------------------------
//
//  MParContainer assignment operator.
//
MParContainer& MParContainer::operator=(const MParContainer& rhs)
{
    if (this == &rhs)
        return *this;

    TObject::operator=(rhs);

    *fName  = *(rhs.fName);
    *fTitle = *(rhs.fTitle);

    fLog         = rhs.fLog;
    fReadyToSave = rhs.fReadyToSave;

    return *this;
}

// --------------------------------------------------------------------------
//
//  Compare two MParContainer objects. Returns 0 when equal, -1 when this is
//  smaller and +1 when bigger (like strcmp).
//
Int_t MParContainer::Compare(TObject *obj)
{
    if (this == obj) return 0;
    return fName->CompareTo(obj->GetName());
}

// --------------------------------------------------------------------------
//
//  Copy this to obj.
//
void MParContainer::Copy(TObject &obj)
{
    MParContainer &cont = (MParContainer&)obj;

    TObject::Copy(obj);

    *cont.fName  = *fName;
    *cont.fTitle = *fTitle;

    cont.fLog         = fLog;
    cont.fReadyToSave = fReadyToSave;
}

// --------------------------------------------------------------------------
//
//  Encode MParContainer into output buffer.
//
void MParContainer::FillBuffer(char *&buffer)
{
    fName->FillBuffer(buffer);
    fTitle->FillBuffer(buffer);
}

// --------------------------------------------------------------------------
//
//  List MParContainer name and title.
//
void MParContainer::ls(Option_t *)
{
    TROOT::IndentLevel();
    *fLog <<"OBJ: " << IsA()->GetName() << "\t" << GetName() << "\t" << GetTitle() << " : "
        << Int_t(TestBit(kCanDelete)) << endl;
}

// --------------------------------------------------------------------------
//
//  Print MParContainer name and title.
//
void MParContainer::Print(Option_t *) const
{
    *fLog <<"OBJ: " << IsA()->GetName() << "\t" << GetName() << "\t" << GetTitle() << endl;
}

// --------------------------------------------------------------------------
//
//  Change (i.e. set) the name of the MParContainer.
//  WARNING !!
//  If the object is a member of a THashTable, THashList container
//  The HashTable must be Rehashed after SetName
//  For example the list of objects in the current directory is a THashList
//
void MParContainer::SetName(const char *name)
{
    *fName = name;
    if (gPad && TestBit(kMustCleanup)) gPad->Modified();
}

// --------------------------------------------------------------------------
//
//  Change (i.e. set) all the MParContainer parameters (name and title).
//  See also WARNING in SetName
//
void MParContainer::SetObject(const char *name, const char *title)
{
    *fName  = name;
    *fTitle = title;
    if (gPad && TestBit(kMustCleanup)) gPad->Modified();
}

// --------------------------------------------------------------------------
//
//  Change (i.e. set) the title of the MParContainer.
//
void MParContainer::SetTitle(const char *title)
{
    *fTitle = title;
    if (gPad && TestBit(kMustCleanup)) gPad->Modified();
}

// --------------------------------------------------------------------------
//
//  Return size of the MParContainer part of the TObject.
//
Int_t MParContainer::Sizeof() const
{
    Int_t nbytes = fName->Sizeof() + fTitle->Sizeof();
    return nbytes;
}

void MParContainer::AsciiRead(ifstream &fin)
{
    *fLog << "To use the the ascii input of " << GetName();
    *fLog << " you have to overload " << GetName() << "::AsciiRead." << endl;
}

void MParContainer::AsciiWrite(ofstream &fout) const
{
    *fLog << "To use the the ascii output of " << GetName();
    *fLog << " you have to overload " << GetName() << "::AsciiWrite." << endl;
}
