/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Harald Kornmayer 1/2001 (harald@mppmu.mpg.de)
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

#include "MMars.h"

#include <TApplication.h>  // gROOT->GetApplication()->...

#include <TGTab.h>         // TGTab
#include <TGMenu.h>        // TGPopupMenu
#include <TGMsgBox.h>      // TGMsgBox
#include <TGButton.h>      // TGPictureButton
#include <TG3DLine.h>      // TGHorizontal3DLine
                           // use TGSplitter.h for root<3.00
#include "MEvtDisp.h"
#include "MAnalysis.h"
#include "MDataCheck.h"
#include "MMonteCarlo.h"
#include "MCameraDisplay.h"

ClassImp(MMars)

enum {
    M_FILE_EXIT,
    M_FILE_ABOUT,

    M_PICTURE_MAGIC,
    M_PICTURE_MARS,

    M_BUTTON_EVTDISP,
    M_BUTTON_DATACHECK,
    M_BUTTON_ANALYSE,
    M_BUTTON_MONTECARLO,
    M_BUTTON_CAMDISPLAY
};

void MMars::CreateMenuBar()
{
    TGLayoutHints *laymenubar  = new TGLayoutHints(kLHintsTop|kLHintsLeft|kLHintsExpandX, 2, 2, 2, 2);
    TGLayoutHints *laymenuitem = new TGLayoutHints(kLHintsTop|kLHintsLeft, 0, 4, 0, 0);
    TGLayoutHints *laylinesep  = new TGLayoutHints(kLHintsTop|kLHintsExpandX);

    fList->Add(laymenubar);
    fList->Add(laymenuitem);
    fList->Add(laylinesep);

    TGPopupMenu *filemenu = new TGPopupMenu(gClient->GetRoot());
    filemenu->AddEntry("Exit", M_FILE_EXIT);
    //fFileMenu->Associate(this);

    TGMenuBar *menubar = new TGMenuBar(this, 1, 1, kHorizontalFrame);
    menubar->AddPopup("File", filemenu, laymenuitem);
    AddFrame(menubar, laymenubar);

    TGHorizontal3DLine *linesep = new TGHorizontal3DLine(this);
    AddFrame(linesep, laylinesep);

    fList->Add(filemenu);
    fList->Add(menubar);
    fList->Add(linesep);
}

void MMars::CreateTopFrame(TGHorizontalFrame *top)
{
    fPic1 = gClient->GetPicture("magiclogo.xpm");
    fPic2 = gClient->GetPicture("marslogo.xpm");

    TGPictureButton *magic = new TGPictureButton(top, fPic1, M_PICTURE_MAGIC);
    TGPictureButton *mars  = new TGPictureButton(top, fPic2, M_PICTURE_MARS);

    fList->Add(magic);
    fList->Add(mars);

    magic->Associate(this);
    mars->Associate(this);

    TGLayoutHints *lay1 = new TGLayoutHints(kLHintsLeft,  10., 10., 20., 10.);
    TGLayoutHints *lay2 = new TGLayoutHints(kLHintsRight, 10., 10., 10., 10.);

    fList->Add(lay1);
    fList->Add(lay2);

    top->AddFrame(magic, lay1);
    top->AddFrame(mars,  lay2);
}

void MMars::CreateTextButton(TGVerticalFrame *tab, const char *text,
                             const UInt_t id, TGLayoutHints *hints) const
{
    //
    // Create the button
    //
    TGTextButton *button = new TGTextButton(tab, text, id);

    //
    // Add button for 'auto-delete'
    //
    fList->Add(button);

    //
    // Send button events (meesages) to this object (s. ProcessMessage)
    //
    button->Associate(this);

    //
    // Add button with corresponding layout to containing frame
    //
    tab->AddFrame(button, hints);
}

void MMars::CreateBottomFrame(TGHorizontalFrame *low)
{
    //
    // Create Tab Container
    //
    TGLayoutHints *laytabs = new TGLayoutHints(kLHintsBottom|kLHintsExpandX|kLHintsExpandY, 5, 5, 5, 5);
    fList->Add(laytabs);

    TGTab *tabs = new TGTab(low, 400, 400);
    fList->Add(tabs);
    low->AddFrame(tabs, laytabs);

    //
    // Create Tab1
    //
    TGCompositeFrame *tf = tabs->AddTab("Control");

    TGVerticalFrame *tab1 = new TGVerticalFrame(tf, 300, 100);
    TGVerticalFrame *tab2 = new TGVerticalFrame(tf, 300, 100);

    fList->Add(tab1);
    fList->Add(tab2);

    TGLayoutHints *laytabx = new TGLayoutHints(kLHintsTop|kLHintsExpandX);
    fList->Add(laytabx);

    tf->AddFrame(tab1, laytabx);
    tf->AddFrame(tab2, laytabx);

    TGLayoutHints *laybut = new TGLayoutHints(kLHintsTop|kLHintsCenterX, 10, 10, 10, 10);
    fList->Add(laybut);

    CreateTextButton(tab2, "Event Display",  M_BUTTON_EVTDISP,    laybut);
    CreateTextButton(tab2, "Data Check",     M_BUTTON_DATACHECK,  laybut);
    CreateTextButton(tab2, "Analysis",       M_BUTTON_ANALYSE,    laybut);
    CreateTextButton(tab2, "Monte Carlo",    M_BUTTON_MONTECARLO, laybut);
    CreateTextButton(tab2, "Camera Display", M_BUTTON_CAMDISPLAY, laybut);
}

MMars::MMars(/*const TGWindow *p,*/ UInt_t w, UInt_t h)
: TGMainFrame(gClient->GetRoot(), w, h)
//: MBrowser (gClient->GetRoot(), gClient->GetRoot(), w, h)
{
    //
    // Create the deletion list
    //
    fList = new TList;
    fList->SetOwner();

    //
    // Create the MenuBar
    //
    CreateMenuBar();

    //
    // create and layout the frame/contents
    //
    TGHorizontalFrame *top = new TGHorizontalFrame(this, 300, 100);
    TGHorizontalFrame *low = new TGHorizontalFrame(this, 300, 100);

    TGHorizontal3DLine *linesep = new TGHorizontal3DLine(this);

    fList->Add(top);
    fList->Add(low);
    fList->Add(linesep);

    CreateTopFrame(top);
    CreateBottomFrame(low);

    TGLayoutHints *layout = new TGLayoutHints(kLHintsTop|kLHintsExpandX);
    fList->Add(layout);

    AddFrame(top,     layout);
    AddFrame(linesep, layout);
    AddFrame(low,     layout);

    //    CreateTopFrame(fTop2);
    //    CreateBottomFrame(fTop3);

    //
    //   Map the window, set up the layout, etc.
    //
    SetWMSizeHints(330, 410, 330, 410, 10, 10);  // set the smallest and biggest size of the Main frame

    MapSubwindows();

    Layout();

    SetWindowName("MARS Main Window");
    SetIconName("MARS");

    MapWindow();
} 

// ======================================================================

MMars::~MMars()
{
    gClient->FreePicture(fPic1);
    gClient->FreePicture(fPic2);

    delete fList;
}  
// ======================================================================

void MMars::CloseWindow()
{
   // Got close message for this MainFrame. Calls parent CloseWindow()
   // (which destroys the window) and terminate the application.
   // The close message is generated by the window manager when its close
   // window menu item is selected.

   TGMainFrame::CloseWindow();
   gROOT->GetApplication()->Terminate(0);
}

void MMars::DisplWarning(const char *txt)
{
    Int_t retval;
    new TGMsgBox(fClient->GetRoot(), this,
                 "WARNING!", txt,
                 kMBIconExclamation, 4, &retval);
}

Bool_t MMars::ProcessMessage(Long_t msg, Long_t parm1, Long_t)
{     
    // Process events generated by the buttons in the frame.

    switch (GET_MSG(msg))
    {
    case kC_COMMAND:
        switch (GET_SUBMSG(msg))
        {
        case kCM_BUTTON:

            switch (parm1)
            {

            case M_BUTTON_EVTDISP:
                new MEvtDisp(this);
                return kTRUE;

            case M_BUTTON_DATACHECK:
                new MDataCheck(this);
                return kTRUE;

            case M_BUTTON_ANALYSE:
                new MAnalysis(this);
                return kTRUE;

            case M_BUTTON_MONTECARLO:
                new MMonteCarlo(this);
                return kTRUE;

            case M_BUTTON_CAMDISPLAY:
                new MCameraDisplay(this);
                return kTRUE;

            case M_PICTURE_MAGIC:
                DisplWarning("Please open a Netscape to the MAGIC homepage\n"
                             "http://hegra1.mppmu.mpg.de/MAGICWeb/");
                return kTRUE;

            case M_PICTURE_MARS:
                DisplWarning("Please open a Netscape to the MARS homepage\n"
                             "http://magic.uni-sw.gwdg.de/mars/");
                return kTRUE;
            }

	case kCM_MENU:
            if (parm1!=M_FILE_EXIT)
                return kTRUE;

            CloseWindow();
            return kTRUE;
	}
    }

    return kTRUE;
}
