#include "MLogManip.h"

/*****************************************************************

 DISPLAY -- Display calibrated data

 datafile:
    A calibarted root file which came out of callist,
    e.g. 20170727_006_C.root

 lvl1, lvl2:
    Lower (1) and higher (2) cleaning level


 To run the macro from the command line (assuming you are in a directory
 Mars/build where you have built your Mars environment) ou can do

    root ../hawc/display.C\(\"filename_C.root\"\)

 or from within root

    [0] .x ../hawc/display.C("filename_C.root")


 You will get a status display. You can step through the events
 either by selecting Loop/SingleStep from the menu bar or
 by pressing the space key in the status window.

 Note that as a default (see below in the code), events which
 do not survive image cleaning are skipped.

******************************************************************/
int display(const char *datafile, Double_t lvl1=7.8, Double_t lvl2=3.9, const char *outpath=".")
{
    // The delta t [ns/deg] for the image cleaning
    double deltat = 17.5;

    // --------------------------------------------------------

    gLog.Separator("Star");
    gLog << all << "Calculate image parameters of sequence ";
    gLog << datafile << endl;
    gLog << endl;

    // ------------------------------------------------------

    // Allocate/open the status display
    MStatusDisplay *d = new MStatusDisplay;
    d->SetLoopStep();

    // Instantiate the list of tasks
    MTaskList tlist;

    // Instantiate the list of parameter containers
    MParList plist2;
    plist2.AddToList(&tlist);

    // Instatiate the event loop
    MEvtLoop loop;
    loop.SetDisplay(d);
    loop.SetParList(&plist2);

    // Instantiate the reading task
    MReadMarsFile read("Events");
    read.DisableAutoScheme();
    read.AddFile(datafile);

    // Instantiate the task which takes care of the size of all containers
    MGeomApply apply;

    // Instantiate the image cleaning

    // This is the most simple image cleaning possible, standard
    // tail cut based on absolute amplitudes
    // MImgCleanStd clean(lvl1, lvl2);
    // clean.SetMethod(MImgCleanStd::kAbsolute);
    // clean.SetPostCleanType(3);

    // Instantiate the image cleaning as described in the TRAC
    MImgCleanTime clean;
    clean.SetMinCount(0);
    clean.SetMinSize(50);
    clean.SetDeltaT(17.5);

    // Instantiate the calculation of the image parameters
    MHillasCalc hcalc;
    hcalc.Disable(MHillasCalc::kCalcConc);

    // Instantiate all the displays
    MHEvent evt01a(MHEvent::kEvtSignalDensity);
    MHEvent evt01b(MHEvent::kEvtArrTime);
    MHEvent evt02a(MHEvent::kEvtSignalDensity);
    MHEvent evt02b(MHEvent::kEvtArrTimeCleaned);
    MHEvent evt03 (MHEvent::kEvtIslandIndex);

    evt01a.SetName("CalibSig");
    evt01b.SetName("CalibTm");
    evt02a.SetName("CleanSig");
    evt02b.SetName("CleanTm");
    evt03 .SetName("IslandIdx");

    evt01a.SetMinimum(0);
    evt01b.SetMinimum(0);
    evt01b.SetMaximum(250);
    evt02a.SetMinimum(0);

    // Instantiate the tasks to fill the displays
    MFillH fill01a(&evt01a, "MSignalCam", "MFillH1b");
    MFillH fill01b(&evt01b, "MSignalCam", "MFillH1b");
    MFillH fill02a(&evt02a, "MSignalCam", "MFillH2a");
    MFillH fill02b(&evt02b, "MSignalCam", "MFillH2b");
    MFillH fill03 (&evt03,  "MSignalCam", "MFillH3");

    // Instantiate printing the hillas parameters
    MPrint print("MHillas");

    // Setup the task list
    tlist.AddToList(&read);
    tlist.AddToList(&apply);

    tlist.AddToList(&fill01a);
    tlist.AddToList(&fill01b);
    tlist.AddToList(&clean);

    // Remove the two following tasks from the loop
    // if you want to see all events not just the ones
    // which survived image cleaning successfully
    tlist.AddToList(&hcalc);
    tlist.AddToList(&print);

    // All those which come after MHillasCalc display the ellipse
    tlist.AddToList(&fill02a);
    tlist.AddToList(&fill02b);
    tlist.AddToList(&fill03);

    // Run the eventloop
    if (!loop.Eventloop())
        return 3;

    // Check if the display was closed (deleted) by the user
    if (!loop.GetDisplay())
        return 4;

    // ============================================================

    return 0;
}
