/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz 03/2017 <mailto:tbretz@physik.rwth-aachen.de>
!
!   Copyright: MAGIC Software Development, 2000-2017
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MGeomCamFAMOUS
//
// This class stores the geometry information of the final FAMOUS camera.
//    MGeomCamFAMOUS cam;        // Creates the final FAMOUS camera
//
////////////////////////////////////////////////////////////////////////////
#include "MGeomCamFAMOUS.h"
#include <iostream>
#include <TMath.h>

#include "MGeomPix.h"

ClassImp(MGeomCamFAMOUS);

using namespace std;

// --------------------------------------------------------------------------
//
MGeomCamFAMOUS::MGeomCamFAMOUS(const char *name)
    : MGeomCam(64, 0.55, name, "Geometry information of the final FAMOUS Camera")
{
    CreateCam();
    InitGeometry();
}

// --------------------------------------------------------------------------
//
MGeomCamFAMOUS::MGeomCamFAMOUS(Double_t dist, const char *name)
    : MGeomCam(64, dist, name, "Geometry information of the final FAMOUS Camera")
{
    CreateCam();
    InitGeometry();
}

// --------------------------------------------------------------------------
//
// Check if the photon which is flying along the trajectory u has passed
// (or will pass) the frame of the camera (and consequently get
// absorbed). The position p and direction u must be in the
// telescope coordinate frame, which is z parallel to the focal plane,
// x to the right and y upwards, looking from the mirror towards the camera.
//
// The units are cm.
//
Bool_t MGeomCamFAMOUS::HitFrame(MQuaternion p, const MQuaternion &u) const
{
    // z is defined from the mirror (0) to the camera (z>0).
    // Thus we just propagate to the focal plane (z=fDist)
    //p -= 1700./u.Z()*u;
    p.PropagateZ(u, GetCameraDist()*100); // m->cm

    // Add 10% to the max radius and convert from mm to cm
    return p.R()<GetMaxRadius()*0.11;//TMath::Abs(p.X())<65 && TMath::Abs(p.Y())<65;
}

// --------------------------------------------------------------------------
//
void MGeomCamFAMOUS::CreateCam()
{
    const static Int_t dir[6] =
    {
        MGeomPix::kDirSE,
        MGeomPix::kDirNE,
        MGeomPix::kDirN,
        MGeomPix::kDirNW,
        MGeomPix::kDirSW,
        MGeomPix::kDirS
    };

    const Int_t    rings    = 4;
    const Double_t diameter = 16;

    //
    //  add the first pixel to the list
    //
    const MGeomPix pix0(0, 0, diameter);
    SetAt(0, pix0);

    Double_t x, y;

    Int_t cnt  = 1;
    for (int ring=1; ring<=rings; ring++)
    {
        for (Int_t s=0; s<6; s++)
        {
            for (int i=1; i<=ring; i++)
            {
                MGeomPix::CalcXY(dir[s], ring, i, x, y);

                const MGeomPix pix(x*diameter, -y*diameter, diameter);
                SetAt(cnt++, pix);
            }
        }
    }

    MGeomPix::CalcXY(MGeomPix::kDirNW, 6, 2, x, y);
    const MGeomPix pix62(x*diameter, -y*diameter, diameter);
    SetAt(cnt++, pix62);

    MGeomPix::CalcXY(MGeomPix::kDirSW, 6, 4, x, y);
    const MGeomPix pix63(x*diameter, -y*diameter, diameter);
    SetAt(cnt++, pix63);

    MGeomPix::CalcXY(MGeomPix::kDirSE, 6, 2, x, y);
    const MGeomPix pix64(x*diameter, -y*diameter, diameter);
    SetAt(cnt++, pix64);


}
