#ifndef MARS_MFresnelLens
#define MARS_MFresnelLens

#include <TVector3.h>

#ifndef MARS_MOptics
#include "MOptics.h"
#endif
#ifndef MARS_MSpline3
#include "MSpline3.h"
#endif
#ifndef MARS_MQuaternion
#include "MQuaternion.h"
#endif

class MFresnelLens : public MOptics
{
public:
    // --------------- Helper class ---------------------
    // This is a TVector3 which is nromalized and takes theta and phi as arguent
    // same as calling
    // TVector3 v;
    // v.SetMagThetaPhi(1, theta, phi);
    class VectorNorm : public TVector3
    {
    public:
        VectorNorm(double theta, double phi) :
            TVector3(
                     sin(theta) * cos(phi),
                     sin(theta) * sin(phi),
                     cos(theta)
                    )
        {
        }
    };

    // ----------- Properties of the grooves -------------

    struct Cone
    {
        double z;          // z-coordinate of the peak of the cone (x=y=0)
        double theta;      // Opening angle of the cone [rad]
        double tan_theta;  // tan(theta)
        double tan_theta2; // tan(theta)*tan(theta)
        double h;          // height of the slope (intersection point in the valley), h<0
        double theta_norm; // theta angle of the normal vector corresponding to the cone surface
    };

    struct Groove
    {
        double r;    // Radius of the intersection point between slope and draft angle

        Cone slope;  // Description of the slope angle
        Cone draft;  // Description of the draft angle
    };

private:

    // --------------------------------------------------

    Double_t fMaxR;  //!

    // ------------- Properties of the lens --------------

    double fF;      // Focal length
    double fR;      // Radius of lens
    double fW;      // Width of groove
    double fH;      // Thickness of lens
    double fN;      //! Reference refractive index for lens design
    double fLambda; // Wavelength [nm] corresponding to fN (at which lens is optimized)
    double fVc;     //! Velocity of light within the lens material [cm/ns]
    double fPSF;    // Measure for the surface roughness

    std::vector<Groove> fGrooves; //! Collection of all grooves

    // ----------- Properties of the material -------------

    MSpline3 fAbsorptionLength;   // Spline storing the absorption length vs wavelength

    // ----------------------------------------------------

    bool fSlopeAbsorption;   //! Absorb rays which hit the slope surface
    bool fDraftAbsorption;   //! Absorb rays which hit the draft surface
    bool fBottomReflection;  //! Absorb rays which would be reflected at the exit surface
    bool fDisableMultiEntry; //! Absorb upgoing rays inside the material which do not originate from the exit surface
    bool fFresnelReflection; //! Disable Fresnel reflection
    UInt_t fMinHits;         //! Minimum number of surface contacts to define a successfull passage
    UInt_t fMaxHits;         //! Maximum number of surface contacts to define a successfull passage (0=unlimited)

    void InitMaxR();
    void InitGeometry(double maxr, double width, double N0, double F, double d);
    bool Transmission(double dt, double lambda) const;

    double CalcIntersection(const MQuaternion &p, const MQuaternion &u, const Cone &cone) const;
    int FindPeak(size_t i, const MQuaternion &p, const MQuaternion &u) const;

    //int EnterGroove(int surface, double n0, double lambda, MQuaternion &pos, MQuaternion &dir) const;
    //int LeavePeak(int surface, double n0, double lambda, MQuaternion &pos, MQuaternion &dir, double T0) const;

    int EnterGroove(int surface, double n0, MQuaternion &pos, MQuaternion &dir) const;
    int LeavePeak(int surface, double n0, MQuaternion &pos, MQuaternion &dir, double T0) const;

public:
    MFresnelLens(const char *name=NULL, const char *title=NULL);

    Double_t GetMaxR() const { return fMaxR; }
    Double_t GetA() const;

    virtual Bool_t CanHit(const MQuaternion &p) const;

    Int_t ExecuteOptics(MQuaternion &p, MQuaternion &u, const Short_t &) const;
    Int_t TraceRay(std::vector<MQuaternion> &vec, MQuaternion &p, MQuaternion &u, const Short_t &wavelength, bool verbose=false) const;

    Bool_t IsValid() const { return fGrooves.size(); }

    // -----------------------------------------------------------

    void SetPSF(const double &psf) { fPSF = psf; }
    void DefineLens(double F=50.21, double D=54.92, double w=0.01, double h=0.25, double lambda=546);

    Int_t ReadTransmission(const TString &file, float thickness, bool correction=true);

    // -----------------------------------------------------------

    void EnableSlopeAbsorption(bool abs=true)    { fSlopeAbsorption   =  abs; }
    void EnableDraftAbsorption(bool abs=true)    { fDraftAbsorption   =  abs; }
    void DisableBottomReflection(bool ref=true)  { fBottomReflection  = !ref; }
    void DisableMultiEntry(bool mul=true)        { fDisableMultiEntry =  mul; }
    void DisableFresnelReflection(bool ref=true) { fFresnelReflection = !ref; }

    void SetMinHits(UInt_t min) { fMinHits = min; }
    void SetMaxHits(UInt_t max) { fMaxHits = max; }

    // -----------------------------------------------------------

    const std::vector<MFresnelLens::Groove> GetGrooves() const { return fGrooves; }
    const Groove &GetGroove(int i) const { return fGrooves[i]; }
    const Groove &operator[](int i) const { return fGrooves[i]; }
    size_t GetNumGrooves() const { return fGrooves.size(); }

    // -----------------------------------------------------------

    static double RefractiveIndex(double lambda);
    static double SlopeAngle(double r, double F, double n, double d);
    static double DraftAngle(double r);

    static double SlopeAngleParabolic(double r, double F, double n0, double n1, double d);
    static double SlopeAngleAspherical(double r);
    static double SlopeAngleOptimized(double r, double F, double n);

    double SlopeAngle(const double &r) const
    {
        return SlopeAngle(r, fF, fN, fH);
    }

    // -----------------------------------------------------------

    ClassDef(MFresnelLens, 1) // Parameter container storing the description of a fresnel lens
};
    
#endif
