/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  07/2011 <mailto:thomas.bretz@epfl.ch>
!
!   Copyright: MAGIC Software Development, 2000-2011
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MRawFitsRead
//
//  This tasks reads the fits data file in the form used by FACT.
//
//  Input Containers:
//   -/-
//
//  Output Containers:
//   MRawRunHeader, MRawEvtHeader, MRawEvtData, MRawEvtTime
//
//////////////////////////////////////////////////////////////////////////////
#include "MRawFitsRead.h"

#include <TClass.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MFits.h"
#include "MTime.h"

#include "MArrayB.h"

#include "MRawRunHeader.h"
#include "MRawEvtHeader.h"
#include "MRawEvtData.h"

ClassImp(MRawFitsRead);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. It tries to open the given file.
//
MRawFitsRead::MRawFitsRead(const char *fname, const char *name, const char *title)
    : MRawFileRead(fname, name, title)
{
}

Bool_t MRawFitsRead::IsFits(const char *name)
{
    MZlib fin(name);
    if (!fin)
        return 0;

    Byte_t c[6];
    fin.read((char*)c, 6);
    if (!fin)
        return 0;

    return memcmp(c, "SIMPLE", 6)==0;
}

istream *MRawFitsRead::OpenFile(const char *filename)
{
    return new MFits(filename);
}

Bool_t MRawFitsRead::ReadRunHeader(istream &stream)
{
    MFits &fin = static_cast<MFits&>(stream);

    fRawRunHeader->SetValidMagicNumber();
    fRawRunHeader->SetNumEvents(fin.GetUInt("NAXIS2"));
    fRawRunHeader->InitPixels(fin.GetUInt("NPIX"));
    fRawRunHeader->SetObservation("", "FACT");
    fRawRunHeader->SetRunInfo(0, fin.GetUInt("NIGHT"), fin.GetUInt("RUNID"));
    fRawRunHeader->InitFact(fin.GetUInt("NPIX")/9, 9, fin.GetUInt("NROI"));
    fRawRunHeader->SetFormat(0xf172, fin.GetUInt("BLDVER"));
    fRawRunHeader->SetRunType(0/*data*/);

    return kTRUE;
}

Bool_t::MRawFitsRead::InitReadData(istream &stream)
{
    MFits &fin = static_cast<MFits&>(stream);

    MArrayB **data   = reinterpret_cast<MArrayB**>(fRawEvtData1->DataMember("fHiGainFadcSamples"));
    UInt_t   *evtnum = reinterpret_cast<UInt_t*>  (fRawEvtHeader->DataMember("fDAQEvtNumber"));

    if (!data || !evtnum)
        return kFALSE;

    fRawEvtData1->ResetPixels();
    fRawEvtData2->ResetPixels(0, 0);

    if (!fin.SetRefAddress("EventNum", *evtnum))
        return kFALSE;

    if (!fin.SetRefAddress("PCTime", fPCTime))
        return kFALSE;

    if (!fin.SetPtrAddress("Data", (uint16_t*)(*data)->GetArray(), (*data)->GetSize()/2))
        return kFALSE;

    fRawEvtData1->SetIndices();

    return kTRUE;
}

Bool_t MRawFitsRead::ReadEvent(istream &stream)
{
    if (!static_cast<MFits&>(stream).GetNextRow())
        return kFALSE;

    fRawEvtTime->SetUnixTime(fPCTime, 0);

    fRawEvtData1->SetReadyToSave();
    fRawEvtData2->SetReadyToSave();

    return kTRUE;
}

void MRawFitsRead::SkipEvent(istream &fin)
{
    static_cast<MFits&>(fin).SkipNextRow();
}
