#ifndef MARS_DrsCalib
#define MARS_DrsCalib

#include <math.h>   // fabs
#include <errno.h>  // errno

#include "fits.h"

class DrsCalibrate
{
protected:
    uint64_t fNumEntries;

    size_t fNumSamples;
    size_t fNumChannels;

    std::vector<int64_t> fSum;
    std::vector<int64_t> fSum2;

    /*
    vector<map<int16_t, uint8_t> > fMedian;
    vector<int16_t> fResult;
     */
public:
    DrsCalibrate() : fNumEntries(0), fNumSamples(0), fNumChannels(0) { }
    void Reset()
    {
        fNumEntries  = 0;
        fNumSamples  = 0;
        fNumChannels = 0;

        fSum.clear();
        fSum2.clear();
    }

    void InitSize(uint16_t channels, uint16_t samples)
    {
        fNumChannels = channels;
        fNumSamples  = samples;

        fSum.resize(samples*channels);
        fSum2.resize(samples*channels);

        //fMedian.resize(40*9*1024);
    }
/*
                continue;

                if (ch<40*9)
                {
                    fMedian[abs][val[rel]]++;

                    int n= 8;
                    if (fNumEntries>0 && fNumEntries%(n*100)==0)
                    {
                        fResult.resize(40*9*1024);

                        uint16_t sum = 0;
                        for (map<int16_t, uint8_t>::const_iterator it=fMedian[abs].begin();
                             it!=fMedian[abs].end(); it++)
                        {
                            map<int16_t, uint8_t>::const_iterator is = it;
                            is++;
                            if (is==fMedian[abs].end())
                                break;

                            sum += it->second;

                            double med = 0;

                            med += int64_t(it->second)*int64_t(it->first);
                            med += int64_t(is->second)*int64_t(is->first);
                            med /= int64_t(it->second)+int64_t(is->second);

                            if (sum==n*50)
                            {
                                fResult[abs] = it->first;
                                cout << ch << " MED+(50): " << it->first << endl;
                            }
                            if (sum<n*50 && sum+is->second>n*50)
                            {
                                fResult[abs] = med;
                                cout << ch << " MED-(50): " << med << endl;
                            }
                        }
                        cout << ch << " AVG=" << float(fSum[abs])/fNumEntries << endl;
                        fMedian[abs].clear();
                    }
                } // -2029 -2012 -2003 -1996 -1990 // -1955
                */

    void AddRel(const int16_t *val, const int16_t *start)
    {
        for (size_t ch=0; ch<fNumChannels; ch++)
        {
            const int16_t spos = start[ch];
            if (spos<0)
                continue;

            const size_t pos = ch*fNumSamples;

            for (size_t i=0; i<fNumSamples; i++)
            {
                // Value is relative to trigger
                // Abs is corresponding index relative to DRS pipeline
                const size_t rel = pos +  i;
                const size_t abs = pos + (spos+i)%fNumSamples;

                const int64_t v = val[rel];

                fSum[abs]  += v;
                fSum2[abs] += v*v;
            }
        }

        fNumEntries++;
    }

    void AddRel(const int16_t *val,    const int16_t *start,
                const int32_t *offset, const uint32_t scale)
    {
        for (size_t ch=0; ch<fNumChannels; ch++)
        {
            const int16_t spos = start[ch];
            if (spos<0)
                continue;

            const size_t pos = ch*fNumSamples;

            for (size_t i=0; i<fNumSamples; i++)
            {
                // Value is relative to trigger
                // Offset is relative to DRS pipeline
                // Abs is corresponding index relative to DRS pipeline
                const size_t rel = pos +  i;
                const size_t abs = pos + (spos+i)%fNumSamples;

                const int64_t v = int64_t(val[rel])*scale-offset[abs];

                fSum[abs]  += v;
                fSum2[abs] += v*v;
            }
        }

        fNumEntries++;
    }

    void AddAbs(const int16_t *val,    const  int16_t *start,
                const int32_t *offset, const uint32_t scale)
    {
        for (size_t ch=0; ch<fNumChannels; ch++)
        {
            const int16_t spos = start[ch];
            if (spos<0)
                continue;

            const size_t pos = ch*fNumSamples;

            for (size_t i=0; i<fNumSamples; i++)
            {
                // Value is relative to trigger
                // Offset is relative to DRS pipeline
                // Abs is corresponding index relative to DRS pipeline
                const size_t rel = pos +  i;
                const size_t abs = pos + (spos+i)%fNumSamples;

                const int64_t v = int64_t(val[rel])*scale-offset[abs];

                fSum[rel]  += v;
                fSum2[rel] += v*v;
            }
        }

        fNumEntries++;
    }

    static void ApplyCh(float *vec, const int16_t *val, int16_t start, uint32_t roi,
                        const int32_t *offset, const uint32_t scaleabs,
                        const int64_t *gain,   const uint64_t scalegain,
                        const int64_t *trgoff, const uint64_t scalerel)
    {
        if (start<0)
        {
            memset(vec, 0, roi);
            return;
        }

        for (size_t i=0; i<roi; i++)
        {
            // Value is relative to trigger
            // Offset is relative to DRS pipeline
            // Abs is corresponding index relative to DRS pipeline
            const size_t rel = i;
            const size_t abs = (start+i)%1024;

            const int64_t v =
                + (int64_t(val[rel])*scaleabs-offset[abs])*scalerel
                - trgoff[rel]
                ;

            const int64_t div = gain[abs]*scalerel;
            vec[rel] = double(v)*scalegain/div;
        }
    }

    static void Apply(float *vec, const int16_t *val, const int16_t *start, uint32_t roi,
                      const int32_t *offset, const uint32_t scaleabs,
                      const int64_t *gain,   const uint64_t scalegain,
                      const int64_t *trgoff, const uint64_t scalerel)
    {
        for (size_t ch=0; ch<1440; ch++)
        {
            const size_t pos = ch*roi;
            const size_t drs = ch*1024;
            ApplyCh(vec+pos, val+pos, start[ch], roi,
                    offset+drs, scaleabs,
                    gain  +drs, scalegain,
                    trgoff+drs, scalerel);
        }
    }

    static void RemoveSpikes(float *vec, uint32_t roi)
    {
        if (roi<4)
            return;

        for (size_t ch=0; ch<1440; ch++)
        {
            float *p = vec + ch*roi;

            for (size_t i=1; i<roi-2; i++)
            {
                if (p[i]-p[i-1]>25 && p[i]-p[i+1]>25)
                {
                    p[i] = (p[i-1]+p[i+1])/2;
                }

                if (p[i]-p[i-1]>22 && fabs(p[i]-p[i+1])<4 && p[i+1]-p[i+2]>22)
                {
                    p[i] = (p[i-1]+p[i+2])/2;
                    p[i+1] = p[i];
                }
            }
        }
    }

    std::pair<std::vector<double>,std::vector<double> > GetSampleStats() const
    {
        if (fNumEntries==0)
            return make_pair(std::vector<double>(),std::vector<double>());

        std::vector<double> mean(fSum.size());
        std::vector<double> error(fSum.size());

        std::vector<int64_t>::const_iterator it = fSum.begin();
        std::vector<int64_t>::const_iterator i2 = fSum2.begin();
        std::vector<double>::iterator        im = mean.begin();
        std::vector<double>::iterator        ie = error.begin();

        while (it!=fSum.end())
        {
            *im = /*cnt<fResult.size() ? fResult[cnt] :*/ double(*it)/fNumEntries;
            *ie = sqrt(double(*i2*int64_t(fNumEntries) - *it * *it))/fNumEntries;

            im++;
            ie++;
            it++;
            i2++;
        }


        /*
         valarray<double> ...

         mean /= fNumEntries;
         error = sqrt(error/fNumEntries - mean*mean);
         */

        return make_pair(mean, error);
    }

    void GetSampleStats(float *ptr, float scale) const
    {
        if (fNumEntries==0)
        {
            memset(ptr, 0, sizeof(float)*1024*1440*2);
            return;
        }

        std::vector<int64_t>::const_iterator it = fSum.begin();
        std::vector<int64_t>::const_iterator i2 = fSum2.begin();

        while (it!=fSum.end())
        {
            *ptr             = scale*double(*it)/fNumEntries;
            *(ptr+1024*1440) = scale*sqrt(double(*i2*int64_t(fNumEntries) - *it * *it))/fNumEntries;

            ptr++;
            it++;
            i2++;
        }
    }

    static void GetPixelStats(float *ptr, const float *data, uint16_t roi)
    {
        if (roi==0)
            return;

        for (int i=0; i<1440; i++)
        {
            const float *vec = data+i*roi;

            int    pos  = 0;
            double sum  = vec[0];
            double sum2 = vec[0]*vec[0];
            for (int j=1; j<roi; j++)
            {
                sum  += vec[j];
                sum2 += vec[j]*vec[j];

                if (vec[j]>vec[pos])
                    pos = j;
            }
            sum  /= roi;
            sum2 /= roi;

            *(ptr+0*1440+i) = sum;
            *(ptr+1*1440+i) = sqrt(sum2 - sum * sum);
            *(ptr+2*1440+i) = vec[pos];
            *(ptr+3*1440+i) = pos;
        }
    }

    static void GetPixelMax(float *max, const float *data, uint16_t roi, int32_t first, int32_t last)
    {
        if (roi==0 || first<0 || last<0 || first>=roi || last>=roi || last<first)
            return;

        for (int i=0; i<1440; i++)
        {
            const float *beg = data+i*roi+first;
            const float *end = data+i*roi+last;

            const float *pmax = beg;

            for (const float *ptr=beg+1; ptr<=end; ptr++)
                if (*ptr>*pmax)
                    pmax = ptr;

            max[i] = *pmax;
        }

    }

    const std::vector<int64_t> &GetSum() const { return fSum; }

    uint64_t GetNumEntries() const { return fNumEntries; }
};

struct DrsCalibration
{
    std::vector<int32_t> fOffset;
    std::vector<int64_t> fGain;
    std::vector<int64_t> fTrgOff;

    uint64_t fNumOffset;
    uint64_t fNumGain;
    uint64_t fNumTrgOff;

    uint32_t fStep;

    DrsCalibration() :
        fOffset(1440*1024, 0),
        fGain(1440*1024, 4096),
        fTrgOff(1440*1024, 0),
        fNumOffset(1),
        fNumGain(2000),
        fNumTrgOff(1),
        fStep(0)
    {
    }

    void Clear()
    {
        // Default gain:
        // 0.575*[45590]*2.5V / 2^16 = 0.99999 V
        fOffset.assign(1440*1024, 0);
        fGain.assign  (1440*1024, 4096);
        fTrgOff.assign(1440*1024, 0);

        fNumOffset = 1;
        fNumGain   = 2000;
        fNumTrgOff = 1;

        fStep = 0;
    }

    std::string ReadFitsImp(const std::string &str, std::vector<float> &vec)
    {
        std::fits file(str);
        if (!file)
        {
            std::ostringstream msg;
            msg << "Could not open file " << str << ": " << strerror(errno);
            return msg.str();
        }

        if (file.GetStr("TELESCOP")!="FACT")
        {
            std::ostringstream msg;
            msg << "Reading " << str << " failed: Not a valid FACT file (TELESCOP not FACT in header)";
            return msg.str();
        }

        if (!file.HasKey("STEP"))
        {
            std::ostringstream msg;
            msg << "Reading " << str << " failed: Is not a DRS calib file (STEP not found in header)";
            return msg.str();
        }

        if (file.GetNumRows()!=1)
        {
            std::ostringstream msg;
            msg << "Reading " << str << " failed: Number of rows in table is not 1.";
            return msg.str();
        }

        vec.resize(1440*1024*6+3);

        float *base = vec.data();

        file.SetPtrAddress("RunNumberBaseline",      base,   1);
        file.SetPtrAddress("RunNumberGain",          base+1, 1);
        file.SetPtrAddress("RunNumberTriggerOffset", base+2, 1);
        file.SetPtrAddress("BaselineMean",           base+0*1024*1440+3, 1024*1440);
        file.SetPtrAddress("BaselineRms",            base+1*1024*1440+3, 1024*1440);
        file.SetPtrAddress("GainMean",               base+2*1024*1440+3, 1024*1440);
        file.SetPtrAddress("GainRms",                base+3*1024*1440+3, 1024*1440);
        file.SetPtrAddress("TriggerOffsetMean",      base+4*1024*1440+3, 1024*1440);
        file.SetPtrAddress("TriggerOffsetRms",       base+5*1024*1440+3, 1024*1440);

        if (!file.GetNextRow())
        {
            std::ostringstream msg;
            msg << "Reading data from " << str << " failed.";
            return msg.str();
        }

        fStep      = file.GetUInt("STEP");
        fNumOffset = file.GetUInt("NBOFFSET");
        fNumGain   = file.GetUInt("NBGAIN");
        fNumTrgOff = file.GetUInt("NBTRGOFF");

        fOffset.resize(1024*1440);
        fGain.resize(1024*1440);
        fTrgOff.resize(1024*1440);

        // Convert back to ADC counts: 256/125 = 4096/2000
        // Convert back to sum (mean * num_entries)
        for (int i=0; i<1024*1440; i++)
        {
            fOffset[i] = fNumOffset           *256*base[i+1024*1440*0+3]/125;
            fGain[i]   = fNumOffset*fNumGain  *256*base[i+1024*1440*2+3]/125;
            fTrgOff[i] = fNumOffset*fNumTrgOff*256*base[i+1024*1440*4+3]/125;
        }

        // DAC:  0..2.5V == 0..65535
        // V-mV: 1000
        //fNumGain *= 2500*50000;
        //for (int i=0; i<1024*1440; i++)
        //    fGain[i] *= 65536;
        if (fStep==0)
        {
            for (int i=0; i<1024*1440; i++)
                fGain[i] = fNumOffset*4096;
        }
        else
        {
            fNumGain *= 1953125;
            for (int i=0; i<1024*1440; i++)
                fGain[i] *= 1024;
        }

        // Now mark the stored DRS data as "officially valid"
        // However, this is not thread safe. It only ensures that
        // this data is not used before it is completely and correctly
        // read.
        fStep++;

        return std::string();
    }

    std::string ReadFitsImp(const std::string &str)
    {
        std::vector<float> vec;
        return ReadFitsImp(str, vec);
    }

    bool IsValid() { return fStep>2; }

    void Apply(float *vec, int16_t *val, const int16_t *start, uint32_t roi)
    {
        DrsCalibrate::Apply(vec, val, start, roi,
                            fOffset.data(), fNumOffset,
                            fGain.data(),   fNumGain,
                            fTrgOff.data(), fNumTrgOff);

    }
};

#endif
