/*
 * factfits.h
 *
 *  Created on: May 26, 2013
 *      Author: lyard
 */

#ifndef MARS_FACTFITS
#define MARS_FACTFITS

#include "zfits.h"

#ifndef __MARS__
namespace std
{
#endif

class factfits : public zfits
{
public:
    // Default constructor
    factfits(const string& fname, const string& tableName="", bool force=false) :
        zfits(fname, tableName, force)
    {
        if (init())
            readDrsCalib(fname);
    }

    // Alternative constructor
    factfits(const string& fname, const string& fout, const string& tableName, bool force=false) :
        zfits(fname, fout, tableName, force)
    {
        if (init())
            readDrsCalib(fname);
    }

    //
#if !defined(__MARS__) && !defined(__CINT__)
    bool GetRow(size_t row, bool check=true)
#else
    bool GetRowNum(size_t row, bool check=true)
#endif
    {
#if !defined(__MARS__) && !defined(__CINT__)
        if (!zfits::GetRow(row, check))
            return false;
#else
        if (!zfits::GetRowNum(row, check))
            return false;
#endif
        // This file does not contain fact data or no calibration to be applied
        if (fOffsetCalibration.empty())
            return true;

        //re-get the pointer to the data to access the offsets
        const uint8_t offset = (row*fTable.bytes_per_row)%4;

        int16_t *startCell = reinterpret_cast<int16_t*>(fBufferRow.data() + offset + fOffsetStartCellData);

        const Pointers::iterator dtaIt = fPointers.find("Data");
        if (dtaIt == fPointers.end()) return true;
        int16_t *data      = reinterpret_cast<int16_t*>(dtaIt->second);

        for (uint32_t i=0; i<1440*1024; i+=1024, startCell++)
        {
            for (uint32_t j=0; j<fNumRoi; j++, data++)
                *data += fOffsetCalibration[i + (*startCell+j)%1024];
        }

        return true;
    }

private:

    bool init()
    {
        if (!HasKey("NPIX") || !HasKey("NROI"))
            return false;

        if (Get<uint16_t>("NPIX")!=1440)
            return false;

        fNumRoi = Get<uint16_t>("NROI");
        if (fNumRoi>1024)
            return false;

        // check column details for Data
        const Table::Columns::const_iterator it = fTable.cols.find("Data");
        if (it==fTable.cols.end() || it->second.num!=1440*fNumRoi || it->second.type!='I')
            return false;

        // check column details for StartCellData
        const Table::Columns::const_iterator is = fTable.cols.find("StartCellData");
        if (is==fTable.cols.end() || is->second.num!=1440 || is->second.type!='I')
            return false;

        fOffsetStartCellData = is->second.offset;
        fOffsetData          = it->second.offset;

        return true;
    }

    //  Read the Drs calibration data
    void readDrsCalib(const string& fileName)
    {
        zfits calib(fileName, "DrsCalib");
        if (calib.bad())
        {
            clear(rdstate()|ios::badbit);
            return;
        }

        if (calib.eof())
            return;

        // Check correct size and format of table
        if (calib.GetNumRows() != 1)
        {
            clear(rdstate()|ios::badbit);
#ifdef __EXCEPTIONS
            throw runtime_error("Table 'DrsCalib' found, but not with one row as expected");
#else
            gLog << ___err___ << "ERROR - Table 'DrsCalib' found, but not with one row as expected" << endl;
            return;
#endif
        }
        if (calib.GetStr("TTYPE1") != "OffsetCalibration")
        {
            clear(rdstate()|ios::badbit);
#ifdef __EXCEPTIONS
            throw runtime_error("Table 'DrsCalib' found, but first column is not the one expected");
#else
            gLog << ___err___ << "ERROR - Table 'DrsCalib' found, but first column is not the one expected" << endl;
            return;
#endif
        }
        if (calib.GetStr("TFORM1") != "1474560I")  // 1024*1440
        {
            clear(rdstate()|ios::badbit);
#ifdef __EXCEPTIONS
            throw runtime_error("Table 'DrsCalib' has wrong column format (TFROM1)");
#else
            gLog << ___err___ << "ERROR - Table 'DrsCalib' has wrong column format (TFORM1)" << endl;
            return;
#endif
        }

        fOffsetCalibration.resize(1024*1440);

        calib.SetPtrAddress("OffsetCalibration", fOffsetCalibration.data());
        if (calib.GetNextRow())
            return;

        clear(rdstate()|ios::badbit);

#ifdef __EXCEPTIONS
        throw runtime_error("Reading column 'OffsetCalibration' failed.");
#else
        gLog << ___err___ << "ERROR - Reading column 'OffsetCalibration' failed." << endl;
#endif

    }

    vector<int16_t> fOffsetCalibration; ///< integer values of the drs calibration used for compression

    size_t fOffsetData;
    size_t fOffsetStartCellData;

    uint16_t fNumRoi;

#warning Time marker channels currently unhandled

}; //class factfits

#ifndef __MARS__
}; //namespace std
#endif

#endif
