#/bin/bash
#
# Uploads star or callisto data to SQL database
#
# Example:
# 	bash run-root2sql2.sh -I cred-file starrc 1 /data/HE01/star 20201111

set -o errexit
set -o errtrace
set -o nounset
set -o pipefail

function ErrExit()
{
	echo "ERROR: Line `caller`: ${BASH_COMMAND}" >&2
	exit 1
}

function StrgCExit()
{
	echo " "
	echo "$0 was forcefully terminated" >&2
	exit 1
}

trap ErrExit ERR
trap StrgCExit INT

# Path to the executable
readonly ROOT2SQL="/home/frankm/Dateien/FACT++/build/root2sql"
readonly PROGRAM=${0}

# Check if root2sql  exists
if [ ! -f "${ROOT2SQL}" ]
then
	echo "root2sql does not exists at ${ROOT2SQL}. Please change in script ${PROGRAM}."
	exit 1
fi

usage()
{
	echo "usage: $PROGRAM [-hIY] [Credentials Resource Telescope Dir Night]"
	echo "	-h 		display help"
	echo "	-I 		is Star data"
	echo "	-Y 		is Callisto data"
	exit 1;
}

# Check for flags
while getopts 'hIY' flag
do
	case "${flag}" in
		h) usage ;;
		I) SUFFIX=I;;
		Y) SUFFIX=Y;;
		*) usage ;;
	esac
done
shift $(($OPTIND-1))

# Check if at least two arguments are provided
if [ -z "${1}" ] || [ -z "${2}" ] || [ -z "${3}" ] || [ -z "${4}" ] || [ -z "${5}" ]
then
	echo "Not enough arguments. Check -h for help!"
	exit 1
fi

# File containing the access credentials for the database
CRED="${1}"
# Resource file to use (should match the input files)
RESOURCE="${2}"
# Specify the telescope for which to run the script.
TEL="${3}"
# Path where the data is stored... can be callisto or star
BASEIN="${4}"
# Night
NIGHT="${5}"

# The table to insert the data into (and create if not exists)
if [ $SUFFIX = I ]
then
	TABLE=Star
elif [ $SUFFIX = Y ]
then
	TABLE=Callisto
else
	echo "Wrong suffix."
	exit 1
fi

# Get all runs that can (and should) be calibrated
# Pipe the output to mysql and read the
# mysql output line-by-libe
echo \
"\
SELECT 
	NIGHT, RUNID
FROM 
	Calibration
WHERE
	Telescope = ${TEL}
AND
	NIGHT = ${NIGHT}
ORDER BY
	NIGHT, RUNID\

"\
 |  mysql \
       --defaults-file=${CRED} \
       --skip-column-names \
       --batch --raw \
       --compress \
 | \
while read -r -a LINE
do
	# Get NIGHT and RUNID of all files
	NIGHT=${LINE[0]}
	RUNID=${LINE[1]}

  	# Format file and path names
	DIR="${BASEIN}"/${NIGHT:0:4}/${NIGHT:4:2}/${NIGHT:6:2}

	PREFIX=`printf ${NIGHT}_%03d ${RUNID}`

	echo ""
	echo "${DIR}"/${PREFIX}

	# Check if processing was successfull
	if [ -f "${DIR}"/.${PREFIX}.success ]
	then

		${ROOT2SQL} "${DIR}"/${PREFIX}_${SUFFIX}.root \
			-C ${RESOURCE} \
			--table=${TABLE} \
			--const.Telescope=${TEL} \
			--const.NIGHT=${NIGHT} \
			--const.RUNID=${RUNID} \
			2>&1 | tee "${DIR}"/${PREFIX}-root2sql.log

		echo RC=${PIPESTATUS[0]} >> "${DIR}"/${PREFIX}-root2sql.log
	else
		echo Skipped.
	fi

done
