#!/bin/bash
#
# Run star macro on directory DIR for a given telescope TEL and night NIGHT 
# and write output to DIR/../star
#
# Example:
# 	bash run-star2.sh cred-file 1 /data/HE01/raw 20201111

set -o errexit
set -o errtrace
set -o nounset
set -o pipefail

function ErrExit()
{
	echo "ERROR: Line `caller`: ${BASH_COMMAND}" >&2
	exit 1
}

function StrgCExit()
{
	echo " "
	echo "$0 was forcefully terminated" >&2
	exit 1
}

trap ErrExit ERR
trap StrgCExit INT

readonly CALLISTO=callisto
# The .success file in /data/star will be resetted
readonly STAR=star
# Directory to Mars environment (usually Mars/build)
readonly MARS="/home/frankm/Dateien/Mars/build/"
# Absolut path to macros
readonly MACROS="/home/frankm/Dateien/Mars/hawc"
readonly PROGRAM=$0
REPLACE=false

usage()
{
	echo "usage: $PROGRAM [-hr] [Credentials Telescope Dir Night]" 
	echo "	-h 		display help"
	echo "	-r 		replace data"
	exit 1;
}

# Check for flags
while getopts 'hr' flag
do
	case "${flag}" in
		h) usage ;;
		r) REPLACE=true ;;
		*) usage ;;
	esac
done
shift $(($OPTIND-1))

# Check if Mars exists
if [ ! -d "${MARS}" ]
then
	echo "Mars does not exists at ${MARS}. Please change in \
		script ${PROGRAM}."
	exit 1
fi

# Check if MACROS exists
if [ ! -d "${MACROS}" ]
then
	echo "Macros directorey does not exists at ${MACROS}. Please change in \
		script ${PROGRAM}."
	exit 1
fi

# Check if at least two arguments are provided
if [ -z "${1}" ] || [ -z "${2}" ] || [ -z "${3}" ] || [ -z "${4}" ]
then
	echo "Not enough arguments. Check -h for help!"
	exit 1
fi

# File containing the access credentials for the database
CRED="${1}"
# Specify the telescope for which to run the script.
TEL="${2}"
# The data is expected to be found in /data/raw and data/callisto
DATA="${3}"
NIGHT="${4}"

# Get all runs that can (and should) be calibrated
# Pipe the output to mysql and read the
# mysql output line-by-libe
echo \
"\
SELECT 
    	NIGHT, RUNID
FROM 
    	Calibration
WHERE
	Telescope=${TEL}
AND
	NIGHT=${NIGHT}
ORDER BY
    	NIGHT, RUNID\
"\
 |  mysql \
       --defaults-file=${CRED} \
       --skip-column-names \
       --batch --raw \
       --compress \
       | \
       while read -r -a LINE
       do
	       # Get NIGHT and RUNID of all files
	       NIGHT=${LINE[0]}
	       RUNID=${LINE[1]}

  		# Formatting of the file paths and names
		DATPATH=${NIGHT:0:4}/${NIGHT:4:2}/${NIGHT:6:2}

		PREFIX=`printf ${NIGHT}_%03d ${RUNID}`

		DATNAME=`printf ${PREFIX}_Y.root ${RUNID}`

		OUT="${DATA}"/../${STAR}/${DATPATH}
		IN="${DATA}"/../${CALLISTO}/${DATPATH}

		echo ${DATNAME}
		echo ${IN} " => " ${OUT}

		if [ ${REPLACE} = true ]
		then
			echo "deleting ${OUT}/.${PREFIX}.succsess !!!"
			rm -f "${OUT}"/.${PREFIX}.success
		fi

  		# Check if not yet successfully processed but sucessfully calibrated
		if [[ -f "${IN}/.${PREFIX}.success" && ! -f "${IN}/.${PREFIX}.running" && ! -f "${OUT}/.${PREFIX}.success" ]]
		then

			cd ${MARS}

			mkdir -p "${OUT}"

     			# Flag that a process is running
			rm -f "${OUT}"/.${PREFIX}.done
			rm -f "${OUT}"/.${PREFIX}.success

			touch "${OUT}"/.${PREFIX}.running

     			# Execute image parameter calculation and write log-file
			root -b -q -l ${MACROS}/star.C'("'"${IN}"/${DATNAME}'","'"${OUT}"'")' \
				2>&1 | tee "${OUT}"/${PREFIX}.log

     			# Remember exit status of callisto
			RC=${PIPESTATUS[0]}

			echo RC=${RC} | tee -a "${OUT}"/${PREFIX}.log
			echo ${RC} > "${OUT}"/.${PREFIX}.done

     			# Flag that processing is finished
			rm -f "${OUT}"/.${PREFIX}.running

     			# If processing was successfull write coresponding flag
			if [ "${RC}" == "0" ]
			then
				touch "${OUT}"/.${PREFIX}.success
			fi

			cd -

		else
			echo Skipped.
		fi

	done
