#include <iostream.h>

#include <fstream.h>

#include <TStopwatch.h>
#include <TF1.h>
#include <TH1.h>
#include <TH2.h>
#include <TList.h>
#include <TFile.h>
#include <TTree.h>
#include <TBranch.h>
#include <TStyle.h>
#include <TRandom.h>
#include <TCanvas.h>

#include "mbase/MParContainer.h"
#include "mphys/MPhoton.h"
#include "mphys/MElectron.h"
#include "mphys/MPairProduction.h"
#include "mhist/MBinning.h"
#include "mhist/MH.h"

// 2.96
// 2.87
// 2.73
Double_t PrimSpect(Double_t *x, Double_t *k)
{
    return pow(pow(10, x[0]), k[0]);
}

Double_t PhotonSpect(Double_t *x, Double_t *k=NULL)
{
    Double_t Ep  = pow(10, x[0]);
    Double_t res = MPhoton::Int2(&Ep, k);

    return res*1e55; //65/k[0];
    // return MPhoton::Planck(&Ep, &k[1]);
}

Double_t Sbar_sigmas(Double_t *x, Double_t *k)
{
    Double_t sbar = pow(10, x[0]);

    Double_t s = 1./(sbar*4);

    Double_t sigma = MPhoton::Sigma_gg(&s);

    return sigma*sbar*1e28;
}

Double_t RandomTheta(Double_t Eg, Double_t Ep)
{
    Double_t E0 = 511e-6; // [GeV]

    Double_t f = Eg/E0*Ep/E0;

    if (f<1)
        return 0;

    TF1 func("RndTheta", Sbar_sigmas, 0, log10(f), 0);

    Double_t sbar  = pow(10, func.GetRandom());
    Double_t theta = acos(1.-sbar*2/f);

    return theta;
}

Double_t GetEnergy(Int_t n, Double_t lo, Double_t hi)
{
    static int bin=0;
    static TRandom rnd(0);

    Double_t w = log10(hi/lo)/n;

    Double_t E = lo*pow(10, rnd.Uniform(w) + w*bin);

    //cout << endl << w << " " << n << " " << w*bin << " " << E << endl;

    if (++bin==n)
        bin=0;

    return E;

    /*
    // TF1 src ("PrimarySpectrum", PrimSpect, log10(lo), log10(hi), 1); // 10GeV-1000PeV
    // src.SetParameter(0, 0);
    // Double_t E = pow(10, src.GetRandom());

    //
    // 0: 11111111111111|22222222222222   2   2^0 + 1    1
    // 1: 11111111|22222222222|33333333   3   2^1 + 1    2
    // 2: 11111|22222|33333|44444|55555   5   2^2 + 1    7
    // 3:  |     | |   | |  |  |   |                    15
    //

    static int run=0;
    static int num=1;

    Double_t w = log10(hi/lo)/((1<<run) + 1);

    Double_t E = lo*pow(10, num*w);

    if (num == (1<<run))
    {
        run++;
        num=1;
    }
    else
        num++;

        return E;
        */
}

void DoIt()
{
    //    Double_t R = 1798; // [kpc]
    Double_t startz = 0.003; //MParticle::ZofR(&R);

    //    cout << "R = " << R << "kpc" << endl;
    cout << "Z = " << startz << endl;

    static TRandom rand(0);
    MPairProduction pair;

    Double_t runtime = 15*60; // [s]

    Double_t lo = 1e4; 
    Double_t hi = 1e11;
    Double_t alpha = -2;

    Int_t nbins = 18; //4*log10(hi/lo);

    TH1D hist;
    TH1D histsrc;

    hist.SetMinimum(lo*lo * pow(lo, alpha)/10);
    histsrc.SetMinimum(lo*lo * pow(lo, alpha)/10);

    TList listg;
    TList liste;

    listg.SetOwner();
    liste.SetOwner();

    gStyle->SetOptStat(10);

    TH2D position;
    TH2D angle;
    TH1D histpos;
    TH1D hista;

    MBinning binspolx;
    MBinning binspoly;
    binspolx.SetEdges(16, -180, 180);
    binspoly.SetEdgesLog(20, 1e-10, 1e-3);
    MH::SetBinning(&position, &binspolx, &binspoly);
    MH::SetBinning(&angle,    &binspolx, &binspoly);
    MH::SetBinning(&histpos,  &binspoly);
    MH::SetBinning(&hista,    &binspoly);

    hista.SetTitle("Particle Angle");
    angle.SetTitle("Particle Angle");
    histpos.SetTitle("Particle Position");
    position.SetTitle("Particle Position");

    //
    // Don't change the order!!!
    //
    hist.SetFillStyle(0);
    hist.SetMarkerStyle(kPlus);
    histsrc.SetFillStyle(0);
    histsrc.SetMarkerStyle(kMultiply);

    MBinning bins;
    bins.SetEdgesLog(nbins, lo, hi);

    MH::SetBinning(&hist,    &bins);
    MH::SetBinning(&histsrc, &bins);

    MH::MakeDefCanvas();

    gPad->SetGrid();
    gPad->SetLogx();
    gPad->SetLogy();

    histsrc.SetName("Spectrum");
    histsrc.SetXTitle("E [GeV]");
    histsrc.SetYTitle("E^{2}\\dotCounts");
    histsrc.GetXaxis()->SetLabelOffset(-0.015);
    histsrc.GetXaxis()->SetTitleOffset(1.1);

    histsrc.Draw("P");
    hist.Draw("Psame");
    histsrc.Draw("Csame");
    hist.Draw("Csame");

    // ------------------------------

    MPhoton *p4file  = NULL;
    MElectron *e4file  = NULL;

    TFile file("cascade.root", "RECREATE", "Intergalactic cascade", 9);
    TTree *T  = new TTree("Photons",   "Photons from Cascade");
    TTree *T2 = new TTree("Electrons", "Electrons in the Cascade");
    TBranch *B =T ->Branch("MPhoton.",   "MPhoton",   &p4file, 32000);
    TBranch *B2=T2->Branch("MElectron.", "MElectron", &e4file, 32000);

    // ------------------------------

    TStopwatch clock;
    clock.Start();

    Int_t timer = 0;

    Int_t n=0;
    Double_t starttime = TStopwatch::GetRealTime(); // s
    while (TStopwatch::GetRealTime()<starttime+runtime)
        for (int i=0; i<nbins; i++)
    {
        n++;

        Double_t E = GetEnergy(nbins, lo, hi);
        Double_t weight = pow(E, alpha);
        histsrc.Fill(E, E*E * weight);

        MPhoton *gamma=new MPhoton(E, startz);
        gamma->SetIsPrimary();
        listg.Add(gamma);

        B->SetAddress(&gamma);
        T->Fill();

        cout << "--> " << n << ". " << Form("%d: %3.1e", (int)(starttime+runtime-TStopwatch::GetRealTime()), E) << flush;

        while (1)
        {
            if (listg.GetSize()!=0)
                cout << " |P:" << flush;

            TIter NextP(&listg);
            MPhoton *p = NULL;
            B->SetAddress(&p);
            while ((p=(MPhoton*)NextP()))
            {
                Double_t Eg = p->GetEnergy();

                if (!p->SetNewPosition())
                {
                    cout << "!" << flush;

                    hist.Fill(Eg, Eg*Eg*weight);
                    position.Fill(p->GetPhi()*kRad2Deg, p->GetR(), weight);
                    angle.Fill(p->GetPsi()*kRad2Deg, p->GetTheta()*kRad2Deg, weight);
                    histpos.Fill(p->GetR(), weight);
                    hista.Fill(p->GetTheta()*kRad2Deg, weight);

                    T->Fill();

                    delete listg.Remove(p);
                    continue;
                }

                //
                // Sample phtoton from background and interaction angle
                //
                TF1 phot("PhotonSpectrum", PhotonSpect, -log10(Eg)-8, -10.5, 2);
                phot.SetParameter(0, Eg);
                phot.SetParameter(1, p->GetZ());
                if (phot.Integral(-log10(Eg)-8, -10.5)==0)
                {
                    cout << "z" << flush;
                    continue;
                }

                Double_t Ep    = pow(10, phot.GetRandom());
                Double_t theta = RandomTheta(Eg, Ep);
                if (theta==0)
                {
                    cout << "t" << flush;
                    continue;
                }

                if (!pair.Process(p, Ep, theta, &liste))
                {
                    cout << "0" << flush;
                    continue;
                }

                delete listg.Remove(p);
                cout << "." << flush;
            }

            if (liste.GetSize()==0 && listg.GetSize()==0)
                break;

            if (liste.GetSize()!=0)
                cout << " |E" << flush;

            TIter Next(&liste);
            MElectron *e = NULL;
            B2->SetAddress(&e);
            while ((e=(MElectron*)Next()))
            {
                e->SetIsPrimary();
                T2->Fill();
                e->SetIsPrimary(kFALSE);

                if (e->GetEnergy()<lo)
                    continue;

                cout << ":" << flush;
                while(1)
                {
                    if (!e->SetNewPosition())
                    {
                        cout << "!" << flush;
                        break;
                    }

                    // WRONG!
                    Double_t theta = rand.Uniform(TMath::Pi()/2)+TMath::Pi()*3/4;
                    MPhoton *p = e->DoInvCompton(theta);

                    T2->Fill();

                    if (fabs(p->GetTheta()*3437)<1 &&  // < 1min
                        p->GetEnergy()>lo)
                    {
                        cout << "." << flush;
                        listg.Add(p);
                    }
                    else
                    {
                        cout << "i" << flush; // ignored
                        delete p;
                    }

                    if (fabs(e->GetTheta()*3437)<1 &&  // < 1min
                        e->GetEnergy()>lo          &&
                        e->GetEnergy()>3e4)
                        continue;

                    cout << "0" << flush;
                    break;
                }
            }
            liste.Delete();
        }
        cout << endl;

        Int_t now = (int)(TStopwatch::GetRealTime()- starttime)/5;
        if (now!=timer || n<10)
        {
            histsrc.SetTitle(Form("E^%.1f, z=%f, T=%d'%d\", N=%d", alpha, startz,
                                  (int)runtime/60, (int)runtime%60, hist.GetEntries()));
            gPad->Modified();
            gPad->Update();
            timer = now;
        }

    }
    cout << endl;

    clock.Stop();
    clock.Print();

    file.Write();

    cout << "Created " << n << " gammas from source with E^" << alpha << endl;
    cout << "Processed " << Form("%.1f", n/(TStopwatch::GetRealTime()-starttime)) << " gammas/sec." << endl;

    // ------------------------------

    gPad->Clear();

    hist.SetTitle(Form("E^%.1f, z=%f, T=%d'%d\", N=%d", alpha, startz, (int)runtime/60, (int)runtime%60, n));
    gPad->Modified();
    gPad->Update();

    hist.DrawCopy("P");
    histsrc.DrawCopy("Psame");
    hist.DrawCopy("Csame");
    histsrc.DrawCopy("Csame");

    gPad->SetGrid();
    gPad->SetLogx();
    gPad->SetLogy();

    MH::MakeDefCanvas();
    position.SetXTitle("Pos: \\Phi [\\circ]");
    position.SetYTitle("Pos: R [kpc]");
    position.DrawCopy("surf1pol");
    gPad->SetLogy();

    MH::MakeDefCanvas();
    angle.SetXTitle("Angle: \\Psi [\\circ]");
    angle.SetYTitle("Angle: \\Theta [\\circ]");
    angle.DrawCopy("surf1pol");
    gPad->SetLogy();

    MH::MakeDefCanvas();
    histpos.SetXTitle("R [kpc]");
    histpos.SetYTitle("Counts");
    histpos.DrawCopy();
    gPad->SetLogx();

    MH::MakeDefCanvas();
    hista.SetXTitle("\\Theta [\\circ]");
    hista.SetYTitle("Counts");
    hista.DrawCopy();
    gPad->SetLogx();
}

// -------------------------------------------------------------------
void phys()
{
    DoIt();

    /*
     Double_t E = 1e10;
     TF1 phot("PhotonSpectrum", PhotonSpect, -log10(E)-8, -10.5, 2);
     phot.SetParameter(0, E);
     phot.SetParameter(1, 5);
     phot.DrawCopy();
     return;
     */

    /*
     Double_t Eg = 1e5;

     Double_t E0    = 511e-6;                  // [GeV]
     Double_t lolim = E0*E0/Eg;
     Double_t inf   = 4e-12; //E0*E0/Eg * sqrt(Eg);

     // 1e5:   5e-13, 4e-12
     // 1e6:   5e-14, 2e-12
     // 1e8:   5e-16, 1e-12
     // 1e10:  5e-18, 1e-12

     // 1e5:   2.6e-12, 4e-12
     // 1e6:   2.6e-13, 2e-12
     // 1e8:   2.6e-15, 1e-12
     // 1e10:  1.6e-17, 1e-12

     TF1 f("int2", MPhoton::Int2, lolim, inf, 2);
     f.SetParameter(0, Eg);
     f.SetParameter(1, 0);

     MH::MakeDefCanvas();
     gPad->SetLogx();
     f.DrawCopy();
    */
}

