#include <iostream.h>

#include <fstream.h>

#include <TStopwatch.h>
#include <TF1.h>
#include <TH1.h>
#include <TH2.h>
#include <TList.h>
#include <TFile.h>
#include <TTree.h>
#include <TTimer.h>
#include <TStyle.h>
#include <TBranch.h>
#include <TRandom.h>
#include <TCanvas.h>

#include "mbase/MParContainer.h"
#include "mphys/MPhoton.h"
#include "mphys/MElectron.h"
#include "mphys/MPairProduction.h"
#include "mhist/MBinning.h"
#include "mhist/MH.h"

// 2.96
// 2.87
// 2.73
Double_t PrimSpect(Double_t *x, Double_t *k)
{
    return pow(pow(10, x[0]), k[0]);
}

Double_t PhotonSpect(Double_t *x, Double_t *k=NULL)
{
    Double_t Ep  = pow(10, x[0]);

    Double_t res = MPhoton::Int2(&Ep, k);
    return res*1e55; //65/k[0];

    //return MPhoton::Planck(&Ep, &k[1]);
}

Double_t Sbar_sigmas(Double_t *x, Double_t *k)
{
    Double_t sbar = pow(10, x[0]);

    Double_t s = 1./(sbar*4);

    Double_t sigma = MPhoton::Sigma_gg(&s);

    return sigma*sbar*1e28;
}

Double_t RandomTheta(Double_t Eg, Double_t Ep)
{
    Double_t E0 = 511e-6; // [GeV]

    Double_t f = Eg/E0*Ep/E0;

    if (f<1)
        return 0;

    TF1 func("RndTheta", Sbar_sigmas, 0, log10(f), 0);

    func.SetNpx(50);

    Double_t sbar  = pow(10, func.GetRandom());
    Double_t theta = acos(1.-sbar*2/f);

    return theta;
}

Double_t GetEnergy(Int_t n, Double_t lo, Double_t hi)
{
    static int bin=0;
    static TRandom rnd(0);

    Double_t w = log10(hi/lo)/n;

    Double_t E = lo*pow(10, rnd.Uniform(w) + w*(n-bin-1));

    //cout << endl << w << " " << n << " " << w*bin << " " << E << endl;

    if (++bin==n)
        bin=0;

    return E;

    /*
    // TF1 src ("PrimarySpectrum", PrimSpect, log10(lo), log10(hi), 1); // 10GeV-1000PeV
    // src.SetParameter(0, 0);
    // Double_t E = pow(10, src.GetRandom());

    //
    // 0: 11111111111111|22222222222222   2   2^0 + 1    1
    // 1: 11111111|22222222222|33333333   3   2^1 + 1    2
    // 2: 11111|22222|33333|44444|55555   5   2^2 + 1    7
    // 3:  |     | |   | |  |  |   |                    15
    //

    static int run=0;
    static int num=1;

    Double_t w = log10(hi/lo)/((1<<run) + 1);

    Double_t E = lo*pow(10, num*w);

    if (num == (1<<run))
    {
        run++;
        num=1;
    }
    else
        num++;

        return E;
        */
}

void DoIt()
{
    // a -->  5
    // b --> 10
    // c -->  2

    // delme, delmeB starting integration at lolim
    // delme2, delme2B starting integration at -log10(Eg)-8
    // delme3,  lolim, 24, 1e2-1e6,   2x inv.Compton
    // delme3B, lolim, 24, 1e2-1e6,   4x inv.Compton
    // delme3C, lolim, 24, 1e2-1e6,   8x inv.Compton
    // delme3D, lolim, 24, 1e2-1e6,  16x inv.Compton
    // delme3E, lolim, 24, 1e2-1e6,  32x inv.Compton

    // delme3H, lolim, 24, 1e2-1e6, 256x inv.Compton 8*60

    Double_t startz = 0.03; //MParticle::ZofR(&R);
    Double_t R      = MParticle::RofZ(&startz); // [kpc]

    const char *filename = "delme3H.root";

    const Double_t B = 0; // [T] mean magnetic field

    Double_t runtime = 8*60*60; // [s] maximum time to run the simulation

    Int_t nbins = 24;     // number of bins produced in energy spectrum

    Double_t lo = 1e2;    // lower limit of displayed spectrum
    Double_t hi = 1e6;    // upper limit of spectrum (cutoff)

    Int_t counter = 256;  // maximum number of inv. Compton (-1 means infinite)

    Double_t alpha = -1;  // -1 means a E^2 plot
    Double_t plot  =  1;  //  1 means a E^-2 spectrum

    // --------------------------------------------------------------------

    cout << "R = " << R << "kpc" << endl;
    cout << "Z = " << startz << endl;

    MPairProduction pair;

    TH1D hist;
    TH1D histsrc;

    hist.SetMinimum(pow(lo, plot+alpha)/100);
    histsrc.SetMinimum(pow(lo, plot+alpha)/100);

    TList listg;
    TList liste;

    listg.SetOwner();
    liste.SetOwner();

    gStyle->SetOptStat(10);

    //
    // Don't change the order!!!
    //
    hist.SetFillStyle(0);
    hist.SetMarkerStyle(kPlus);
    histsrc.SetFillStyle(0);
    histsrc.SetMarkerStyle(kMultiply);
    histsrc.SetMarkerColor(kRed);
    histsrc.SetLineColor(kRed);

    MBinning bins;
    bins.SetEdgesLog(nbins, lo, hi);

    MH::SetBinning(&hist,    &bins);
    MH::SetBinning(&histsrc, &bins);

    TCanvas *c=MH::MakeDefCanvas();

    gPad->SetGrid();
    gPad->SetLogx();
    gPad->SetLogy();

    hist.SetName("Spectrum");
    hist.SetXTitle("E [GeV]");
    hist.SetYTitle(Form("E^{%.1f} Counts", plot));
    hist.GetXaxis()->SetLabelOffset(-0.015);
    hist.GetXaxis()->SetTitleOffset(1.1);

    hist.Draw("P");
    histsrc.Draw("Psame");
    histsrc.Draw("same");
    hist.Draw("same");

    // ------------------------------

    void *ptr = NULL;
    TFile file(filename, "RECREATE", "Intergalactic cascade", 9);
    TTree   *T1 = new TTree ("Photons",    "Photons from Cascade");
    TTree   *T2 = new TTree ("Electrons",  "Electrons in the Cascade");
    TBranch *B1 = T1->Branch("MPhoton.",   "MPhoton",   &ptr);
    TBranch *B2 = T2->Branch("MElectron.", "MElectron", &ptr);

    // ------------------------------

    TTimer timer("gSystem->ProcessEvents();", 250, kFALSE);
    timer.TurnOn();

    TStopwatch clock;
    clock.Start();

    Int_t n=0;
    Double_t starttime = TStopwatch::GetRealTime(); // s
    while (TStopwatch::GetRealTime()<starttime+runtime)
        for (int i=0; i<nbins; i++)
    {
        n++;

        Double_t E = GetEnergy(nbins, lo, hi);
        Double_t weight = pow(E, alpha);
        histsrc.Fill(E, pow(E, plot) * weight);

        MPhoton *gamma=new MPhoton(E, startz);
        gamma->SetIsPrimary();
        gamma->InitRandom();
        listg.Add(gamma);

        B1->SetAddress(&gamma);
        T1->Fill();

        cout << "--> " << n << ". " << Form("%d: %3.1e", (int)(starttime+runtime-TStopwatch::GetRealTime()), E) << flush;

        Double_t Esum=0;
        Double_t Emis=0;
        while (1)
        {
            if (listg.GetSize()!=0)
                cout << " |P" << flush;

            TIter NextP(&listg);
            MPhoton *p = NULL;
            B1->SetAddress(&p);
            while ((p=(MPhoton*)NextP()))
            {
                cout << ":" << flush;
                Double_t Eg = p->GetEnergy();

                Double_t E0 = 511e-6;
                Double_t z  = p->GetZ();
                Double_t lolim = E0*E0/Eg;
                Double_t inf   = (Eg<1e6 ? 3e-11*(z+1) : 3e-12*(z+1));
                if (Eg<5e4)
                    inf = 3e-11*(z+1)*pow(10, 9.4-log10(Eg)*2);

                //TF1 phot("PhotonSpectrum", PhotonSpect, -log10(Eg)-8, -10.5, 2);
                //Double_t E0 = 511e-6; // [GeV]
                TF1 phot("PhotonSpectrum", PhotonSpect, log10(lolim), log10(inf), 2);
                phot.SetParameter(0, Eg);
                while (1)
                {
                    if (!p->SetNewPosition())
                    {
                        cout << "!" << flush;

                        hist.Fill(Eg, pow(Eg, plot)*weight);

                        p->SetIsPrimary(kFALSE);
                        T1->Fill();

                        Esum += Eg;

                        break;
                    }

                    //
                    // Sample phtoton from background and interaction angle
                    //
                    phot.SetParameter(1, p->GetZ());

                    Double_t Ep = pow(10, phot.GetRandom());
                    if (Ep==pow(10,0))
                    {
                        cout << "z" << flush;
                        continue;
                    }

                    Double_t theta = RandomTheta(Eg, Ep);
                    if (theta==0)
                    {
                        cout << "t" << flush;
                        continue;
                    }

                    if (!pair.Process(p, Ep, theta, &liste))
                    {
                        cout << "0" << flush;
                        continue;
                    }

                    cout << "." << flush;
                    break;
                }
                delete listg.Remove(p);
            }

            if (liste.GetSize()==0 && listg.GetSize()==0)
                break;

            if (liste.GetSize()!=0)
                cout << " |E" << flush;

            TIter Next(&liste);
            MElectron *e = NULL;
            B2->SetAddress(&e);
            while ((e=(MElectron*)Next()))
            {
                e->SetIsPrimary(kTRUE);
                T2->Fill();
                e->SetIsPrimary(kFALSE);

                Double_t Ee = e->GetEnergy();

                if (Ee<lo)
                    continue;

                cout << ":" << flush;
                int test = counter<0 ? -1 : counter;
                while (test<0 ? true : (test++<256))
                {

                    if (!e->SetNewPositionB(B))
                    {
                        cout << "!" << flush;
                        break;
                    }

                    // WRONG!
                    MPhoton *p = e->DoInvCompton(0);

                    T2->Fill();

                    cout << "." << flush;
                    listg.Add(p);
                    /*
                    if (fabs(p->GetTheta()*3437)<60 &&  // < 60min
                        p->GetEnergy()>lo)
                    {
                        cout << "." << flush;
                        listg.Add(p);
                    }
                    else
                    {
                        if (p->GetEnergy()<E*pow(10, alpha)/5 || p->GetEnergy()<=lo)
                            cout << "e" << flush;
                        else
                            cout << "t" << flush; // ignored
                        delete p;
                        Esum += p->GetEnergy();
                        Emis += p->GetEnergy();
                    }
                    */
                    if (fabs(e->GetTheta()*3437)>60)  // < 60min
                    {
                        cout << "T" << flush;
                        break;
                    }

                    if (e->GetEnergy()<Ee*1e-3) // <2e3
                    {
                        cout << "E" << flush;
                        break;
                    }
                }
                Esum += e->GetEnergy();
                Emis += e->GetEnergy();
            }
            liste.Delete();
        }
        cout << " ----> " << Form("%3.1f %3.1e / %3.1f %3.1e", Emis/E, Emis, Esum/E, Esum) << endl;

        hist.SetTitle(Form("E^{%.1f}  z=%f  T=%d'%d\"  N=%d", alpha, startz,
                              (int)runtime/60, (int)runtime%60, n));

        timer.Stop();
        c->Update();
        timer.Start(250);

    }
    cout << endl;

    clock.Stop();
    clock.Print();

    timer.Stop();

    file.Write();

    cout << "Created " << n << " gammas from source with E^" << alpha << endl;
    cout << "Processing time: " << Form("%.1f", (TStopwatch::GetRealTime()-starttime)/n) << " sec/gamma." << endl;

    // ------------------------------

    gPad->Clear();

    hist.SetTitle(Form("E^{%.1f}  z=%f  T=%d'%d\"  N=%d", alpha, startz, (int)runtime/60, (int)runtime%60, n));

    TH1 &h1 = *hist.DrawCopy("P");
    TH1 &h2 = *histsrc.DrawCopy("Psame");
    h2.Draw("Csame");
    h1.Draw("Csame");
}

// -------------------------------------------------------------------
void phys()
{
    DoIt();

    /*
     Double_t E = 1e10;
     TF1 phot("PhotonSpectrum", PhotonSpect, -log10(E)-8, -10.5, 2);
     phot.SetParameter(0, E);
     phot.SetParameter(1, 5);
     phot.DrawCopy();
     return;
     */

    /*
     Double_t Eg = 1e5;

     Double_t E0    = 511e-6;                  // [GeV]
     Double_t lolim = E0*E0/Eg;
     Double_t inf   = 4e-12; //E0*E0/Eg * sqrt(Eg);

     // 1e5:   5e-13, 4e-12
     // 1e6:   5e-14, 2e-12
     // 1e8:   5e-16, 1e-12
     // 1e10:  5e-18, 1e-12

     // 1e5:   2.6e-12, 4e-12
     // 1e6:   2.6e-13, 2e-12
     // 1e8:   2.6e-15, 1e-12
     // 1e10:  1.6e-17, 1e-12

     TF1 f("int2", MPhoton::Int2, lolim, inf, 2);
     f.SetParameter(0, Eg);
     f.SetParameter(1, 0);

     MH::MakeDefCanvas();
     gPad->SetLogx();
     f.DrawCopy();
    */
}

