/** @file fileopen.c 
 *  @short Allow searching of files in declared include paths (fopen replacement).
 *
 *  The functions provided in this file provide an enhanced replacement
 *  @c fileopen() for the C standard library's @c fopen() function.
 *  The enhancements are in several areas:
 *
 *  @li Where possible files are opened such that more than 2 gigabytes
 *      of data can be accessed on 32-bit systems when suitably compiled.
 *      This also works with software where a '-D_FILE_OFFSET_BITS=64' at
 *      compile-time cannot be used (of which ROOT is an infamous example).
 *  @li For reading files, a list of paths can be configured before the
 *      the first fileopen() call and all files without absolute paths
 *      will be searched in these paths. Writing always strictly
 *      follows the given file name and will not search in the path list.
 *  @li Files compressed with @c gzip or @c bzip2 can be handled on the
 *      fly. Files with corresponding file name extensions will be
 *      automatically decompressed when reading or compressed when
 *      writing (in a pipe, i.e. without producing temporary copies).
 *
 *  @author  Konrad Bernloehr 
 *  @date    Nov. 2000
 *  @date    CVS $Date: 2006/06/20 09:28:14 $
 *  @version CVS $Revision: 1.1 $
 */

#include "initial.h"
#include "straux.h"
#include "fileopen.h"
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

/** An element in a linked list of include paths. */
struct incpath
{
   char *path;       	   /**< The path name */
   struct incpath *next;   /**< The next element */
};

/** The starting element of include paths. */
static struct incpath *root_path = NULL;

static void freepath(void);

/** Init the path list, with default_path as the only entry. */

void initpath (const char *default_path)
{
   int ipos = 0;
   char pathname[1024];

   if ( default_path == NULL )
      default_path = ".";

   if ( root_path != NULL )
      freepath();
      
   while ( getword(default_path,&ipos,pathname,sizeof(pathname)-1,':','\n') > 0 )
      addpath(pathname);
}

/** Free a whole list of include path elements. */

static void freepath()
{
   struct incpath *path, *next = NULL;
   for ( path = root_path; path != NULL; path = next )
   {
      next = path->next;
      if ( path->path != NULL )
      	 free(path->path);
      free(path);
   }
   root_path = NULL;
}

/** Show the list of include paths. */

void listpath (char *buffer, int bufsize)
{
   struct incpath *path;
   if ( buffer == NULL || bufsize <= 0 )
      return;
   for ( path = root_path; path != NULL; path = path->next )
   {
      if ( path->path == NULL )
      	 continue;
      if ( bufsize > strlen(path->path)+3 )
      {
      	 strcpy(buffer," -I");
	 buffer += 3;
	 strcpy(buffer,path->path);
	 buffer += strlen(path->path);
	 bufsize -= strlen(path->path);
      }
   }
}

/** 
 *  @short Add a path to the list of include paths, if not already there.
 *  The path name is always copied to a newly allocated memroy location.
*/

void addpath (const char *name)
{
   struct incpath *path, *last = NULL;
   if ( name == NULL )
      return;
   if ( *name == '\0' )
      return;
   for ( path = root_path; path != NULL; path = path->next )
   {
      last = path;
      if ( strcmp(name,path->path) == 0 )
      	 return;
   }
   if ( last == NULL )
      path = root_path = calloc(1,sizeof(struct incpath));
   else
      path = last->next = calloc(1,sizeof(struct incpath));
   if ( path != NULL )
      path->path = strdup(name);
}

/** Helper function for opening a compressed file through a fifo. */

static FILE *cmp_popen (const char *fname, const char *mode, int compression)
{
   char *cmd = NULL; /* Fifo command without input/output file name */
   char *s;          /* Full fifo command with redirection */
   FILE *f = NULL;   /* File pointer returned */
   char *pmd = "?";  /* Fifo mode ("w" or "r") */

   int rc = 0;

   if ( fname == NULL || mode == NULL )
      return NULL;

   errno = 0;
   switch ( *mode )
   {
      case 'w':
         /* If we have no write access, we do not try to open the */
         /* fifo since it would result in a broken pipe later. */
         if ( (rc=access(fname,W_OK)) != 0 )
            if ( errno != ENOENT )
            {
               int k = errno;
               fprintf(stderr,"Cannot write to ");
               perror(fname);
               errno = k;
               return NULL;
            }
         pmd = "w";
         switch ( compression )
         {
            case 1:
               cmd = "gzip -c >";
               break;
            case 2:
               cmd = "bzip2 -c >";
               break;
         }
         break;
      case 'a':
         /* If we have no write access, we do not try to open the */
         /* fifo since it would result in a broken pipe later. */
         if ( (rc=access(fname,W_OK)) != 0 )
            if ( errno != ENOENT )
            {
               int k = errno;
               fprintf(stderr,"Cannot write to ");
               perror(fname);
               errno = k;
               return NULL;
            }
         pmd = "w";
         switch ( compression )
         {
            case 1:
               cmd = "gzip -c >>";
               break;
            case 2:
               cmd = "bzip2 -c >>";
               break;
         }
         break;
      case 'r':
         /* If we have no read access, we do not try to open the */
         /* fifo since it would result in a broken pipe later. */
         if ( (rc=access(fname,R_OK)) != 0 )
            return NULL;
         pmd = "r";
         switch ( compression )
         {
            case 1:
               cmd = "gzip -d -c <";
               break;
            case 2:
               cmd = "bzip2 -d -c <";
               break;
         }
         break;
   }

   if ( cmd == NULL )
   {
      errno = EPERM;
      return NULL;
   }

   s = malloc(strlen(cmd)+strlen(fname)+1);
   if ( s == NULL )
      return NULL;
   strcpy(s,cmd);
   strcat(s,fname);

   f = popen(s,pmd);
   free(s);

   return f;
}

/** Search for a file in the include path list and open it if possible. */

FILE *fileopen (const char *fname, const char *mode)
{
   struct incpath *path = root_path;
   char try_fname[1024];
   FILE *f = NULL;
   int nerr = 0;
   int compression = 0;
   int l;
   
   /* The set of search paths might not be initialized yet */
   if ( path == NULL )
   {
      initpath(NULL);
      if ( (path = root_path) == NULL )
      	 return NULL;
   }
   /* Arguments could be incomplete */
   if ( fname == NULL || mode == NULL )
      return NULL;

   /* Check if compressed files are meant. */
   l = strlen(fname);
   if ( l > 3 && strcmp(fname+l-3,".gz") == 0 )
      compression = 1;
   else if ( l > 4 && strcmp(fname+l-4,".bz2") == 0 )
      compression = 2;

   /* For modes other than read-only, no search is done. */
   if ( *mode != 'r' )
   {
      switch (compression)
      {
         case 0: /* Normal file */
#ifdef __USE_LARGEFILE64
            /* With explicit support of large files on 32-bit machines. */
            return fopen64(fname,mode);
#else
            /* Support for large files is either implicit or not available. */
            return fopen(fname,mode);
#endif
            break;

         case 1: /* Create FIFO to gzip to file */
         case 2: /* Create FIFO to bzip2 to file */
            return cmp_popen(fname,mode,compression);

         default:
            return NULL;
      }
   }            

   /* If the name includes (part of) a path, no search is done. */
   if ( strchr(fname,'/') != NULL )
   {
      switch (compression)
      {
         case 0: /* Normal file */
#ifdef __USE_LARGEFILE64
            return fopen64(fname,mode);
#else
            return fopen(fname,mode);
#endif
            break;

         case 1: /* Create FIFO from gunzip from file */
         case 2: /* Create FIFO from bunzip2 from file */
            return cmp_popen(fname,mode,compression);
            break;

         default:
            return NULL;
      }
   }            

   /* Now try all the paths we have */
   for ( ; path != NULL; path=path->next )
   {
#ifdef __USE_LARGEFILE64
      struct stat64 st;
#else
      struct stat st;
#endif
      if ( path->path == NULL )
      	 continue;
      if ( strlen(path->path)+strlen(fname)+1 >= sizeof(try_fname) )
      	 continue;
      if ( strcmp(path->path,".") == 0 )
      	 strcpy(try_fname,fname); /* No need to expand local path */
      else
      	 sprintf(try_fname,"%s/%s",path->path,fname);

      errno = 0;
      /* Check if the file is available */
#ifdef __USE_LARGEFILE64
      if ( stat64(try_fname,&st) != 0 )
#else
      if ( stat(try_fname,&st) != 0 )
#endif
      {
         if ( errno != 0 && errno != ENOENT )
         {
      	    perror(try_fname);
	    nerr++;
         }
         continue;
      }

      errno = 0;
      /* The file exists and is hopefully readable */
      switch (compression)
      {
         case 0: /* Normal file */
#ifdef __USE_LARGEFILE64
            if ( (f = fopen64(try_fname,mode)) != NULL )
#else
            if ( (f = fopen(try_fname,mode)) != NULL )
#endif
      	       return f;
            break;

         case 1: /* Create FIFO from gunzip from file */
         case 2: /* Create FIFO from bunzip2 from file */
            if ( (f = cmp_popen(try_fname,"r",compression)) != NULL )
               return f;
            break;

         default:
            return NULL;
      }

      if ( errno != 0 && errno != ENOENT )
      {
      	 perror(try_fname);
	 nerr++;
      }
   }

   if ( f != NULL )
      return f;

   if ( f == NULL && nerr == 0 )
   {
      fprintf(stderr,"%s: Not found in any include directory.\n",fname);
      if ( errno == 0 )
      	 errno = ENOENT;
   }

   return NULL;
}

/** Close a file or fifo but not if it is one of the standard streams. */

int fileclose (FILE *f)
{
#ifdef __USE_LARGEFILE64
   struct stat64 st;
#else
   struct stat st;
#endif

   if ( f == NULL )
      return -1;

   errno = 0;

   /* We never close the standard streams here. */
   if ( f == stdin || f == stdout || f == stderr )
      return 0;

      /* Check what kind of stream we have */
   if ( fileno(f) == -1 )
   {
      fprintf(stderr,"Trying to close stream: no file handle\n");
      return -1;
   }

#ifdef __USE_LARGEFILE64
   if ( fstat64(fileno(f),&st) != 0 )
#else
   if ( fstat(fileno(f),&st) != 0 )
#endif
   {
      if ( errno != 0 )
         perror("Trying to close stream");
      return -1;
   }

#ifdef S_ISFIFO
   if ( S_ISFIFO(st.st_mode) )
#else
   if ( (st.st_mode & S_IFMT) == S_IFIFO )
#endif
      return pclose(f);
   else
      return fclose(f);
}
