
/********************************************************************\

  ProcessIO.cc

  Main class processing user input
 
  Sebastian Commichau, Sabrina Stark-Schneebeli, Oliver Grimm

\********************************************************************/

#include "ProcessIO.h"

static char* state_str[]    = {"active", "stopped", "n.a."};


ProcessIO::ProcessIO(const char *ConfigFile) {

  // Get program start time
  time (&StartTime);

  // Create instances
  config = new HVConfig(ConfigFile);
  calib  = new HVCalib(config);
  pm 	 = new PixelMap(config->fPixMapTable);

  // Initialize status variables
  state     = active;
  Exit      = false;
  Verbose   = false;
  CmdFromSocket = false;
  Socket    = -1;
  
  NumHVBoards = 0;
  FirstBoard  = 0;
  LastBoard   = -1;
  FirstChain  = 0;
  LastChain   = NUM_CHAINS-1;
  
  if (config->fStatusRefreshRate >= MIN_RATE && config->fStatusRefreshRate <= MAX_RATE) 
    fStatusRefreshRate = config->fStatusRefreshRate;
  else fStatusRefreshRate = 1.;
    
  // Open HV devices
  fHVBoard = new HVBoard* [config->NumHVBoards]; 
  for (int i=0; i<config->NumHVBoards; i++) {
    fHVBoard[NumHVBoards] = new HVBoard(i, config->fUSBDevice[i], this);
    if(fHVBoard[NumHVBoards]->fDescriptor >= 0) {
       printf("Synchronized and reset HV board %d (%s)\n",i,config->fUSBDevice[i]);
       NumHVBoards++;
    }
    else {
      printf("Failed to synchronize to HV board %d (%s)\n",i,config->fUSBDevice[i]);
      delete fHVBoard[NumHVBoards];
    }
  } 
  LastBoard = NumHVBoards-1;

  // Open log file
  if ((Logfile = fopen(config->fLogFile, "a")) == NULL) printf("Warning: Could not open log file '%s'\n", config->fLogFile);
  PrintMessage(MsgToLog,"********** Logging started **********\n");

  // Create instance of slow data class
  SlowDataClass = new SlowData("HV", config->fSlowDir);
  if (SlowDataClass->ErrorCode != 0) {
    PrintMessage("Warning: Could not open slowdata file (%s)\n", strerror(SlowDataClass->ErrorCode));
  }
  SlowDataClass->NewEntry("Value-Info", "Issued if new HV value set successfull: Board-Num HV-Board-Name Chain Channel DAC-Target Converted-Value");
  SlowDataClass->NewEntry("Error-Info", "Issued if error occurs when trying to set new HV value: Board-Num HV-Board-Name Chain Channel Attempted-DAC-Target Converted-Value");
}


ProcessIO::~ProcessIO() {
  
  delete SlowDataClass;     

  for (int i=0; i<NumHVBoards; i++) delete fHVBoard[i];
  delete[] fHVBoard;
    
  delete pm;   delete calib;    delete config;    
  
  if(Logfile != NULL) {
    if(fclose(Logfile) != 0) perror("Error closing logfile");
  }
}


// Process user input
void ProcessIO::CommandControl(char *Command) {

  if (strlen(Command)==0) return;  // Ignore empty commands

  if(Command[0]=='.') {   // Shell command
    system(&(Command[1]));
    return;
  }

  for(int i=0; i<MAX_NUM_TOKEN; i++) Param[i] = "";  // All pointers point initially to empty string
    NParam = ParseInput(Command, Param);

  // Adress HV board
  if (Match(Param[0], "board")) {
    
    if (!NumHVBoards) return;

    // Print list of boards
    if (NParam == 1) {
      for (int i=0; i<NumHVBoards; i++) {
        PrintMessage("Board %d: %s\n", fHVBoard[i]->GetBoardNumber(), fHVBoard[i]->BoardName);
      }
      return;
    }

    //Select board(s)
    if (Match(Param[1],"all")) {
      FirstBoard = 0;
      LastBoard = NumHVBoards-1;
    } 
    else if (NParam==2 && atoi(Param[1])>=0 && atoi(Param[1])<NumHVBoards) {
      FirstBoard = atoi(Param[1]);
      LastBoard = FirstBoard;
    }
    else if (NParam==3 && atoi(Param[1])>=0 && atoi(Param[1])<NumHVBoards && 
           atoi(Param[2])>0 && atoi(Param[2])<NumHVBoards) {
      FirstBoard = atoi(Param[1]);
      LastBoard = atoi(Param[2]);
    }
    else PrintMessage("Cannot address board(s), out of range.\n");

    return;
  } 


  // Adress chains
  else if (Match(Param[0], "chain")) {
    
    if (!NumHVBoards) return;
    
    if (Match(Param[1],"all")) {
      FirstChain = 0;
      LastChain = 3;
    } 
    else if (NParam==2 && atoi(Param[1])>=0 && atoi(Param[1])<NUM_CHAINS) {
      FirstChain = atoi(Param[1]);
      LastChain = FirstChain;
    }
    else if (NParam==3 && atoi(Param[1])>=0 && atoi(Param[1])<NUM_CHAINS && 
           atoi(Param[2])>0 && atoi(Param[2])<NUM_CHAINS) {
      FirstChain = atoi(Param[1]);
      LastChain = atoi(Param[2]);
    }
    else PrintMessage("Cannot address chain(s), out of range.\n");
        
    return;
  } 


  // Print HV utility configuration
  else if (Match(Param[0], "config")) {
    PrintMessage( " Log file:          %s\n"
    	    	  " Pixel map table:   %s\n"
    	    	  " %d USB devices:\n", config->fLogFile, config->fPixMapTable,
		  config->NumHVBoards);  
    for (int i=0; i<config->NumHVBoards; i++) { 
      PrintMessage(" Board %d: %s\n ", i, config->fUSBDevice[i]);
    }
    PrintMessage( " TimeOut:           %.2f s\n"
    	    	  " StatusRefreshRate: %.2f Hz\n"
    	    	  " CCPort:            %d\n"
    	    	  " DACMin value:      %d\n"
    	    	  " DACMax value:      %d\n"
    	    	  " HVCalibOffset :    %f\n"
    	    	  " HVCalibSlope :     %f\n"
    	    	  " HVMaxDiff :        %u\n", config->fTimeOut,
		  config->fStatusRefreshRate, config->fCCPort, config->DACMin,
		  config->DACMax, config->fHVCalibOffset, config->fHVCalibSlope,
		  config->fHVMaxDiff);

    return;
  }


  // Print help
  if (Match(Param[0], "help")) {
    puts(" board <i>|<i> <j>|<all>           Address board i, boards i-j or all boards or list boards");
    puts(" chain <i>|<i> <j>|<all>           Address chain i, chains i-j or all chains");
    puts(" hv <PXL id>|<ch>|<all> <v>        Set HV of pixel ID, ch. or all ch. of active chain(s)/board(s)");
    puts(" hvdiff <PXL id>|<ch>|<all> <diff> Change HV by <diff>");
    puts(" status [dac]                      Show status information (DAC values if requested)");
    puts(" config                            Print configuration");
    puts(" load <file>                       Load HV settings from <file>");
    puts(" save <file>                       Save current HV settings to [file]");
    puts(" exit                              Exit program");
    puts(" rate <rate>                       Set status refresh rate to <rate> [Hz]");
    puts(" timeout <time>                    Set timeout to return from read to <time> [s]");
    puts(" reset                             Reset active HV board");
    puts(" start                             Start HV status monitor");
    puts(" stop                              Stop HV status monitor - not recommended!");
    puts(" uptime                            Get program uptime [h:m:s]");
    puts(" verbose <on|off>|                 Enable|disable verbosity");
    puts(" help                              Print help");
    puts(" .<cmd>                   	     Execute shell command <cmd>");

    return;
  } 


  // Set new high voltage (if no boards available, simply returns OK)
  // First reponse to socket should be 'OK' if no error occurred.
  if (Match(Param[0], "hv") || Match(Param[0], "hvdiff")) {

    int hvoltage, DACValue, Errors=0, Board=-1, Chain=-1, Channel=-1;
    double hvoltageV;
    bool SetDac;
    
    // Need two or parameters
    if (NParam<3 || NParam>4) {
      PrintMessage("Usage: hv/hvdiff <channel>|<all> <voltage> [dac]\n");
      return;
    }

    // Evaluate voltage parameter
    if (NParam==4 && Match(Param[3], "dac")) {
      SetDac = true;
      if (!ConvertToInt(Param[2], &hvoltage)) {
        PrintMessage("Error: Wrong number format for DAC voltage setting\n");
        return;
      } 
    }
    else {
      SetDac = false;
      if (!ConvertToDouble(Param[2], &hvoltageV)) {
        PrintMessage("Error: Wrong number format for voltage setting\n");
        return;
      }
    } 

    // Evaluate pixel or channel parameter    
    if(pm->Pixel_to_HVboard(Param[1]) != 999999999) {
      Board = pm->Pixel_to_HVboard(Param[1]);
      Chain = pm->Pixel_to_HVchain(Param[1]);
      Channel = pm->Pixel_to_HVchannel(Param[1]);
    }
    else if (!Match(Param[1], "all") && !ConvertToInt(Param[1], &Channel)) {
      PrintMessage("Error: Wrong channel identification\n");
      return;
    }

    for (int i=FirstBoard; i<=LastBoard; i++) {
      if (i!=Board && Board!=-1) continue;

      for (int j=FirstChain; j<=LastChain; j++) {
        if (j!=Chain && Chain!=-1) continue;

	for (int k=0; k<NUM_CHANNELS; k++) {
	  if (k!=Channel && Channel!=-1) continue;

    	  // Convert from HV to DAC values
	  if (!SetDac){
	    if(strlen(Param[0]) > 2) fHVBoard[i]->HVV[j][k] += hvoltageV; // hvdiff
	    else fHVBoard[i]->HVV[j][k] = hvoltageV;
	    DACValue = calib->HVToDAC(fHVBoard[i]->HVV[j][k], i, j, k);
	  }
	  else {
	    if(strlen(Param[0]) > 2) DACValue = fHVBoard[i]->HV[j][k] + hvoltage; // hvdiff
	    else DACValue = hvoltage;
	  }
	  
	  // Set new voltage (if DAC value, update calibrated value)
	  if(!RampVoltage(DACValue, i, j, k)) Errors++;
	  if(SetDac) fHVBoard[i]->HVV[j][k] = calib->DACToHV(fHVBoard[i]->HV[j][k], i, j, k);

	} // Channels	
      } // Chains
    } // Boards

    if (Errors > 0) PrintMessage("ERROR - Errors on %d channel(s) occurred\n", Errors);
    else PrintMessage("OK - no errors\n");
    
    return;
  }


  // Load HV settings from file
  else if (Match(Param[0], "load")) {

    char Buffer[MAX_COM_SIZE];
    int NBoards = 0, Errors = 0, Chain, Channel;
    unsigned int DACValue;
    FILE *File;
    
    if (NParam != 2) {
      PrintMessage("Usage: load <file>\n");
      return;
    }
    
    if ((File=fopen(Param[1], "r")) == NULL) {
      PrintMessage("Error: Could not open file '%s' (%s)\n", Param[1], strerror(errno));
      return;
    }
	
    while (fgets(Buffer, sizeof(Buffer), File) != NULL) {
      for (int Board=0; Board<NumHVBoards; Board++) {
	if (Match(fHVBoard[Board]->BoardName, Buffer)) {
	  PrintMessage("Found HV settings for board %d (%s)\n\r",fHVBoard[Board]->GetBoardNumber(), fHVBoard[Board]->BoardName);

	  Chain = 0;  Channel = 0;
	  while (fscanf(File, "%u", &DACValue)==1 && Chain<NUM_CHAINS) {
    	    if (!RampVoltage(DACValue, Board, Chain, Channel)) {
	      Errors++;
	      PrintMessage("Error: Could not ramp chain %d, channel %d\n", Chain, Channel);
    	    }
	    else {
	      PrintMessage("Ramped chain %d, channel %d to %u (%.2f V)                         \r",
	      	 Chain,Channel,DACValue,calib->DACToHV(DACValue,Board,Chain,Channel));
    	    }
	    fHVBoard[Board]->HVV[Chain][Channel] = calib->DACToHV(fHVBoard[Board]->HV[Chain][Channel], Board, Chain, Channel);
	    if(++Channel == NUM_CHANNELS) {
	      Chain++;
	      Channel = 0;
	    }
	  }
	  if (ferror(File) != 0) {
	    PrintMessage("Error reading DAC value from file, terminating. (%s)\n",strerror(errno));
    	    return;
	  }
	  else PrintMessage("\nFinished updating board\n");
    	  NBoards++;
	}
      } // Loop over boards
    } // while()
    	    
    if (NBoards != NumHVBoards) {
      PrintMessage("Warning: Could not load HV settings for all connected HV boards\n");
    }
    else if (Errors == 0) PrintMessage("Success: Read HV settings for all connected HV boards\n");
    if (Errors != 0) PrintMessage("Warning: Errors on %d channel(s) occurred\n", Errors);
    
    if (fclose(File) != 0) PrintMessage("Error: Could not close file '%s'\n",Param[1]);

    return;    
  }
	   

  // Set status refresh rate
  if (Match(Param[0], "rate")) {

    double Rate;
    
    if (NParam != 2) {
      PrintMessage("Usage: rate <Hz>\n");
      return;
    }
    
    if (!ConvertToDouble(Param[1], &Rate)) {
       PrintMessage("Error: Wrong number format\n");
       return;   
    }

    // Check limits
    if (Rate<MIN_RATE || Rate>MAX_RATE) {
      PrintMessage("Refresh rate out of range (min: %.2f Hz, max: %.2f Hz)\n", MIN_RATE, MAX_RATE);
      return;
    }

    fStatusRefreshRate = Rate;
    PrintMessage("Refresh rate set to %.2f Hz\n", fStatusRefreshRate);

    return;
  }
  
  // Reset
  if (Match(Param[0], "reset")) {

    if (!NumHVBoards) return;
    for (int i=FirstBoard; i<=LastBoard; i++) ResetBoard(i);
    return;
  }


  // Save HV settings of all boards
  else if (Match(Param[0], "save")) {

    FILE *File;
    time_t time_now_secs;
    struct tm *Time;

    if (NParam != 2) {
      PrintMessage("Usage: save <Filename>\n");
      return;
    }
  
    time(&time_now_secs);
    Time = localtime(&time_now_secs);

    if ((File = fopen(Param[1], "w")) == NULL) {
      PrintMessage("Error: Could not open file '%s' (%s)\n", Param[1], strerror(errno));
      return;
    }
  
    fprintf(File,"********** HV settings, %04d %02d %02d, %02d:%02d:%02d **********\n\n",
	    1900 + Time->tm_year, 1 + Time->tm_mon,
	    Time->tm_mday, Time->tm_hour, Time->tm_min, Time->tm_sec);
  
    for (int i=0; i<NumHVBoards; i++) {
      fprintf(File, "%s\n\n", fHVBoard[i]->BoardName);

      for (int j=0; j<NUM_CHAINS; j++) {
	for (int k=0; k<NUM_CHANNELS; k++) fprintf(File,"%5d ",fHVBoard[i]->HV[j][k]);
	fprintf(File, "\n");
      }
      fprintf(File, "\n");
    }

    if (fclose(File) != 0) {
      PrintMessage("Error: Could not close file '%s' (%s)\n", Param[1], strerror(errno));  
    }

    return;
  }


  // Start monitoring
  else if (Match(Param[0], "start")) {

    state = active;
    pthread_kill(HVMonitor, SIGUSR1);
    PrintMessage("Status monitoring activated\n");
    return;  
  }

 
  // Print status
  else if (Match(Param[0], "status")) {
    
    PrintMessage("\n Status monitor: %s\n", state_str[state]);
    PrintMessage(" Verbose: %s\n", Verbose ? "on" : "off");
    PrintMessage(" Status refresh rate [Hz]: %.2f\n", fStatusRefreshRate);
    PrintMessage(" Socket state: %s\n", Socket==-1 ? "disconnected":"connected");
    PrintMessage(" Total number of HV boards: %d\n", NumHVBoards);
    PrintMessage(" Active HV boards: %d\n\n ", LastBoard - FirstBoard + 1);

    for (int i=FirstBoard; i<=LastBoard; i++) {
      PrintMessage(" BOARD %d (%s)   Wrap counter: %s (%d)  Manual reset: %s  Time-out: %.2f s\n\n",
	fHVBoard[i]->GetBoardNumber(), fHVBoard[i]->BoardName,
	fHVBoard[i]->WrapOK ? "ok":"error",
	fHVBoard[i]->LastWrapCount, 
	fHVBoard[i]->ResetButton ? "yes" : "no",
	fHVBoard[i]->fTimeOut);

      for (int j=FirstChain; j<=LastChain; j++) {
	PrintMessage("  CHAIN %d     Over-current: %s\n ", j, fHVBoard[i]->Overcurrent[j] ? "yes" : "no");
	for (int k=0;k<4;k++) {
	  for (int l=0;l<8;l++) {
	    if(NParam == 2) PrintMessage("%5d ",fHVBoard[i]->HV[j][k*8+l]);
    	    else PrintMessage("%#5.2f ",fHVBoard[i]->HVV[j][k*8+l]);
    	  }
	  PrintMessage("\n ");
	}
	  PrintMessage("\n ");
      }
    }

    return;
  } 


  // Stop monitoring
  else if (Match(Param[0], "stop")) {

    state = stopped;
    pthread_kill(HVMonitor, SIGUSR1);
    PrintMessage("Status monitor stopped\n");

    return;
  }


  // Set timeout to return from read
  if (Match(Param[0], "timeout")) {

    double Timeout;
    
    if (!NumHVBoards) return;

    if (NParam != 2) {
      PrintMessage("Usage: timeout <secs>\n");
      return;
    }
    
    if (!ConvertToDouble(Param[1], &Timeout)) {
       PrintMessage("Error: Wrong number format\n");
       return;   
    }

    for (int i=0; i<NumHVBoards; i++) fHVBoard[i]->SetTimeOut(Timeout);
    PrintMessage("Timeout set to %.2f s for all boards\n", Timeout);
    
    return;
  }

    
  // Print uptime
  if (Match(Param[0], "uptime")) {
    time_t ActualT;
    time (&ActualT);

    PrintMessage("%d:%02d:%02d\n", (int) difftime(ActualT, StartTime)/3600, ((int) difftime(ActualT, StartTime)/60)%60, (int) difftime(ActualT, StartTime)%60);

    return;
  } 


  // Enable/disable verbosity
  else if (Match(Param[0], "verbose")) {

    if (Match(Param[1], "on")) {
      Verbose = true;
      PrintMessage("Verbosity enabled\n");
    }    
    else if (Match(Param[1], "off")) {
      Verbose = false;
      PrintMessage("Verbosity disabled\n");
    }
    else PrintMessage("Usage: verbose <on>|<off>\n");
  
    return;
  }
  
  
  // Exit program
  else if(Match(Param[0], "exit")) {

    if (CmdFromSocket) {
      PrintMessage("Exit command not allowed over socket.\n");
      return;
    }  	

    Exit = true;
    pthread_kill(HVMonitor, SIGUSR1);
    pthread_kill(SocketThread, SIGUSR1);
 
    return;
  }
  
  PrintMessage("Unknown command '%s'\n", Param[0]);
 
  return;
}


// Print message to selected target
void ProcessIO::PrintMessage(int Target, const char *Format, ...) {
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format); 
  PrintMessage(Target, Format, ArgumentPointer);
  va_end(ArgumentPointer);
}

// Print message to log file, and screen or socket (depending on command origin)
void ProcessIO::PrintMessage(const char *Format, ...) {
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format);
  if (CmdFromSocket) PrintMessage(MsgToSocket|MsgToLog, Format, ArgumentPointer);
  else PrintMessage(MsgToConsole|MsgToLog, Format, ArgumentPointer);
  va_end(ArgumentPointer);
}

// Function doing the actual printing work
void ProcessIO::PrintMessage(int Target, const char *Format, va_list ArgumentPointer) {

  char Textbuffer[MAX_COM_SIZE];

  memset(Textbuffer, 0, sizeof(Textbuffer));  
  vsnprintf(Textbuffer, sizeof(Textbuffer), Format, ArgumentPointer);
  
  // Print to console
  if(Target & MsgToConsole) {
    if(strlen(Textbuffer)>0 && Textbuffer[strlen(Textbuffer)-1]=='\n') {
      printf("\r%s%s", Textbuffer, Prompt);   // New prompt
    }
    else printf("%s", Textbuffer);
    fflush(stdout);
  }
  // Print to socket
  if((Target & MsgToSocket) && Socket!=-1) {
    write(Socket, Textbuffer, strlen(Textbuffer));
  }
  // Print to log file (repace carriage return by linefeed)
  if((Target & MsgToLog) && Logfile!=NULL) {
    for (unsigned int i=0; i<strlen(Textbuffer); i++) {
      if(Textbuffer[i] == '\r') Textbuffer[i] = '\n';
    }
    fprintf(Logfile, "%s", Textbuffer);
    fflush(Logfile);
  }
}


// Ramp to new voltage with maximum step size given in config->fHVMaxDiff
// No ramping when decreasing voltage
bool ProcessIO::RampVoltage(unsigned int Target, int Board, int Chain, int Channel) {

  while (fHVBoard[Board]->HV[Chain][Channel] != (int) Target) {	  
    int Diff = Target - fHVBoard[Board]->HV[Chain][Channel];
    if (Diff > (int) config->fHVMaxDiff) Diff = config->fHVMaxDiff;

    if (fHVBoard[Board]->SetHV(Chain, Channel, fHVBoard[Board]->HV[Chain][Channel]+Diff) == 1) {
      if (Verbose) {
	PrintMessage("Board %d: high voltage of chain %d channel %d set to %d | 0X%.4X | %f V\n",fHVBoard[Board]->GetBoardNumber(),Chain, Channel, Target, Target, calib->DACToHV(Target,fHVBoard[Board]->GetBoardNumber(),Chain,Channel));
	PrintBoardStatus(Board);
      }
    }
    else {
      PrintMessage("Error: Could not set HV of board %d, chain %d, channel %d. Skipping channel\n",fHVBoard[Board]->GetBoardNumber(),Chain,Channel);
      //SlowDataClass->NewEntry("Error");
      //SlowDataClass->AddToEntry("%s %d %d %d %d %.2f ",fHVBoard[Board]->BoardName,Board, Chain, Channel, Target, calib->DACToHV(Target,Board,Chain,Channel));
      return false;
    }
  }
  //SlowDataClass->NewEntry("Value");
  //SlowDataClass->AddToEntry("%s %d %d %d %d %.2f ",fHVBoard[Board]->BoardName,Board, Chain, Channel, Target, calib->DACToHV(Target,Board,Chain,Channel));
  return true;
}


// Check board status
void ProcessIO::Monitor() {

  for (int i=0; i<NumHVBoards; i++) {

    if (fHVBoard[i]->GetStatus() != 1) {
      PrintMessage("Error: Monitor, could not read status of board %d\n", fHVBoard[i]->GetBoardNumber());
    }
    
    if (fHVBoard[i]->ResetButton) {
      PrintMessage("Manual reset of board %d\n",fHVBoard[i]->GetBoardNumber());
      ResetBoard(i);
    }
    
    if (!fHVBoard[i]->WrapOK) {
      PrintMessage("Error: Wrap counter mismatch board %d\n",fHVBoard[i]->GetBoardNumber());
    }

    for (int j=0; j<NUM_CHAINS; j++) {
      if (fHVBoard[i]->Overcurrent[j]) {
	PrintMessage("Warning: Overcurrent in chain %d of board %d\n",j,fHVBoard[i]->GetBoardNumber());
	ResetBoard(i);
      }
    }
    
  }
}


// Send reset to board and clear voltage arrays
void ProcessIO::ResetBoard(int i) {
    
  if (fHVBoard[i]->Reset() == 1) {
    PrintMessage("Reset of board %d\n", fHVBoard[i]->GetBoardNumber());
    PrintBoardStatus(i);
  }
  else PrintMessage("Error: Could not reset board %d\n",fHVBoard[i]->GetBoardNumber());
}


// Print current board status
void ProcessIO::PrintBoardStatus(int i) {

  PrintMessage("Status board %d (%s): MR %s OC0 %s OC1 %s OC2 %s OC3 %s WC %s (%d)\n",
    	  fHVBoard[i]->GetBoardNumber(), fHVBoard[i]->BoardName,
	  fHVBoard[i]->ResetButton ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[0] ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[1] ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[2] ? "yes" : "no",
	  fHVBoard[i]->Overcurrent[3] ? "yes" : "no",
	  fHVBoard[i]->WrapOK ? "ok":"error", fHVBoard[i]->LastWrapCount);
}


// Parse command line for white space and double-quote separated tokens 
int ProcessIO::ParseInput(char* Command, const char *Param[]) {
  int Count=0;
   
  while(Count<MAX_NUM_TOKEN) {
    while (isspace(*Command)) Command++; // Ignore initial white spaces
    if(*Command=='\0') break;
    if (*Command == '\"') {
      Param[Count] = ++Command;
      while(*Command!='\"' && *Command!='\0') Command++;
    }
    else {
      Param[Count] = Command;
      while(!isspace(*Command) && *Command!='\0') Command++;
    }
    if(*Command != '\0') *Command++ = '\0';
    Count++;
  }
  return Count;
}


// Check if two strings match (min 1 character must match)
bool Match(const char *str, const char *cmd) {
  return strncasecmp(str,cmd,strlen(str)==0 ? 1:strlen(str)) ? false:true;
}


// Convert string to double
// Returns false if conversion did not stop on whitespace or EOL character
bool ConvertToDouble(const char *String, double *Result) {

  char *EndPointer;
  
  *Result = strtod(String, &EndPointer);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}


// Convert string to int
// Returns false if conversion did not stop on whitespace or EOL character
bool ConvertToInt(const char *String, int *Result) {

  char *EndPointer;
  
  *Result = (int) strtol(String, &EndPointer, 0);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}
