----------------------------------------------------------------------------------
-- Company:        ETH Zurich, Institute for Particle Physics
-- Engineer:       P. Vogler, Q. Weitzel
-- 
-- Create Date:    13 October 2010
-- Design Name:    
-- Module Name:    FTU_test2 - Behavioral 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description:    Test firmware for FTM board:
--                      NIM and LVDS output stages
--                      fast signal distribution (partially)
--                      crate resets
--                      
--
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
--
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
library UNISIM;
use UNISIM.VComponents.all;

entity FTM_test2 is
  port(

    
-- Clock
   clk   : IN  STD_LOGIC;                     -- external clock from
                                              -- oscillator U47

-- connection to the WIZnet W5300 ethernet controller
-- on IO-Bank 1
-------------------------------------------------------------------------------
    -- W5300 data bus
--    W_D  : inout STD_LOGIC_VECTOR(15 downto 0);  -- 16-bit data bus to W5300	


    -- W5300 address bus
--    W_A  : out STD_LOGIC_VECTOR(9 downto 1);   -- there is NO net W_A0 because
                                               -- the W5300 is operated in the 
                                               -- 16-bit mode 

    -- W5300 controll signals
    -- the signals W_INT, W_RD, W_WR and W_RES also go to testpoints T17
    -- W_CS is also routed to testpoint JP7
--    W_CS   : out  STD_LOGIC;                      --  W5300 chip select
--    W_INT  : IN  STD_LOGIC;                       -- interrupt
--    W_RD   : out  STD_LOGIC;                      -- read
--    W_WR   : out  STD_LOGIC;                      -- write
--    W_RES  : out  STD_LOGIC;                      -- reset W5300 chip

    -- W5300 buffer ready indicator
--    W_BRDY :  in STD_LOGIC_VECTOR(3 downto 0); 

    -- testpoints (T18) associated with the W5300 on IO-bank 1
--    W_T    : inout STD_LOGIC_VECTOR(3 downto 0);  
 


-- SPI Interface
-- connection to the EEPROM U36 (AL25L016M) and 
-- temperature sensors U45, U46, U48 and U49 (all MAX6662)
-- on IO-Bank 1
-------------------------------------------------------------------------------
--   S_CLK  : out  STD_LOGIC;     -- SPI clock

   -- EEPROM
--   MOSI   : out  STD_LOGIC;     -- master out slave in
--   MISO   : in   STD_LOGIC;     -- master in slave out
--   EE_CS  : out  STD_LOGIC;     -- EEPROM chip select

   -- temperature sensors U45, U46, U48 and U49
--   SIO    : inout  STD_LOGIC;          -- serial IO
--   TS_CS  : out STD_LOGIC_VECTOR(3 downto 0);     -- temperature sensors chip select

 

-- Trigger primitives inputs
-- on IO-Bank 2
-------------------------------------------------------------------------------
--   Trig_Prim_A  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 0
--   Trig_Prim_B  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 1
--   Trig_Prim_C  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 2
--   Trig_Prim_D  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 3

  

-- NIM inputs
------------------------------------------------------------------------------
  -- on IO-Bank 3  
--   ext_Trig  : in  STD_LOGIC_VECTOR(2 downto 1);      -- external trigger input
--   Veto       : in  STD_LOGIC;                         -- trigger veto input
--   NIM_In     : in  STD_LOGIC_VECTOR(2 downto 0);      -- auxiliary inputs

   -- on IO-Bank 0
--   NIM_In3_GCLK  : in  STD_LOGIC;      -- input with global clock buffer available 

   

-- LEDs on IO-Banks 0 and 3
-------------------------------------------------------------------------------
--   LED_red  : out STD_LOGIC_VECTOR(3 downto 0);    -- red
--   LED_ye   : out STD_LOGIC_VECTOR(1 downto 0);    -- yellow
--   LED_gn   : out STD_LOGIC_VECTOR(1 downto 0);    -- green

   
   
-- Clock conditioner LMK03000
-- on IO-Bank 3
-------------------------------------------------------------------------------
--   CLK_Clk_Cond  : out STD_LOGIC;  -- clock conditioner MICROWIRE interface clock
--   LE_Clk_Cond   : out STD_LOGIC;  -- clock conditioner MICROWIRE interface latch enable   
--   DATA_Clk_Cond : out STD_LOGIC;  -- clock conditioner MICROWIRE interface data
   
--   SYNC_Clk_Cond : out STD_LOGIC;  -- clock conditioner global clock synchronization
--   LD_Clk_Cond   : in STD_LOGIC;   -- clock conditioner lock detect                  
   
  


-- various RS-485 Interfaces
-- on IO-Bank 3
-------------------------------------------------------------------------------
   -- Bus 1: FTU slow control   
--   Bus1_Tx_En    : out STD_LOGIC;  -- bus 1: transmitter enable                                 
--   Bus1_Rx_En    : out STD_LOGIC;  -- bus 1: receiver enable

--   Bus1_RxD_0    : in STD_LOGIC;   -- crate 0
--   Bus1_TxD_0    : out STD_LOGIC;

--   Bus1_RxD_1    : in STD_LOGIC;   -- crate 1
--   Bus1_TxD_1    : out STD_LOGIC;

--  Bus1_RxD_2    : in STD_LOGIC;   -- crate 2
--   Bus1_TxD_2    : out STD_LOGIC;

--   Bus1_RxD_3    : in STD_LOGIC;   -- crate 3
--   Bus1_TxD_3    : out STD_LOGIC;  


   -- Bus 2: Trigger-ID to FAD boards
--   Bus2_Tx_En    : out STD_LOGIC;  -- bus 2: transmitter enable                                 
--   Bus2_Rx_En    : out STD_LOGIC;  -- bus 2: receiver enable
   
--   Bus2_RxD_0    : in STD_LOGIC;   -- crate 0
--   Bus2_TxD_0    : out STD_LOGIC;

--   Bus2_RxD_1    : in STD_LOGIC;   -- crate 1
--   Bus2_TxD_1    : out STD_LOGIC;

--   Bus2_RxD_2    : in STD_LOGIC;   -- crate 2
--   Bus2_TxD_2    : out STD_LOGIC;

--   Bus2_RxD_3    : in STD_LOGIC;   -- crate 3
--   Bus2_TxD_3    : out STD_LOGIC;  
   

-- auxiliary access
--   Aux_Rx_D      : in STD_LOGIC;     -- 
--   Aux_Tx_D      : out STD_LOGIC;    --  
--   Aux_Rx_En     : out STD_LOGIC;   --   Rx- and Tx enable 
--   Aux_Tx_En     : out STD_LOGIC;   --   also for auxiliary Trigger-ID
    		      	      			    	   	  

-- auxiliary Trigger-ID (i.e. to send the Trigger-ID to the counting hut/house/container)
--   TrID_Rx_D     : in STD_LOGIC;      -- 
--   TrID_Tx_D     : out STD_LOGIC;     -- 


-- Crate-Resets
-- on IO-Bank 3
-------------------------------------------------------------------------------
   Crate_Res0   : out STD_LOGIC;     -- 
   Crate_Res1   : out STD_LOGIC;     -- 
   Crate_Res2   : out STD_LOGIC;     -- 
   Crate_Res3   : out STD_LOGIC;     -- 


-- Busy signals from the FAD boards
-- on IO-Bank 3
-------------------------------------------------------------------------------
--   Busy0     : in STD_LOGIC;        -- 
--   Busy1     : in STD_LOGIC;        -- 
--   Busy2     : in STD_LOGIC;        -- 
--   Busy3     : in STD_LOGIC;        -- 



-- NIM outputs
-- on IO-Bank 0
-- LVDS output at the FPGA followed by LVDS to NIM conversion stage
-------------------------------------------------------------------------------
-- calibration
   Cal_NIM1_p  : out STD_LOGIC;     --  Cal_NIM1+ 
   Cal_NIM1_n  : out STD_LOGIC;     --  Cal_NIM1-
   Cal_NIM2_p  : out STD_LOGIC;     --  Cal_NIM2+  
   Cal_NIM2_n  : out STD_LOGIC;     --  Cal_NIM2- 

-- auxiliarry / spare NIM outputs
   NIM_Out0_p  : out STD_LOGIC;   -- NIM_Out0+
   NIM_Out0_n  : out STD_LOGIC;   -- NIM_Out0-
   NIM_Out1_p  : out STD_LOGIC;   -- NIM_Out1+
   NIM_Out1_n  : out STD_LOGIC;   -- NIM_Out1-

  

-- fast control signal outputs
-- LVDS output at the FPGA followed by LVDS to NIM  conversion stage
-- conversion stage
-------------------------------------------------------------------------------
   RES_p      : out STD_LOGIC;    --  RES+   Reset
   RES_n      : out STD_LOGIC;    --  RES-  IO-Bank 0

   TRG_p      : out STD_LOGIC;    -- TRG+  Trigger
   TRG_n      : out STD_LOGIC;    -- TRG-  IO-Bank 0

   TIM_Run_p  : out STD_LOGIC;   -- TIM_Run+  Time Marker
   TIM_Run_n  : out STD_LOGIC;   -- TIM_Run-  IO-Bank 2
   TIM_Sel    : out STD_LOGIC;   -- Time Marker selector on
                                 -- IO-Bank 2
                                                    
--   CLD_FPGA   : out STD_LOGIC;    -- DRS-Clock feedback into FPGA



-- LVDS calibration outputs
-- on IO-Bank 0
-------------------------------------------------------------------------------
-- to connector J13
   Cal_0_p    : out STD_LOGIC;  
   Cal_0_n    : out STD_LOGIC;
   Cal_1_p    : out STD_LOGIC;
   Cal_1_n    : out STD_LOGIC;
   Cal_2_p    : out STD_LOGIC;
   Cal_2_n    : out STD_LOGIC;
   Cal_3_p    : out STD_LOGIC;
   Cal_3_n    : out STD_LOGIC;

-- to connector J12
   Cal_4_p    : out STD_LOGIC;
   Cal_4_n    : out STD_LOGIC;
   Cal_5_p    : out STD_LOGIC;
   Cal_5_n    : out STD_LOGIC;
   Cal_6_p    : out STD_LOGIC;
   Cal_6_n    : out STD_LOGIC; 
   Cal_7_p    : out STD_LOGIC;
   Cal_7_n    : out STD_LOGIC  


-- Testpoints
-------------------------------------------------------------------------------
--   TP    : inout STD_LOGIC_VECTOR(32 downto 0)
--   TP_in    : in STD_LOGIC_VECTOR(34 downto 33);    -- input only

-- Board ID - inputs 
-- local board-ID "solder programmable"
-- all on 'input only' pins
-------------------------------------------------------------------------------
--    brd_id : in STD_LOGIC_VECTOR(7 downto 0)    -- input only		    
  );
end FTM_test2;





architecture Behavioral of FTM_test2 is
  
  component FTM_test1_dcm 
    port ( CLKIN_IN        : in    std_logic; 
           CLKFX_OUT       : out   std_logic; 
           CLKIN_IBUFG_OUT : out   std_logic);
  end component;

  component Clock_Divider
    port(
      clock      : IN  STD_LOGIC;
      enable_out : OUT STD_LOGIC
    );
  end component;

  signal clk_200M_sig : STD_LOGIC;
  signal enable_sig   : STD_LOGIC;        
 
  
begin

   
   OBUFDS_inst_TRG : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      O  => TRG_p,     -- Diff_p output (connect directly to top-level port)
      OB => TRG_n,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   );



   OBUFDS_inst_RES : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      O  => RES_p,     -- Diff_p output (connect directly to top-level port)
      OB => RES_n,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   );
  

   
  OBUFDS_inst_TIM : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      O  => TIM_Run_p,     -- Diff_p output (connect directly to top-level port)
      OB => TIM_Run_n,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   );  

  TIM_Sel <= '0';

  Crate_Res0 <= enable_sig; 
  Crate_Res1 <= enable_sig;
  Crate_Res2 <= enable_sig;
  Crate_Res3 <= enable_sig;
  




     OBUFDS_inst_Cal_NIM1 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      O  => Cal_NIM1_p,     -- Diff_p output (connect directly to top-level port)
      OB => Cal_NIM1_n,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   );  


       OBUFDS_inst_Cal_NIM2 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      O  => Cal_NIM2_p,     -- Diff_p output (connect directly to top-level port)
      OB => Cal_NIM2_n,   -- Diff_n output (connect directly to top-level port)
      I  => not enable_sig       -- Buffer input 
   ); 
   
 


          OBUFDS_inst_NIM_Out0 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (
      O  => NIM_Out0_p,     -- Diff_p output (connect directly to top-level port)
      OB => NIM_Out0_n,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   ); 


       OBUFDS_inst_NIM_Out1 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (    O  => NIM_Out1_p,     -- Diff_p output (connect directly to top-level port)
      OB => NIM_Out1_n,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   ); 



   

     OBUFDS_inst_Cal_0 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map  (  O  => Cal_0_p ,     -- Diff_p output (connect directly to top-level port)
      OB =>  Cal_0_n ,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   ); 

     OBUFDS_inst_Cal_1 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map  (  O  => Cal_1_p ,     -- Diff_p output (connect directly to top-level port)
      OB =>  Cal_1_n ,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   ); 

     OBUFDS_inst_Cal_2 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map  (  O  => Cal_2_p ,     -- Diff_p output (connect directly to top-level port)
      OB =>  Cal_2_n ,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   ); 

     OBUFDS_inst_Cal_3 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (   O  => Cal_3_p ,     -- Diff_p output (connect directly to top-level port)
      OB =>  Cal_3_n ,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   );     

 OBUFDS_inst_Cal_4 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map (   O  => Cal_4_p ,     -- Diff_p output (connect directly to top-level port)
      OB =>  Cal_4_n ,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   ); 

     OBUFDS_inst_Cal_5 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map  (  O  => Cal_5_p ,     -- Diff_p output (connect directly to top-level port)
      OB =>  Cal_5_n ,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   ); 

     OBUFDS_inst_Cal_6 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map  (  O  => Cal_6_p ,     -- Diff_p output (connect directly to top-level port)
      OB =>  Cal_6_n ,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   ); 

     OBUFDS_inst_Cal_7 : OBUFDS
   generic map (
      IOSTANDARD => "DEFAULT")
   port map  (  O  => Cal_7_p ,     -- Diff_p output (connect directly to top-level port)
      OB =>  Cal_7_n ,   -- Diff_n output (connect directly to top-level port)
      I  => enable_sig       -- Buffer input 
   );     

   



   
  
  Inst_FTU_test2_dcm : FTM_test1_dcm
    port map(
      CLKIN_IN => clk,
      CLKFX_OUT => clk_200M_sig,
      CLKIN_IBUFG_OUT => open
    );

  
  Inst_Clock_Divider : Clock_Divider
    port map (
      clock => clk_200M_sig,
      enable_out => enable_sig
    );

  
end Behavioral;




library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity Clock_Divider is
  port(
    clock     : in  std_logic;
    enable_out: out std_logic
    );
end entity Clock_Divider;

architecture RTL of Clock_Divider is
  
  --constant max_count   : integer := 5000000/1000000; -- for simulation
  constant max_count   : integer := 2500/1;   -- for implementation
--  constant final_count : integer := 100;
  
begin

  process(clock)
    variable count  : integer range 0 to max_count;
--    variable count2 : integer range 0 to final_count;
  begin
    if rising_edge(clock) then
      --enable_out <= '0';      
--      if count2 = final_count then
--        enable_out <= '0';
--      else
        if count < max_count/2 then          
          enable_out <= '0';
          count := count + 1;
        elsif count < max_count then
          enable_out <= '1';
          count := count + 1;
        else
          count := 0;
          enable_out <= '0';            
--          count2 := count2 + 1;
        end if; 
--      end if;
    end if;
  end process;

end architecture RTL;

