/**************************************************************\

  Control program for the fACT G-APD bias supply

  Adapted from previous version 'hvcontrol'.
  
  Oliver Grimm, August 2010
  
\**************************************************************/

#include <stdio.h>

#include "User.h"

#include <readline/readline.h>
#include <readline/history.h>

#define LOCKFILE "/tmp/CTX_HV_LOCK"

//
// Remove lock file before running default signal code
//
void CrashHandler(int Signal) {

  remove(LOCKFILE);
  printf("Caught signal number %d. Removing lockfile and performing standard signal action. Good luck.\n",Signal);
  signal(Signal, SIG_DFL);
  raise(Signal);
}

//
// Dummy handler to return from blocking syscalls
//
void DummyHandler(int) {};

//
// This function will be implicitly called by exit()
//
void ExitFunction() {

  if (remove(LOCKFILE) == -1) {
    printf("Could not remove lock file %s (%s)\n", LOCKFILE, strerror(errno));
  }
}

//
// Main program
//
int main() {

  char str[MAX_COM_SIZE], *Command;
  int Lock;

  // Assure only one instance runs
  // O_EXCL with O_CREAT assure that the lock file cannot be 
  // opened by another instance, i.e. there are no parallel write accesses
  if((Lock = open(LOCKFILE,O_WRONLY|O_CREAT|O_EXCL,S_IWUSR|S_IRUSR|S_IRGRP|S_IROTH)) == -1) {
    if(errno==EEXIST) {
      printf("Error: Lock file already existing\n");
      sprintf(str, "paste %s -s -d ' '", LOCKFILE);
      system(str);
    }
    else printf("Could not create lock file %s (%s)\n", LOCKFILE, strerror(errno));
    exit(EXIT_FAILURE);
  }
  close(Lock);
  sprintf(str,"echo Created >%s; date >>%s; echo by $USER@$HOSTNAME>>%s", LOCKFILE, LOCKFILE, LOCKFILE);
  system(str);
  
  system("clear");                   
  printf("\n*** Bias control (compiled %s, %s) ***\n\n", __DATE__, __TIME__);
 
  // Install signal handler and set signal SIGUSR1 to interrupt blocking system calls
  signal(SIGUSR1, &DummyHandler);
  siginterrupt (SIGUSR1, true);

  // Assure lock file is deleted in case of a program crash or call to exit()
  signal(SIGILL, &CrashHandler);
  signal(SIGABRT, &CrashHandler);
  signal(SIGFPE, &CrashHandler);
  signal(SIGSEGV, &CrashHandler);
  signal(SIGBUS, &CrashHandler);
  atexit(&ExitFunction);
  
  // Construct main instance
  static User M;
  
  // These signals were set during construction of EvidenceServer
  signal(SIGQUIT, &CrashHandler);  // CTRL-Backspace
  signal(SIGINT, &CrashHandler);   // CTRL-C
  signal(SIGHUP, &CrashHandler);   // Terminal closed
  signal(SIGTERM, &CrashHandler);

  // Handle command-line input
  while (!M.ExitRequest) {        
    Command = readline("\rBias> ");
	
	// NULL returned if interrupted by signal
    if (Command == NULL) continue;
    if (strlen(Command) > 0) add_history(Command);

    // Process command (via DIM gives automatic thread serialisation)
	DimClient::sendCommand("Bias/Command", Command);
    free(Command);
  }
}
