//
// Class processing user input
//

#include "User.h"
#include <readline/readline.h>

using namespace std;

// Branch table for command evaluation
static const struct CL_Struct { const char *Name;    
								void (User::*CommandPointer)();
								unsigned int MinNumParameter;
								const char *Parameters;
								const char *Help;
  } CommandList[] = 
   {{"synch", &User::cmd_synch, 0, "", "Synchronize board"},
    {"hv", &User::cmd_hv, 2, "<id>|<ch>|<all> <v>", "Change bias of pixel or (all) chan. of active boards"},
    {"gs", &User::cmd_gs, 1, "[crate] <volt>", "Global voltage set"},
	{"status", &User::cmd_status, 0, "[dac|current]", "Show status information (DAC values if requested)"},
	{"ccal", &User::cmd_ccal, 1, "<volt>", "Calibrate current measurement at given voltage"},
	{"mode", &User::cmd_mode, 1, "<static|dynamic>", "Set voltage stabilization mode"},
	{"load", &User::cmd_load, 1, "<file>", "Load and set bias settings from file"},
	{"save", &User::cmd_save, 1, "<file>", "Save current bias settings to file"},
	{"rate", &User::cmd_rate, 1, "<rate>", "Set refresh rate in Hz"},
	{"timeout", &User::cmd_timeout, 1, "<time>", "Set timeout to return from read in seconds"},
	{"reset", &User::cmd_reset, 1, "<crates>", "Reset crates"},
	{"help", &User::cmd_help, 0, "", "Print help"},
	{"exit", &User::cmd_exit, 0, "", "Exit program"}};


//
// Constructor
//
User::User(): EvidenceServer(SERVER_NAME) {

  // DIM console service used in PrintMessage()
  ConsoleText = NULL;
  ConsoleOut = new DimService(SERVER_NAME"/ConsoleOut", (char *) "");

  // Get configuration data
  vector<string> Boards = Tokenize(GetConfig("Boards"), " \t");
  Boards = Tokenize("FTE00FOH", " \t");
  fTimeOut = atof(GetConfig("TimeOut").c_str());
  fStatusRefreshRate = atof(GetConfig("StatusRefreshRate").c_str());
  fMaxDiff = atof(GetConfig("HVMaxDiffNew").c_str());

  if (fStatusRefreshRate < MIN_RATE || fStatusRefreshRate > MAX_RATE)  fStatusRefreshRate = 1;

  // Open devices
  for (unsigned int i=0; i<Boards.size(); i++) {
  
    class Crate *New = new class Crate(Boards[i], Crates.size(), this);

    if (New->InitOK && New->Synch()) {
       PrintMessage("Synchronized and reset crate %s (#%d)\n", Boards[i].c_str(), Crates.size());
       Crates.push_back(New);
    }
    else {
      Message(WARN, "Failed to synchronize crate %s", Boards[i].c_str());
	  delete New;
    }
  }

  // Create instances
  pm 	 = new PixelMap(GetConfig("PixMapTable"));
  
  // Install DIM command (after all initialized)
  DIMCommand = new DimCommand((char *) SERVER_NAME"/Command", (char *) "C", this);

  // Create monitor thread and make accessible for sending signal
  if ((pthread_create(&Thread, NULL, (void * (*)(void *)) LaunchMonitor,(void *) this)) != 0) {
    Message(FATAL, "pthread_create() failed with Monitor thread");
  }
}


//
// Destructor
//
User::~User() {

  // Wait for thread to quit
  if (pthread_join(Thread, NULL) != 0) {
    PrintMessage("pthread_join() failed");
  }

  // Delete all crates
  for (unsigned int i=0; i<Crates.size(); i++) delete Crates[i];

  delete DIMCommand;    
  delete pm;
  delete ConsoleOut;	
  free(ConsoleText);  
}

//
// Process user input
//
void User::commandHandler() {

  // Build string safely
  string Command = string(getCommand()->getString(), getCommand()->getSize());

  // Check if command is legal and ignore empty commands 
  if (getCommand() != DIMCommand || Command.size() < 2) return;

  // Shell command
  if(Command[0]=='.') {
    system(Command.c_str()+1);
    return;
  }

  // Parse command into tokens
  Parameter = Tokenize(Command, " ");

  // Search for command in command list
  for(unsigned int CmdNumber=0; CmdNumber<sizeof(CommandList)/sizeof(CL_Struct); CmdNumber++) {
    if (Match(Parameter[0], CommandList[CmdNumber].Name)) {
      if(Parameter.size()-1 < CommandList[CmdNumber].MinNumParameter) {
		PrintMessage("Usage: %s %s\n", CommandList[CmdNumber].Name, CommandList[CmdNumber].Parameters);
		return;
	  }

	  // Jump to command function
	  (this->*CommandList[CmdNumber].CommandPointer)();
	  return;  
    }
  }
  PrintMessage("Unknown command '%s'\n", Parameter[0].c_str());
}


// Print help
void User::cmd_help() {

  char Buffer[MAX_COM_SIZE];
  for(unsigned int i=0; i<sizeof(CommandList)/sizeof(CL_Struct); i++) {
    snprintf(Buffer, sizeof(Buffer), "%s %s", CommandList[i].Name, CommandList[i].Parameters);
    PrintMessage("%-28s%s\n", Buffer, CommandList[i].Help);
  }
  
  PrintMessage(".<command>                  Execute shell command\n\n"
  	"Items in <> are mandatory, in [] optional, | indicates mutual exclusive or.\n");
} 

//
// Synchronize boards
//
void User::cmd_synch() {

  for (unsigned int i=0; i<Crates.size(); i++) {
	if (Crates[i]->Synch()) PrintMessage("Synchronized crate %d\n", i);
	else PrintMessage("Failed to synchronize crate %d\n", i);
  }
}

//
// Set new bias voltage
//
void User::cmd_hv() {

  unsigned int Errors=0;
  double Double;
  struct Range Crt, Chan;
  vector< map<unsigned int, double> > Voltages (Crates.size());

  // Loop over all parameters
  for (unsigned int n=1; n < Parameter.size()-1; n+=2) {

	// Extract channel identification
	if (pm->Pixel_to_HVboard(Parameter[n]) != 999999999) {
      Crt.Min = Crt.Max = pm->Pixel_to_HVboard(Parameter[n]);
      Chan.Min = Chan.Max = pm->Pixel_to_HVchain(Parameter[n])*NUM_CHANNELS + pm->Pixel_to_HVchannel(Parameter[n]);
	}
	else {
      vector<string> T = Tokenize(Parameter[n], "/");
	  Crt.Min = 0;
	  Crt.Max = Crates.size()-1;
	  Chan.Min = 0;
	  Chan.Max = MAX_NUM_BOARDS*NUM_CHANNELS-1;
	  
	  if (Parameter[n] == "-") continue;

	  if (T.size() == 2) {
		if(!ConvertToRange(T[0], Crt) || !ConvertToRange(T[1], Chan)) {
		  PrintMessage("Numeric conversion or out-of-range error for parameter %d, skipping channel\n", n);
		  continue;
		}
	  }
	  else {
		Crt.Min = Crt.Max = 0;
		if (!ConvertToRange(T[0], Chan)) {
		  PrintMessage("Numeric conversion or out-of-range error for parameter %d, skipping channel\n", n);
		  continue;
		}
	  }
	}
  
	// Convert voltage value and check format 
	if (!ConvertToDouble(Parameter[n+1], &Double)) {
	  PrintMessage("Error: Wrong number format for voltage setting\n");
	  continue;
	}

	// Loop over given crates and channels
	for (int i=Crt.Min; i<=Crt.Max; i++) for (int j=Chan.Min; j<=Chan.Max; j++) {
	  // Voltage change (number starts with + oder -) ignored if current DAC value is zero
	  if (isdigit(Parameter[n+1][0])==0 && Crates[i]->GetDAC(j) == 0) continue;

	  // Relative or absolute change?
	  if (isdigit(Parameter[n+1][0]) == 0) Voltages[i][j] = Crates[i]->GetVoltage(j) + Double;
	  else Voltages[i][j] = Double;
	} // Channels
  } // Loop over command argument

  // Ramp voltages and update DIM services
  for (unsigned int i=0; i<Voltages.size(); i++) {
	Errors += RampVoltages(i, Voltages[i]);
    Crates[i]->UpdateDIM();
  }

  // Error message only if not yet too many errors
  if (Errors > 0) {
	for (unsigned int i=0; i<Crates.size(); i++) {
	  if (Crates[i]->ErrorCount > MAX_ERR_COUNT) return;
	}
    Message(ERROR, "%d errors occurred from SetChannels()", Errors);
  }
}

//
// Load bias settings from file
//
void User::cmd_load() {

  char Buffer[MAX_COM_SIZE];
  int Errors = 0, Channel;
  unsigned int NBoards = 0;
  double Value;
  FILE *File;
  map<unsigned int, double> Voltages;

  // Open file
  if ((File=fopen(Parameter[1].c_str(), "r")) == NULL) {
    PrintMessage("Error: Could not open file '%s' (%s)\n", Parameter[1].c_str(), strerror(errno));
    return;
  }

  // Scan through file line by line
  while (fgets(Buffer, sizeof(Buffer), File) != NULL) {
    for (unsigned int Crate=0; Crate<Crates.size(); Crate++) if (Match(Crates[Crate]->Name, Buffer)) {

	  PrintMessage("Found bias settings for board %s (#%d)\n\r", Crates[Crate]->Name, Crate);

	  Voltages.clear();
	  Channel = 0;
	  while (fscanf(File, "%lf", &Value)==1 && Channel<MAX_NUM_BOARDS*NUM_CHANNELS) {
		Voltages[Channel++] = Value;
	  }

	  // Ramp channels
	  Errors += RampVoltages(Crate, Voltages);

      // Update DIM service
	  Crates[Crate]->UpdateDIM();

	  if (ferror(File) != 0) {
		PrintMessage("Error reading DAC value from file, terminating. (%s)\n",strerror(errno));
    	return;
	  }
	  else PrintMessage("\nFinished updating board\n");
      NBoards++;
    } // Loop over boards
  } // while()
    	    
  if (NBoards != Crates.size()) PrintMessage("Warning: Could not load bias settings for all connected crates\n");
  else if (Errors == 0) PrintMessage("Success: Read bias settings for all connected crates\n");

  if (Errors != 0) PrintMessage("Warning: %d error(s) occurred\n", Errors);

  if (fclose(File) != 0) PrintMessage("Error: Could not close file '%s'\n", Parameter[1].c_str());
}
	   
//
// Set refresh rate
//
void User::cmd_rate() {

  double Rate;

  if (!ConvertToDouble(Parameter[1], &Rate)) {
     PrintMessage("Error: Wrong number format\n");
     return;   
  }

  // Check limits
  if (Rate<MIN_RATE || Rate>MAX_RATE) {
    PrintMessage("Refresh rate out of range (min: %.2f Hz, max: %.2f Hz)\n", MIN_RATE, MAX_RATE);
    return;
  }

  fStatusRefreshRate = Rate;
  PrintMessage("Refresh rate set to %.2f Hz\n", fStatusRefreshRate);
}
  
//
// Reset crates
//
void User::cmd_reset() {

  struct Range R = {0, Crates.size()-1};

  // Check ranges  
  if(!ConvertToRange(Parameter[1], R)) {
	PrintMessage("Error, crate number out of range\n");
	return;
  }
  
  for (int i=R.Min; i<=R.Max; i++) {
	if (Crates[i]->SystemReset() == 1) PrintMessage("System reset of crate %s (#%d)\n", Crates[i]->Name, i);
	else PrintMessage("Error: Could not reset board %s (#%d)\n", Crates[i]->Name, i);
  }
}

//
// Read channel
//
void User::cmd_gs() {

  double Voltage;

  if (!ConvertToDouble(Parameter[1], &Voltage)) return;

  for (unsigned int i=0; i<Crates.size(); i++) {
	if (Crates[i]->GlobalSet(Voltage) != 1) {
	  PrintMessage("Error: Could not global set crate %d\n", i);
	}
  }    
}

//
// Determine current measurement offset
//
void User::cmd_ccal() {

  double Voltage;
    
  if (!ConvertToDouble(Parameter[1], &Voltage)) {
    PrintMessage("Error with format of voltage parameter\n");  
	return;
  }

  // Execute current offset determination
  for (unsigned int i=0; i<Crates.size(); i++) {
	if (!Crates[i]->CurrentCalib(Voltage)) {
      PrintMessage("Error with current calibration of crate %d\n", i);
	  return;
	}

	PrintMessage("Current calibration of crate %d done\n", i);
  }
}

//
// Save bias settings of all boards
//
void User::cmd_save() {

  FILE *File;
  time_t Time = time(NULL);

  if ((File = fopen(Parameter[1].c_str(), "w")) == NULL) {
    PrintMessage("Error: Could not open file '%s' (%s)\n", Parameter[1].c_str(), strerror(errno));
    return;
  }

  fprintf(File,"********** Bias settings of %s **********\n\n", ctime(&Time));

  for (unsigned int i=0; i<Crates.size(); i++) {
    fprintf(File, "%s\n\n", Crates[i]->Name);

    for (int j=0; j<MAX_NUM_BOARDS*NUM_CHANNELS; j++) fprintf(File,"%.3f ",Crates[i]->GetVoltage(j));
    fprintf(File, "\n");
  }

  if (fclose(File) != 0) {
    PrintMessage("Error: Could not close file '%s' (%s)\n", Parameter[1].c_str(), strerror(errno));  
  }
}

//
// Set operation mode
//
void User::cmd_mode() {

  if (Match(Parameter[1], "static")) Mode = mode_static;
  else {
    Mode = mode_dynamic;
	for (unsigned int i=0; i<Crates.size(); i++) {
	  Crates[i]->SetRefCurrent();
	}
  }
} 

//
// Print status
//
void User::cmd_status() {

  PrintMessage(" Number of crates:  %d\n", Crates.size());
  PrintMessage(" Refresh rate:      %.2f Hz\n", fStatusRefreshRate);
  PrintMessage(" Time out:          %.2f s\n\n", fTimeOut);
  PrintMessage(" MaxDiff :          %u\n", fMaxDiff);

  for (unsigned int i=0; i<Crates.size(); i++) {
    PrintMessage(" CRATE %d (%s)\n   Wrap counter: %s (%d)  Reset: %s  Error count: %d\n ",
		i, Crates[i]->Name,	Crates[i]->WrapOK ? "ok":"error", Crates[i]->WrapCount, 
		Crates[i]->ResetHit ? "yes" : "no", Crates[i]->ErrorCount);

	if (Parameter.size() == 1) PrintMessage("Channel voltages (in V)");
    else if (Match(Parameter[1], "dac")) PrintMessage("Channel voltages (in DAC values)");
	else PrintMessage("Channel currents (in uA)");

    for (int j=0; j<MAX_NUM_BOARDS*NUM_CHANNELS; j++) {
	  if (j%12 == 0) PrintMessage("\n%3.1d:  ", j);
	  if (!Crates[i]->Present[j/NUM_CHANNELS][j%NUM_CHANNELS]) PrintMessage("  -   ");
      else if (Parameter.size() == 1) PrintMessage("%#5.2f ",Crates[i]->GetVoltage(j));
	  else if (Match(Parameter[1], "dac")) PrintMessage("%5d ", Crates[i]->GetDAC(j));
	  else PrintMessage("%#5.2f %s ", Crates[i]->GetCurrent(j), Crates[i]->OC[j/NUM_CHANNELS][j%NUM_CHANNELS] ? "OC":"");
    }
	PrintMessage("\n");
  }
} 

//
// Set timeout to return from read
//
void User::cmd_timeout() {

  double Timeout;

  if (!ConvertToDouble(Parameter[1], &Timeout)) {
     PrintMessage("Error: Wrong number format\n");
     return;   
  }

  fTimeOut = Timeout;
  PrintMessage("Timeout set to %.2f s\n", fTimeOut);
}
    
//
// Exit program
//
void User::cmd_exit() {

  ExitRequest = true;
  pthread_kill(Thread, SIGUSR1);  // Make tjread return from usleep()
}
  

//
// Print message to screen and to DIM text service
//
void User::PrintMessage(const char *Format, ...) {

  static char Error[] = "vasprintf() failed in PrintMessage()";
  char *Text;

  // Evaluate arguments    
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format);
  if (vasprintf(&Text, Format, ArgumentPointer) == -1) Text = Error;
  va_end(ArgumentPointer);
 
  // Print to console
  printf("%s", Text); // New prompt
  fflush(stdout);
  if (strlen(Text)>0 && Text[strlen(Text)-1]=='\n') rl_on_new_line(); // New prompt

  // Send to DIM text service
  ConsoleOut->updateService(Text); 

  // Free old text
  if (ConsoleText != Error) free(ConsoleText);
  ConsoleText = Text; 
}


// Ramp to new voltage with maximum step size given in fMaxDiff
// No ramping when decreasing voltage
unsigned int User::RampVoltages(int Crate, map<unsigned int, double> Voltages) {

  map<unsigned int, double> Target;
  int Errors = 0;

  // Ramp until all channels at desired value
  while (!Voltages.empty() && Errors < MAX_ERR_COUNT) {
    // Remove channels already at target (check for DAC, not for floating-point voltage)
	for (map<unsigned int, double>::iterator it = Voltages.begin(); it != Voltages.end(); ++it) {
	  //if (Crates[Crate]->GetDAC(it->first) == (unsigned int ) (it->second/90.0*0x0fff)) Voltages.erase(it);
	  if (fabs(Crates[Crate]->GetVoltage(it->first)-it->second) < 0.001) Voltages.erase(it);
	}
	
	// Limit voltage changes to fMaxDiff
	Target = Voltages;
	for (map<unsigned int, double>::iterator it = Target.begin(); it != Target.end(); ++it) {
	  if (Crates[Crate]->GetVoltage(it->first) + fMaxDiff < it->second) {
		it->second = Crates[Crate]->GetVoltage(it->first) + fMaxDiff;
	  }
	}	
	
	// Set channels to next target and wait 10 ms
	if (Crates[Crate]->SetChannels(Target) != 1) Errors++;
	usleep(10000);
  }
  
  return Errors;
}


//
// Check status
//
void User::Monitor() {

  static bool Warned = false;

  while (!ExitRequest) {
	for (unsigned int i=0; i<Crates.size(); i++) {
      if (Crates[i]->ErrorCount > MAX_ERR_COUNT) {
    	if (!Warned) {
          Warned = true;
          Message(WARN, "Warning: Crate %d has many read/write errors, further error reporting disabled", i);
    	}
    	continue;
      }

      if (Crates[i]->ResetHit) {
    	Message(INFO, "Manual reset of board %d, setting voltages to zero and issuing system reset", i);
		Crates[i]->GlobalSet(0);
		Crates[i]->SystemReset();
      }

      if (!Crates[i]->WrapOK) {
    	Message(ERROR, "Wrap counter mismatch of board %d", i);
      }

      if (Crates[i]->ReadAll() != 1) {
    	Message(ERROR, "Monitor could not read status from crate %d", i);
		continue;
      }

	  map<unsigned int, double> Voltages;
	  
      for (int j=0; j<MAX_NUM_BOARDS*NUM_CHANNELS; j++) {
    	if (Crates[i]->OC[j/NUM_CHANNELS][j%NUM_CHANNELS]) {
		  Message(WARN, "Overcurrent on crate %d, board %d, channel %d, setting voltage to zero", i, j/NUM_CHANNELS, j%NUM_CHANNELS);
		  Voltages[j] = 0;
    	}
      }
	  if (!Voltages.empty()) {
		Crates[i]->SetChannels(Voltages);
		Crates[i]->SystemReset();
	  }
	  
	  if (Mode == mode_dynamic) Crates[i]->AdaptVoltages();
	} // for

	// Wait
	usleep((unsigned long) floor(1000000./fStatusRefreshRate));
  } // while
}

// Call monitor loop inside class
void User::LaunchMonitor(User *m) {

  m->Monitor();
}


//
// Check if two strings match (min 1 character must match)
//
bool User::Match(string str, const char *cmd) {

  return strncasecmp(str.c_str(),cmd,strlen(str.c_str())==0 ? 1:strlen(str.c_str())) ? false:true;
}

//
// Conversion function from string to double or int
//
// Return false if conversion did not stop on whitespace or EOL character
bool User::ConvertToDouble(string String, double *Result) {

  char *EndPointer;
  
  *Result = strtod(String.c_str(), &EndPointer);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}

bool User::ConvertToInt(string String, int *Result) {

  char *EndPointer;
  
  *Result = (int) strtol(String.c_str(), &EndPointer, 0);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}

//
// Interprets a range
//
bool User::ConvertToRange(string String, struct User::Range &R) {

  int N, M;

  // Full range
  if (Match(String, "all")) return true;

  // Single number
  if (ConvertToInt(String, &N)) {
	if (N>= R.Min && N<=R.Max) {
	  R.Max = R.Min = N;
	  return true;
	}
	return false;
  }
  
  // Range a-b
  vector<string> V = EvidenceServer::Tokenize(String, "-");
  if (V.size()==2 && ConvertToInt(V[0], &N) && ConvertToInt(V[1], &M) && N>=R.Min && M<=R.Max) {
	R.Min = N;
	R.Max = M;
	return true;
  }
  
  return false;
}
