----------------------------------------------------------------------------------
-- Company:        ETH Zurich, Institute for Particle Physics
-- Engineer:       P. Vogler, Q. Weitzel
-- 
-- Create Date:    13 January 2011
-- Design Name:    
-- Module Name:    FTM_test9 - Behavioral 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description:    Test firmware for FTM board: try to contact FTU via RS485
--                                              
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
--
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
library UNISIM;
use UNISIM.VComponents.all;

entity FTM_test9 is
  port(
    
-- Clock
   clk   : IN  STD_LOGIC;                     -- external clock from
                                              -- oscillator U47
-- connection to the WIZnet W5300 ethernet controller
-- on IO-Bank 1
-------------------------------------------------------------------------------
    -- W5300 data bus
--   W_D  : inout STD_LOGIC_VECTOR(15 downto 0);  -- 16-bit data bus to W5300	


    -- W5300 address bus
--   W_A  : out STD_LOGIC_VECTOR(9 downto 1);   -- there is NO net W_A0 because
                                               -- the W5300 is operated in the 
                                               -- 16-bit mode 

    -- W5300 controll signals
    -- the signals W_INT, W_RD, W_WR and W_RES also go to testpoints T17
    -- W_CS is also routed to testpoint JP7
--   W_CS   : out  STD_LOGIC;                      --  W5300 chip select
--   W_INT  : IN  STD_LOGIC;                       -- interrupt
--   W_RD   : out  STD_LOGIC;                      -- read
--   W_WR   : out  STD_LOGIC;                      -- write
--   W_RES  : out  STD_LOGIC                      -- reset W5300 chip

    -- W5300 buffer ready indicator
--   W_BRDY :  in STD_LOGIC_VECTOR(3 downto 0); 

    -- testpoints (T18) associated with the W5300 on IO-bank 1
--    W_T    : inout STD_LOGIC_VECTOR(3 downto 0);  
 


-- SPI Interface
-- connection to the EEPROM U36 (AL25L016M) and 
-- temperature sensors U45, U46, U48 and U49 (all MAX6662)
-- on IO-Bank 1
-------------------------------------------------------------------------------
--   S_CLK  : out  STD_LOGIC;     -- SPI clock

   -- EEPROM
--   MOSI   : out  STD_LOGIC;     -- master out slave in
--   MISO   : in   STD_LOGIC;     -- master in slave out
--   EE_CS  : out  STD_LOGIC;     -- EEPROM chip select

   -- temperature sensors U45, U46, U48 and U49
--   SIO    : inout  STD_LOGIC;          -- serial IO
--   TS_CS  : out STD_LOGIC_VECTOR(3 downto 0);     -- temperature sensors chip select

 

-- Trigger primitives inputs
-- on IO-Bank 2
-------------------------------------------------------------------------------
--   Trig_Prim_A  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 0
--   Trig_Prim_B  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 1
--   Trig_Prim_C  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 2
--   Trig_Prim_D  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 3

  

-- NIM inputs
------------------------------------------------------------------------------
   -- on IO-Bank 3  
--   ext_Trig  : in  STD_LOGIC_VECTOR(2 downto 1);      -- external trigger input
--   Veto       : in  STD_LOGIC;                         -- trigger veto input
--   NIM_In     : in  STD_LOGIC_VECTOR(2 downto 0);      -- auxiliary inputs

   -- on IO-Bank 0
--   NIM_In3_GCLK  : in  STD_LOGIC;      -- input with global clock buffer available 

   

-- LEDs on IO-Banks 0 and 3
-------------------------------------------------------------------------------
   LED_red  : out STD_LOGIC_VECTOR(3 downto 0);    -- red
   LED_ye   : out STD_LOGIC_VECTOR(1 downto 0);    -- yellow
   LED_gn   : out STD_LOGIC_VECTOR(1 downto 0);    -- green

-- Clock conditioner LMK03000
-- on IO-Bank 3
-------------------------------------------------------------------------------
--   CLK_Clk_Cond  : out STD_LOGIC;  -- clock conditioner MICROWIRE interface clock
--   LE_Clk_Cond   : out STD_LOGIC;  -- clock conditioner MICROWIRE interface latch enable   
--   DATA_Clk_Cond : out STD_LOGIC;  -- clock conditioner MICROWIRE interface data
   
--   SYNC_Clk_Cond : out STD_LOGIC;  -- clock conditioner global clock synchronization
--   LD_Clk_Cond   : in STD_LOGIC;   -- clock conditioner lock detect                  
   
-- various RS-485 Interfaces
-- on IO-Bank 3
-------------------------------------------------------------------------------
   -- Bus 1: FTU slow control   
   Bus1_Tx_En    : out STD_LOGIC;  -- bus 1: transmitter enable                                 
   Bus1_Rx_En    : out STD_LOGIC;  -- bus 1: receiver enable

   Bus1_RxD_0    : in STD_LOGIC;   -- crate 0
   Bus1_TxD_0    : out STD_LOGIC

--   Bus1_RxD_1    : in STD_LOGIC;   -- crate 1
--   Bus1_TxD_1    : out STD_LOGIC;

--   Bus1_RxD_2    : in STD_LOGIC;   -- crate 2
--   Bus1_TxD_2    : out STD_LOGIC;

--   Bus1_RxD_3    : in STD_LOGIC;   -- crate 3
--   Bus1_TxD_3    : out STD_LOGIC;  


   -- Bus 2: Trigger-ID to FAD boards
--   Bus2_Tx_En    : out STD_LOGIC;  -- bus 2: transmitter enable                                 
--   Bus2_Rx_En    : out STD_LOGIC;  -- bus 2: receiver enable

--   Bus2_RxD_0    : in STD_LOGIC;   -- crate 0
--   Bus2_TxD_0    : out STD_LOGIC;

--   Bus2_RxD_1    : in STD_LOGIC;   -- crate 1
--   Bus2_TxD_1    : out STD_LOGIC;

--   Bus2_RxD_2    : in STD_LOGIC;   -- crate 2
--   Bus2_TxD_2    : out STD_LOGIC;

--   Bus2_RxD_3    : in STD_LOGIC;   -- crate 3
--   Bus2_TxD_3    : out STD_LOGIC;  
   

-- auxiliary access
--   Aux_Rx_D      : in STD_LOGIC;     -- 
--   Aux_Tx_D      : out STD_LOGIC;    --  
--   Aux_Rx_En     : out STD_LOGIC;   --   Rx- and Tx enable 
--   Aux_Tx_En     : out STD_LOGIC;   --   also for auxiliary Trigger-ID
    		      	      			    	   	  

-- auxiliary Trigger-ID (i.e. to send the Trigger-ID to the counting hut/house/container)
--   TrID_Rx_D     : in STD_LOGIC;      -- 
--   TrID_Tx_D     : out STD_LOGIC     -- 


-- Crate-Resets
-- on IO-Bank 3
-------------------------------------------------------------------------------
--   Crate_Res0   : out STD_LOGIC;     -- 
--   Crate_Res1   : out STD_LOGIC;     -- 
--   Crate_Res2   : out STD_LOGIC;     -- 
--   Crate_Res3   : out STD_LOGIC;     -- 


-- Busy signals from the FAD boards
-- on IO-Bank 3
-------------------------------------------------------------------------------
--   Busy0     : in STD_LOGIC;        -- 
--   Busy1     : in STD_LOGIC;        -- 
--   Busy2     : in STD_LOGIC;        -- 
--   Busy3     : in STD_LOGIC;        -- 



-- NIM outputs
-- on IO-Bank 0
-- LVDS output at the FPGA followed by LVDS to NIM conversion stage
-------------------------------------------------------------------------------
-- calibration
--   Cal_NIM1_p  : out STD_LOGIC;     --  Cal_NIM1+ 
--   Cal_NIM1_n  : out STD_LOGIC;     --  Cal_NIM1-
--   Cal_NIM2_p  : out STD_LOGIC;     --  Cal_NIM2+  
--   Cal_NIM2_n  : out STD_LOGIC;     --  Cal_NIM2- 

-- auxiliarry / spare NIM outputs
--   NIM_Out0_p  : out STD_LOGIC;   -- NIM_Out0+
--   NIM_Out0_n  : out STD_LOGIC;   -- NIM_Out0-
--   NIM_Out1_p  : out STD_LOGIC;   -- NIM_Out1+
--   NIM_Out1_n  : out STD_LOGIC;   -- NIM_Out1-

  

-- fast control signal outputs
-- LVDS output at the FPGA followed by LVDS to NIM  conversion stage
-- conversion stage
-------------------------------------------------------------------------------
--  RES_p      : out STD_LOGIC;    --  RES+   Reset
--  RES_n      : out STD_LOGIC;    --  RES-  IO-Bank 0

--  TRG_p      : out STD_LOGIC;    -- TRG+  Trigger
--  TRG_n      : out STD_LOGIC;    -- TRG-  IO-Bank 0

--  TIM_Run_p  : out STD_LOGIC;   -- TIM_Run+  Time Marker
--  TIM_Run_n  : out STD_LOGIC;   -- TIM_Run-  IO-Bank 2
--  TIM_Sel    : out STD_LOGIC   -- Time Marker selector on
                                   -- IO-Bank 2
                                                    
--   CLD_FPGA   : out STD_LOGIC;    -- DRS-Clock feedback into FPGA



-- LVDS calibration outputs
-- on IO-Bank 0
-------------------------------------------------------------------------------
-- to connector J13
--   Cal_0_p    : out STD_LOGIC;  
--   Cal_0_n    : out STD_LOGIC;
--   Cal_1_p    : out STD_LOGIC;
--   Cal_1_n    : out STD_LOGIC;
--   Cal_2_p    : out STD_LOGIC;
--   Cal_2_n    : out STD_LOGIC;
--   Cal_3_p    : out STD_LOGIC;
--   Cal_3_n    : out STD_LOGIC;

-- to connector J12
--   Cal_4_p    : out STD_LOGIC;
--   Cal_4_n    : out STD_LOGIC;
--   Cal_5_p    : out STD_LOGIC;
--   Cal_5_n    : out STD_LOGIC;
--   Cal_6_p    : out STD_LOGIC;
--   Cal_6_n    : out STD_LOGIC; 
--   Cal_7_p    : out STD_LOGIC;
--   Cal_7_n    : out STD_LOGIC;  


-- Testpoints
-------------------------------------------------------------------------------
--   TP    : inout STD_LOGIC_VECTOR(32 downto 0)
--   TP_in    : in STD_LOGIC_VECTOR(34 downto 33);    -- input only

-- Board ID - inputs 
-- local board-ID "solder programmable"
-- all on 'input only' pins
-------------------------------------------------------------------------------
--    brd_id : in STD_LOGIC_VECTOR(7 downto 0)    -- input only		    
 );
end FTM_test9;

architecture Behavioral of FTM_test9 is

  COMPONENT FTM_test9_dcm
    PORT(
      CLKIN_IN : IN std_logic;
      RST_IN : IN std_logic;          
      CLKFX_OUT : OUT std_logic;
      CLK0_OUT : OUT std_logic;
      LOCKED_OUT : OUT std_logic
      );
  END COMPONENT;

  component FTM_test9_rs485_interface
    GENERIC( 
      CLOCK_FREQUENCY : integer;      -- Hertz
      BAUD_RATE       : integer       -- bits / sec
    );
    PORT( 
      clk      : IN     std_logic;
      -- RS485
      rx_d     : IN     std_logic;
      rx_en    : OUT    std_logic;
      tx_d     : OUT    std_logic;
      tx_en    : OUT    std_logic;
      -- FPGA
      rx_data  : OUT    std_logic_vector(7 DOWNTO 0);
      -- rx_busy  : OUT    std_logic := '0';
      rx_valid : OUT    std_logic := '0';
      tx_data  : IN     std_logic_vector(7 DOWNTO 0);
      tx_busy  : OUT    std_logic := '0';
      tx_start : IN     std_logic
    );
  end component;
  
  signal reset_sig   : STD_LOGIC := '0'; -- initialize reset to 0 at power up 
  signal clk_50M_sig : STD_LOGIC;
  
  signal LED_red_sig : STD_LOGIC_VECTOR(3 downto 0) := (others => '1');
  signal LED_ye_sig  : STD_LOGIC_VECTOR(1 downto 0) := (others => '1');
  signal LED_gn_sig  : STD_LOGIC_VECTOR(1 downto 0) := (others => '1');
  
  signal rx_data_sig  : STD_LOGIC_VECTOR(7 DOWNTO 0) := (others => '0');
 -- signal rx_busy_sig  : STD_LOGIC;
  signal rx_valid_sig : STD_LOGIC;

  signal txcnt : integer range 0 to 28 := 0;  -- count 28 1-byte frames

  signal tx_start_sig : std_logic := '0';
  signal tx_data_sig  : std_logic_vector (7 DOWNTO 0) := (others => '0');
  signal tx_busy_sig  : std_logic;  -- initialized in FTU_rs485_interface

  
  type FTM_test9_StateType is (INIT, SEND, WAIT_FOR_ANSWER, FINISHED);
  signal FTM_test9_State : FTM_test9_StateType;

begin

  Inst_FTM_test9_dcm: FTM_test9_dcm PORT MAP(
    CLKIN_IN => clk,
    RST_IN => reset_sig,
    CLKFX_OUT => clk_50M_sig,
    CLK0_OUT => open,
    LOCKED_OUT => open
  );

  Inst_FTM_test9_rs485_interface : FTM_test9_rs485_interface
    generic map(
      CLOCK_FREQUENCY => 50000000,
      BAUD_RATE       => 250000
    )
    port map(
      clk      => clk_50M_sig,
      -- RS485
      rx_d     => Bus1_RxD_0,
      rx_en    => Bus1_Rx_En,
      tx_d     => Bus1_TxD_0,
      tx_en    => Bus1_Tx_En,
      -- FPGA
      rx_data  => rx_data_sig,
  --    rx_busy  => rx_busy_sig,
      rx_valid => rx_valid_sig,
      tx_data  => tx_data_sig,
      tx_busy  => tx_busy_sig,
      tx_start => tx_start_sig
    );
    
  --FTM test9 state machine

  FTM_test9_FSM: process (clk_50M_sig)
  begin
    if Rising_edge(clk_50M_sig) then
      case FTM_test9_State is

        when INIT =>
          LED_red_sig(0) <= '0';
	  LED_ye_sig(0) <= '1';
          LED_gn_sig(0) <= '1';
          FTM_test9_State <= SEND;
          
        when SEND =>   
          LED_red_sig(0) <= '0';
	  LED_ye_sig(0) <= '1';
          LED_gn_sig(0) <= '1';
          if tx_busy_sig = '0' then
            if txcnt = 0 then           -- start delimiter
              txcnt <= txcnt + 1;
              tx_data_sig <= "01000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 1 then        -- FTU address
              txcnt <= txcnt + 1;
              tx_data_sig <= "00111111";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 2 then        -- FTM address
              txcnt <= txcnt + 1;
              tx_data_sig <= "11000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 3 then        -- firmware ID
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000001";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 4 then        -- command
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 5 then        -- data: DAC A low
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 6 then        -- data: DAC A high
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 7 then        -- data: DAC B low
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 8 then        -- data: DAC B high
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 9 then        -- data: DAC C low
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 10 then        -- data: DAC C high
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 11 then        -- data: DAC D low
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 12 then        -- data: DAC D high
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 13 then        -- data: DAC E low
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = 14 then        -- data: DAC E high
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;              
            elsif txcnt < (28 - 2) then        -- data: not used
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = (28 - 2) then        -- CRC error counter
              txcnt <= txcnt + 1;
              tx_data_sig <= "00000000";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            elsif txcnt = (28 - 1) then        -- check sum
              txcnt <= txcnt + 1;
              tx_data_sig <= "00011101";
              tx_start_sig <= '1';
              FTM_test9_State <= SEND;
            else                        -- transmission finished
              txcnt <= 0;
              LED_red_sig(0) <= '1';
              LED_ye_sig(0) <= '0';
              LED_gn_sig(0) <= '1';
              FTM_test9_State <= WAIT_FOR_ANSWER;
            end if;
          else
            tx_start_sig <= '0';
            FTM_test9_State <= SEND;
          end if;

        when WAIT_FOR_ANSWER =>
          LED_red_sig(0) <= '1';
          LED_ye_sig(0) <= '0';
          LED_gn_sig(0) <= '1';
          if rx_valid_sig = '1' then
            LED_red_sig(0) <= '1';
            LED_ye_sig(0) <= '1';
            LED_gn_sig(0) <= '0';
            FTM_test9_State <= FINISHED;
          else
            FTM_test9_State <= WAIT_FOR_ANSWER;
          end if;
            
        when FINISHED =>
          LED_red_sig(0) <= '1';
	  LED_ye_sig(0) <= '1';
          LED_gn_sig(0) <= '0';
        
      end case;
    end if;
  end process FTM_test9_FSM;

  LED_red <= LED_red_sig;
  LED_ye  <= LED_ye_sig;
  LED_gn  <= LED_gn_sig;
  
  LED_red_sig(3 downto 1) <= "111"; 
  LED_ye_sig(1) <= '1';
  LED_gn_sig(1) <= '1';

end Behavioral;
