--
-- VHDL Architecture FACT_FAD_lib.data_generator.beha
--
-- Created:
--          by - FPGA_Developer.UNKNOWN (EEPC8)
--          at - 14:36:14 10.02.2010
--
-- using Mentor Graphics HDL Designer(TM) 2008.1 (Build 17)

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
library fact_fad_lib;
use fact_fad_lib.fad_definitions.all;

-- -- Uncomment the following library declaration if instantiating
-- -- any Xilinx primitives in this code.
-- library UNISIM;
-- use UNISIM.VComponents.all;

entity data_generator is
generic(
	RAM_ADDR_WIDTH : integer := 12
);
port( 
	clk						: in	std_logic;					-- CLK_25.
	data_out				: out	std_logic_vector (63 downto 0);
	addr_out				: out	std_logic_vector (RAM_ADDR_WIDTH-1 downto 0);
	dataRAM_write_ea_o		: out	std_logic_vector (0 downto 0) := "0";
	ram_start_addr			: in	std_logic_vector (RAM_ADDR_WIDTH-1 downto 0);
	ram_write_ea			: in	std_logic;
	ram_write_ready			: out	std_logic := '0';
	ram_write_ready_ack		: IN 	std_logic;
	config_start_mm			: out	std_logic := '0'; 
	config_start_cm			: out	std_logic := '0';
	config_start_spi		: out	std_logic := '0';
	config_ready_mm			: in	std_logic;
	config_ready_cm			: in	std_logic;
	config_ready_spi		: in	std_logic;
	config_started_mm		: in	std_logic;
	config_started_cm		: in	std_logic;
	config_started_spi		: in	std_logic;
	roi_array				: in	roi_array_type;
	roi_max					: in	roi_max_type;
	sensor_array			: in	sensor_array_type;
	sensor_ready			: in	std_logic;
	dac_array				: in	dac_array_type;
	
	mode					: in	std_logic := '0';			-- 0: config mode | 1: run mode
	idling				: out	std_logic;
	
-- EVT HEADER - part 1
	package_length			: in	std_logic_vector (15 downto 0);
	pll_lock				: in	std_logic_vector ( 3 downto 0);

-- EVT HEADER - part 2  --> FTM trigger informaton, comes in late ...
-- during EVT header wrinting, this field is left out ... and only written into event header,
-- when the DRS chip were read out already.
	FTM_RS485_ready			: in	std_logic;
	FTM_trigger_info		: in	std_logic_vector (55 downto 0); --7 byte

-- EVT HEADER - part 3
	fad_event_counter		: in	std_logic_vector (31 downto 0);
	refclk_counter			: in	std_logic_vector (11 downto 0);
	refclk_too_high			: in	std_logic;
	refclk_too_low			: in	std_logic;

-- EVT HEADER - part 4
	board_id				: in	std_logic_vector (3 downto 0);
	crate_id				: in	std_logic_vector (1 downto 0);
	DCM_PS_status			: in	std_logic_vector (7 downto 0);
	TRG_GEN_div				: in	std_logic_vector (15 downto 0);

-- EVT HEADER - part 5
	dna						: in	std_logic_vector (63 downto 0);

-- EVT HEADER - part 6
	timer_value				: in	std_logic_vector (31 downto 0); -- time in units of 100us

	trigger					: in	std_logic;
	start_config_chain		: in	std_logic;				-- here W5300_MODUL can start the whole config chain
	config_chain_done		: out	std_logic;
	
	adc_data_array			: in	adc_data_array_type;
	adc_output_enable_inverted					: out	std_logic := '1';
	adc_clk_en				: out	std_logic := '0';
	adc_otr					: in	std_logic_vector (3 downto 0);
	drs_channel_id			: out	std_logic_vector (3 downto 0) := (others => '0');

--drs_dwrite : out std_logic := '1';
	drs_readout_ready		: out	std_logic := '0';
	drs_readout_ready_ack	: in	std_logic;
	drs_clk_en				: out	std_logic := '0';
	start_read_drs_stop_cell: out	std_logic := '0';

	drs_srin_write_8b		: out	std_logic := '0';
	drs_srin_write_ack		: in	std_logic;
	drs_srin_data			: out	std_logic_vector (7 downto 0) := (others => '0');
	drs_srin_write_ready	: in	std_logic;

	drs_read_s_cell_ready	: in	std_logic;
	drs_s_cell_array		: in	drs_s_cell_array_type;

	drs_readout_started		: out	std_logic := '0'
);
end data_generator ;

architecture Behavioral of data_generator is

type state_generate_type is (
	CONFIG_CHAIN_START,		-- IDLE branches into this state, if needed.
	CONFIG_MEMORY_MANAGER,
	CONFIG_SPI_INTERFACE,
	WAIT_FOR_CONFIG_SPI_INTERFACE,
	CONFIG_DRS_01,			-- these four states configure the DRS shift registers,
	CONFIG_DRS_02,			-- the make great use of the drs_pulser entity.
	CONFIG_DRS_03,			-- maybe they should be moved into the drs_pulser entity.
	WAIT_FOR_DRS_CONFIG_READY,

	IDLE,
	WRITE_HEADER, WRITE_FTM_INFO, WRITE_EVENTCOUNTER_AND_REFCLK_COUNTER, WRITE_BOARD_ID,
	WRITE_DNA, WRITE_TIMER, WRITE_TEMPERATURES, 
	WRITE_DAC1, WRITE_DAC2,
	WAIT_FOR_STOP_CELL,
	START_DRS_READING,
	WRITE_CHANNEL_ID, WRITE_START_CELL, WRITE_ROI, WRITE_FILLING,
	WAIT_FOR_ADC, WRITE_ADC_DATA,
	WRITE_EXTERNAL_TRIGGER,
	WRITE_END_FLAG,
	WRITE_DATA_END, WRITE_DATA_END_WAIT,
	WRITE_DATA_STOP, WRITE_DATA_STOP1
);

-- configuration stuff:
		-- this flag is set, when ever a rising edge on 'start_config_chain' is detected.
		-- this flag is cleared only, when a configuration chain was successfully processed
signal start_config_chain_flag : std_logic;
signal start_config_chain_sr : std_logic_vector(1 downto 0);

signal state_generate : state_generate_type := CONFIG_CHAIN_START;
signal start_addr : std_logic_vector (RAM_ADDR_WIDTH-1 downto 0) := (others => '0');

signal data_cntr : integer  range 0 to 1024 := 0;
signal addr_cntr : integer range 0 to RAM_SIZE_64B := 0;    -- counts 64 bit words
signal channel_id : integer range 0 to 9 := 0;
signal adc_wait_cnt : integer range 0 to 7 := 0;

signal trigger_sr :std_logic_vector(1 downto 0) := "00";
signal ram_write_ea_flag : std_logic := '0';
signal new_config_int : std_logic := '0';

-- internal signal: to be sampled once and used instead of inputs!
signal roi_max_int : roi_max_type;

signal sig_drs_readout_started : std_logic := '0';

signal sig_idling : std_logic := '1';

begin
drs_readout_started <= sig_drs_readout_started;
idling <= sig_idling;
generate_data : process (clk)
begin
	if rising_edge (clk) then
		start_config_chain_sr <= start_config_chain_sr(0) & start_config_chain;
		if (start_config_chain_sr = "01") then
			start_config_chain_flag <= '1';
			config_chain_done <= '0';
		end if;
		trigger_sr <= trigger_sr(0) & trigger; --synching in of asynchrounous trigger signal.
		addr_out <= start_addr + conv_std_logic_vector(addr_cntr, RAM_ADDR_WIDTH);
		
		case state_generate is
		
		when CONFIG_CHAIN_START =>						-- CONFIG_CONTROL_MANAGER
			config_start_cm <= '1';
			if (config_started_cm = '1') then
				config_start_cm <= '0';
				state_generate <= CONFIG_MEMORY_MANAGER;
			end if;
		when CONFIG_MEMORY_MANAGER =>					-- CONFIG_MEMORY_MANAGER		
			if (config_ready_cm = '1') then
				config_start_mm <= '1';
			end if;
			if (config_started_mm = '1') then
				config_start_mm <= '0';
				state_generate <= CONFIG_SPI_INTERFACE;
			end if;
		when CONFIG_SPI_INTERFACE =>					-- CONFIG_SPI_INTERFACE
			if (config_ready_mm = '1') then
				config_start_spi <= '1';
			end if;
			if (config_started_spi = '1') then
				config_start_spi <= '0';
				state_generate <= WAIT_FOR_CONFIG_SPI_INTERFACE;
			end if;
		when WAIT_FOR_CONFIG_SPI_INTERFACE =>
			if (config_ready_spi = '1') then
				state_generate <= CONFIG_DRS_01;
			end if;
		-- configure DRS
		-- all this might be done in the drs_pulser entity
		when CONFIG_DRS_01 =>							-- BEGIN CONFIG DRS
			drs_channel_id <= DRS_WRITE_SHIFT_REG;
			drs_srin_data <= "11111111";
			drs_srin_write_8b <= '1';
			if (drs_srin_write_ack = '1') then
				drs_srin_write_8b <= '0';
				state_generate <= CONFIG_DRS_02;
			end if;
		when CONFIG_DRS_02 =>
			if (drs_srin_write_ready = '1') then
				state_generate <= CONFIG_DRS_03;
			end if;
		when CONFIG_DRS_03 =>
			drs_channel_id <= DRS_WRITE_CONFIG_REG;
			drs_srin_data <= "11111111";
			drs_srin_write_8b <= '1';
			if (drs_srin_write_ack = '1') then
				drs_srin_write_8b <= '0';
				state_generate <= WAIT_FOR_DRS_CONFIG_READY;
			end if;
			
		-- last state of CONFIG CHAIN:
			-- here the input roi_max is sampled
			-- all other interesting input signals should be sampled here as well!
		when WAIT_FOR_DRS_CONFIG_READY =>							-- END OF CONFIG CHAIN
			if (drs_srin_write_ready = '1') then
				drs_channel_id <= DRS_ADDR_IDLE; -- to make sure not to write accidentally into DRS shift registers
				roi_max_int <= roi_max;
				config_chain_done <= '1';
				state_generate <= IDLE;
			end if;
			-- end configure DRS

		when IDLE =>
			if (mode = '0') then										-- do not accept any triggers ! stay in idle, or do a configuration.
				sig_idling <= '1';
				if (start_config_chain_flag = '1') then
					sig_idling <= '0';
					start_config_chain_flag <= '0';
					state_generate <= CONFIG_CHAIN_START;
				else
					state_generate <= IDLE;
				end if;							
			else --mode = '1'										-- check if trigger arrived.
				sig_idling<= '0';
				if (ram_write_ea = '1' and trigger_sr = "01") then
					sig_drs_readout_started <= '1'; -- is set to '0' in next state ... just a pulse.
					start_read_drs_stop_cell <= '1';
					adc_output_enable_inverted <= '0';
					-- at this moment the ADC ist beeing clocked. 
					-- this is not the start of the readout.
					-- the DRS needs to be clocked as well.
					adc_clk_en <= '1';
					start_addr <= ram_start_addr;
					state_generate <= WRITE_HEADER;
				end if;
			end if;
			
		when WRITE_HEADER =>
			sig_drs_readout_started <= '0'; -- is set to '1' in state IDLE
			dataRAM_write_ea_o <= "1";
			data_out <= X"000" & pll_lock & PACKAGE_VERSION & PACKAGE_SUB_VERSION & package_length & X"FB01";
			addr_cntr <= addr_cntr + 1;  
			state_generate <= WRITE_FTM_INFO;
					
		when WRITE_FTM_INFO =>
			-- THIS is just a dummy STATE just to make reading easier.
			-- at this point normally the FTM RS485 data would be written .. but we do not know it
			-- so here we do not write the FTM info ... just jump over it.
			addr_cntr <= addr_cntr + 1;  
			state_generate <= WRITE_EVENTCOUNTER_AND_REFCLK_COUNTER;
					
		when WRITE_EVENTCOUNTER_AND_REFCLK_COUNTER =>
			data_out <= X"0000" &
					refclk_too_high & refclk_too_low & "00" & refclk_counter & 
					fad_event_counter(15 downto 0) & 
					fad_event_counter(31 downto 16) ;
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_BOARD_ID;
			
		when WRITE_BOARD_ID =>     
			data_out <= TRG_GEN_div & 							-- this is a kind of prescaler for the continouus trigger generator
						X"0000" & 								-- this might be the number of soft triggers beeing generated in a 'burst' not implemented yet
						X"00" & DCM_PS_status & "000000" & 		-- number of steps, the phase shifter was shifted...
						crate_id & "1000" & board_id;			-- position of the board inside the camera
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_DNA;

		when WRITE_DNA =>
			data_out <= X"00" & dna(55 downto 0);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_TIMER;

		when WRITE_TIMER =>
			data_out <= X"0000" & X"0000" & timer_value; -- 2times 16bit reserved for additional status info
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_TEMPERATURES;

		-- DANGER: thist state can wait endlessly, if somethings wrong.
		when WRITE_TEMPERATURES =>     -- temperatures
			if (sensor_ready = '1') then
				data_out <= conv_std_logic_vector (sensor_array (3), 16) &
							conv_std_logic_vector (sensor_array (2), 16) &
							conv_std_logic_vector (sensor_array (1), 16) &
							conv_std_logic_vector (sensor_array (0), 16);
				addr_cntr <= addr_cntr + 1;
				state_generate <= WRITE_DAC1;
			end if;

		when WRITE_DAC1 =>
			data_out <= conv_std_logic_vector (dac_array (3), 16) &
						conv_std_logic_vector (dac_array (2), 16) &
						conv_std_logic_vector (dac_array (1), 16) &
						conv_std_logic_vector (dac_array (0), 16);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_DAC2;
		when WRITE_DAC2 =>
			data_out <= conv_std_logic_vector (dac_array (7), 16) &
						conv_std_logic_vector (dac_array (6), 16) &
						conv_std_logic_vector (dac_array (5), 16) &
						conv_std_logic_vector (dac_array (4), 16);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WAIT_FOR_STOP_CELL;

		when WAIT_FOR_STOP_CELL =>
			start_read_drs_stop_cell <= '0';
			if (drs_read_s_cell_ready = '1') then
				state_generate <= START_DRS_READING;
			end if;

		when START_DRS_READING =>
			--drs channel number
			drs_channel_id <= conv_std_logic_vector (channel_id, 4);
			
			--adc_output_enable_inverted <= '0'; -- nur fr Emulator ??????????????????
			-- this has been done earlier already ... why does it need to be repeated?

			--starte drs-clocking
			-- this is an interesting point:
				-- here the DRS clock starts to tick. but only some states later
				-- the ADC data is actually read out. 
				-- the reason is, that the ADC has a latency of 7 clock cycles, which means, 
				-- when the next rising edge of the DRS clock is produced. 
				-- an analog value is put out.
				-- when the next rising edge of the ADC clock is produced.
				-- this very analog value is sampled.
				-- but only seven clock ticks later, the degital result is available.
				-- from that point on, every clock tick produces a valid digital result.
			drs_clk_en <= '1';
			adc_wait_cnt <= 0;
			state_generate <= WRITE_CHANNEL_ID;

		when WRITE_CHANNEL_ID =>    -- write DRS and Channel IDs
			data_out <=		conv_std_logic_vector(3,12) & conv_std_logic_vector(channel_id,4) &
							conv_std_logic_vector(2,12) & conv_std_logic_vector(channel_id,4) &
							conv_std_logic_vector(1,12) & conv_std_logic_vector(channel_id,4) &
							conv_std_logic_vector(0,12) & conv_std_logic_vector(channel_id,4);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_START_CELL;
		when WRITE_START_CELL =>    -- write start cells
			data_out <= 	"000000" & drs_s_cell_array (3) &
							"000000" & drs_s_cell_array (2) &
							"000000" & drs_s_cell_array (1) &
							"000000" & drs_s_cell_array (0); 
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_ROI;
		
		when WRITE_ROI =>    -- write ROI
			data_out <=	"00000" & conv_std_logic_vector (roi_array((3) * 9 + channel_id), 11) &
						"00000" & conv_std_logic_vector (roi_array((2) * 9 + channel_id), 11) &
						"00000" & conv_std_logic_vector (roi_array((1) * 9 + channel_id), 11) &
						"00000" & conv_std_logic_vector (roi_array((0) * 9 + channel_id), 11);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_FILLING;

		when WRITE_FILLING =>    -- write FILLING
			data_out <= conv_std_logic_vector(0,64); -- filling
			addr_cntr <= addr_cntr + 1;
			state_generate <= WAIT_FOR_ADC;

		when WAIT_FOR_ADC =>
		-- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
			if (adc_wait_cnt < 4 ) then -- anpassen!!!! -- 3 fr Simulation, 4 fr FPGA???
				adc_wait_cnt <= adc_wait_cnt + 1;
			else
				state_generate <= WRITE_ADC_DATA;
			end if;
		when WRITE_ADC_DATA =>
			if (data_cntr < roi_max_int (channel_id)) then
				data_out <= "000" & adc_otr(3) & adc_data_array(3) &	--exchange ... with data_cntr when testbenching. 
							"000" & adc_otr(2) & adc_data_array(2) &
							"000" & adc_otr(1) & adc_data_array(1) &
							"000" & adc_otr(0) & adc_data_array(0);
				addr_cntr <= addr_cntr + 1;
				state_generate <= WRITE_ADC_DATA;
				data_cntr <= data_cntr + 1;
			else
				drs_clk_en <= '0';
				--adc_output_enable_inverted <= '1'; -- nur fr Emulator
				if (channel_id = 8) then
					state_generate <= WRITE_EXTERNAL_TRIGGER;
					adc_output_enable_inverted <= '1';
					-- switch off ADC_CLK
					adc_clk_en <= '0';
				else
					channel_id <= channel_id + 1;     -- increment channel_id 
					state_generate <= START_DRS_READING;
					data_cntr <= 0;
				end if;
			end if;
 
		when WRITE_EXTERNAL_TRIGGER =>    -- external trigger ID
			addr_out <= start_addr + conv_std_logic_vector(1, RAM_ADDR_WIDTH);
			data_out <=	FTM_trigger_info(15 downto 0) &
						FTM_trigger_info(31 downto 16) &
						FTM_trigger_info(47 downto 32) &
						X"00" & FTM_trigger_info(55 downto 48);
			state_generate <= WRITE_END_FLAG;

		when WRITE_END_FLAG =>
			data_out <= conv_std_logic_vector(0, 32) & X"04FE" & X"4242";
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_DATA_END;
		when WRITE_DATA_END =>
			dataRAM_write_ea_o <= "0";
				--information to: memory manager. 
				-- one Event was completely written into dataRAM.
			ram_write_ready <= '1';					
			state_generate <= WRITE_DATA_END_WAIT;
		when WRITE_DATA_END_WAIT =>
			-- check if memory manager received the formaer information.
			-- go on to next state.
			if (ram_write_ready_ack = '1') then
				state_generate <= WRITE_DATA_STOP;
				ram_write_ready <= '0';
			end if;
		when WRITE_DATA_STOP =>
			if (ram_write_ready_ack = '0') then
				drs_readout_ready <= '1'; 			--info to: trigger manager.
				data_cntr <= 0;
				addr_cntr <= 0;
				channel_id <= 0;
				state_generate <= WRITE_DATA_STOP1;
			end if;
		when WRITE_DATA_STOP1 =>
			if (drs_readout_ready_ack = '1') then
				drs_readout_ready <= '0';
				state_generate <= IDLE;
			end if;
		when others =>
			null;
		end case; -- state_generate
	end if; -- rising_edge (clk)
end process generate_data;
end Behavioral;