/********************************************************************\

  Class interfacing to FAD board

\********************************************************************/

#include "FADBoard.h"
using namespace std;

//
// Constructor
// 
FADBoard::FADBoard(string Server, unsigned short ServerPort, class FAD *Parent, unsigned int Num) {

  int Ret;

  // Initialization
  m = Parent;
  Active = true;
  Continue = true;
  CommOK = false;
  ACalibTime = -1;
  Status.Update.tv_sec = -1;
  Port = ServerPort;

  Name = new char [Server.size()+1]; // Name in permanent memory for DIM service
  strcpy(Name, Server.c_str());

  // Initialise mutex for synchronization
  pthread_mutexattr_t Attr;

  if ((Ret = pthread_mutexattr_settype(&Attr, PTHREAD_MUTEX_ERRORCHECK)) != 0) {
    m->Message(m->ERROR, "pthread_mutex_settype() failed (%s)", strerror(Ret));
  }
  if ((Ret = pthread_mutex_init(&Mutex, &Attr)) != 0) {
    m->Message(m->FATAL, "pthread_mutex_init() failed (%s)", strerror(Ret));
  }

  // Initialise condition variable for synchronization
  if ((Ret = pthread_cond_init(&CondVar, NULL)) != 0) {
    m->Message(m->FATAL, "pthread_cond_init() failed (%s)", strerror(Ret));
  }

  // Construct DIM service name prefix
  stringstream ID;
  ID << SERVER_NAME"/Board" << setfill('0') << setw(2) << Num << "/";

  DIM_Name = new DimService((ID.str()+"Server").c_str(), Name);
  DIM_ID = new DimService((ID.str()+"BoardID").c_str(), (char *) "S", NULL, 0);
  DIM_Temp = new DimService((ID.str()+"Temperature").c_str(), (char *) "F", NULL, 0);
  DIM_DAC = new DimService((ID.str()+"DAC").c_str(), (char *) "S", NULL, 0);
  DIM_ROI = new DimService((ID.str()+"ROI").c_str(), (char *) "S", NULL, 0);

  // Create thread that connects and receives data
  if ((Ret = pthread_create(&Thread, NULL, (void * (*)(void *)) LaunchThread,(void *) this)) != 0) {
    m->Message(m->FATAL, "pthread_create() failed in FADBoard() (%s)", strerror(Ret));
  }

  // Start thread to connect to other sockets
  DimThread::start();
}

//
// Destructor
//
FADBoard::~FADBoard() {

  int Ret;

  // Cancel thread (if it did not quit already) and wait for it to quit
  if ((Ret = pthread_cancel(Thread)) != 0 && Ret != ESRCH) {
	m->Message(m->ERROR, "pthread_cancel() failed in ~FADBoard() (%s)", strerror(Ret));
  }
  if ((Ret = pthread_join(Thread, NULL)) != 0) {
	m->Message(m->ERROR, "pthread_join() failed in ~FADBoard (%s)", strerror(Ret));
  }

  delete DIM_Name;
  delete DIM_ID;
  delete DIM_Temp;
  delete DIM_DAC;
  delete DIM_ROI;  
  delete[] Name;

  // Delete condition variable 
  if ((Ret = pthread_cond_destroy(&CondVar)) != 0) {
	m->Message(m->ERROR, "pthread_cond_destroy() failed in ~FADBoard (%s)", strerror(Ret));
  }

  // Delete mutex  
  if ((Ret = pthread_mutex_destroy(&Mutex)) != 0) {
	m->Message(m->ERROR, "pthread_mutex_destroy() failed in ~FADBoard (%s)", strerror(Ret));
  }
}


//
// Send data to board
//
void FADBoard::Send(const void *Data, size_t Bytes) {

  // Do not send if not active or communication problem
  if (!Active || !CommOK) return;

  // Write data
  ssize_t Result = write(Socket, Data, Bytes);

  // Check result
  if (Result == -1) m->PrintMessage("Error: Could not write to socket (%s)\n", strerror(errno));
  else if ((size_t) Result < Bytes) m->PrintMessage("Error: Could only write %d bytes out of %d to socket\n", Result, Bytes);
}

void FADBoard::Send(unsigned short Data) {

  unsigned short Buffer = htons(Data);
  
  Send(&Buffer, sizeof(unsigned short));
}


//
// Get board status (mutex protected to avoid concurrent access in ReadLoop)
//
struct FADBoard::BoardStatus FADBoard::GetStatus() {

  int Ret;
  struct BoardStatus S;
  
  // Lock
  if ((Ret = pthread_mutex_lock(&Mutex)) != 0) {
	m->Message(m->FATAL, "pthread_mutex_lock() failed in ReadLoop() (%s)", strerror(Ret));
  }

  S = Status; 

  // Unlock
  if ((Ret = pthread_mutex_unlock(&Mutex)) != 0) {
	m->Message(m->FATAL, "pthread_mutex_unlock() failed in Unlock() (%s)", strerror(Ret));
  }
  
  return S;  
}


//
// Perform amplitude calibration in steps
//
void FADBoard::AmplitudeCalibration() {

  enum StateType {wait, init, baseline, gain, secondary};

  static struct BoardStatus InitialStatus;
  static vector<unsigned short> ROICmd;
  static unsigned short DACCmd[] = {htons(CMD_Write | (BADDR_DAC + 1)), 0, htons(CMD_Write | (BADDR_DAC + 2)), 0, htons(CMD_Write | (BADDR_DAC + 3)), 0};
  static StateType State = wait;
  static int Count = 0;

  // Check if mode is amplitude calibration
  if (m->Mode != m->acalib) {
	State = init;
	return;
  }

  switch (State) {
  // ====== Part A: Initialization =====
  case init:
	// Invalidate current calibration
	ACalibTime = -1;

	// Save initial board status, set all ROIs to 1024 and set DAC values
	InitialStatus = GetStatus();

	for (unsigned int i=0; i<NChips*NChannels; i++) {
	  ROICmd.push_back(htons(CMD_Write | (BADDR_ROI + i)));
	  ROICmd.push_back(htons(NBins));
	}
	Send(&ROICmd[0], ROICmd.size()*sizeof(unsigned short));

    DACCmd[1] = htons(0);
    DACCmd[3] = htons(0);
    DACCmd[5] = htons(0);
	Send(DACCmd, sizeof(DACCmd));

	// Clear sum vector and set state to accumulate
	memset(Sum, 0, sizeof(Sum));
	State = baseline;
	break;

  // ====== Part B: Baseline calibration =====
  case baseline:
	for (unsigned int Chip=0; Chip<NChips; Chip++) {
	  for (unsigned int Chan=0; Chan<NChannels; Chan++) {
		for (int i=0; i<Status.ROI[Chip][Chan]; i++) {
		  Sum[Chip][Chan][(i+Status.TriggerCell[Chip]) % NBins] += Data[Chip][Chan][i];
		}
	  }
	}
    Count++;
	
	// Determine baseline if integration finished
	if (Count == m->NumEventsRequested) {
	  for (unsigned int i=0; i<NChips; i++) {
		for (unsigned int j=0; j<NChannels; j++) {
		  for (unsigned int k=0; k<NBins; k++) {
			Baseline[i][j][k] = Sum[i][j][k] / m->NumEventsRequested;
		  }
		}
	  }
	  
	  // Set new DAC values and start accumulation
	  DACCmd[1] = htons(50000);
	  DACCmd[3] = htons(50000);
	  DACCmd[5] = htons(50000);
	  Send(DACCmd, sizeof(DACCmd));

	  // Clear sum vector and set state to accumulate
	  memset(Sum, 0, sizeof(Sum));
	  Count = 0;
	  State = gain;
	}
	break;

  // ====== Part C: Gain calibration =====
  case gain:
	for (unsigned int Chip=0; Chip<NChips; Chip++) {
	  for (unsigned int Chan=0; Chan<NChannels; Chan++) {
		for (int i=0; i<Status.ROI[Chip][Chan]; i++) {
		  Sum[Chip][Chan][(i+Status.TriggerCell[Chip]) % NBins] += Data[Chip][Chan][i];
		}
	  }
	}
    Count++;
	
	// Determine gain if integration finished
	if (Count == m->NumEventsRequested) {
	  for (unsigned int i=0; i<NChips; i++) {
		for (unsigned int j=0; j<NChannels; j++) {
		  for (unsigned int k=0; k<NBins; k++) {
			Gain[i][j][k] = (Sum[i][j][k] / m->NumEventsRequested) - Baseline[i][j][k];
		  }
		}
	  }
	  
	  // Set new DAC values and start accumulation
	  DACCmd[1] = htons(0);
	  DACCmd[3] = htons(0);
	  DACCmd[5] = htons(0);
	  Send(DACCmd, sizeof(DACCmd));

	  // Clear sum vector and set state to accumulate
	  memset(Sum, 0, sizeof(Sum));
	  Count = 0;
	  State = secondary;
	}
	break;

  // ====== Part D: Secondary calibration =====
  case secondary:
	for (unsigned int Chip=0; Chip<NChips; Chip++) {
	  for (unsigned int Chan=0; Chan<NChannels; Chan++) {
		for (int i=0; i<Status.ROI[Chip][Chan]; i++) {
		  Sum[Chip][Chan][i] = Data[Chip][Chan][i] - Baseline[Chip][Chan][(i-Status.TriggerCell[Chip]) % NBins];
		}
	  }
	}
    Count++;
	
	// Determine secondary baseline if integration finished
	if (Count == m->NumEventsRequested) {
	  for (unsigned int i=0; i<NChips; i++) {
		for (unsigned int j=0; j<NChannels; j++) {
		  for (unsigned int k=0; k<NBins; k++) {
			SecondaryBaseline[i][j][k] = Sum[i][j][k] / (double) m->NumEventsRequested;
		  }
		}
	  }
	  
	  // Write back original ROI and DAC settings
	  ROICmd.clear();
	  for (unsigned int i=0; i<NChips*NChannels; i++) {
		ROICmd.push_back(htons(CMD_Write | (BADDR_ROI + i)));
		ROICmd.push_back(htons(InitialStatus.ROI[i/NChannels][i%NChannels]));
	  }
	  Send(&ROICmd[0], ROICmd.size()*sizeof(unsigned short));

      DACCmd[1] = htons(InitialStatus.DAC[1]);
      DACCmd[3] = htons(InitialStatus.DAC[2]);
      DACCmd[5] = htons(InitialStatus.DAC[3]);
  	  Send(DACCmd, sizeof(DACCmd));

	  // Store calibration time and temperature
	  ACalibTime = time(NULL);
	  ACalibTemp = 0;
	  for (unsigned int i=0; i<NTemp; i++) ACalibTemp += Status.Temp[i] / NTemp;
	  
	  // Inform event thread that calibration is finished
	  string Message = string("ACALIBDONE")+Name;
	  if (write(m->Pipe[1], Message.data(), Message.size()) == -1) {
		m->Message(m->ERROR, "write() to Pipe[1] failed in class FADBoard::AmplitudeCalibration (%s)", strerror(errno));
	  }

	  State = wait;
	}
	break;

  // ====== Wait for Mode not being acalib =====
  case wait:
	if (m->Mode != m->acalib) State = init;
	break;
  }
}  
  
//
// Connect to board and read data
//
void FADBoard::ReadLoop() {

  char Buffer[READ_BUFFER_SIZE];
  unsigned int Pos = 0, Temp;
  const PEVNT_HEADER *Header = (PEVNT_HEADER *) Buffer;
  ssize_t Result;
  struct sockaddr_in SocketAddress;
  struct BoardStatus PrevStatus;
  int Ret;

  // Resolve hostname
  struct hostent *Host = gethostbyname(Name);
  if (Host == 0) {
    m->Message(m->WARN, "Could not resolve host name for %s", Name);
    return;
  }

  SocketAddress.sin_family = PF_INET;
  SocketAddress.sin_port = htons(Port);
  SocketAddress.sin_addr = *(struct in_addr*) Host->h_addr;

  // Open socket descriptor
  if ((Socket = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
    m->Message(m->ERROR, "Could not open socket for %s (%s)\n", Name, strerror(errno));
    return;
  }
    
  // Connect to server
  if (connect(Socket, (struct sockaddr *) &SocketAddress, sizeof(SocketAddress)) == -1) {
    m->PrintMessage("Could not connect to %s at port %d (%s)\n", Name, Port, strerror(errno));
  }
  else CommOK = true;
  memset(&PrevStatus, 0, sizeof(PrevStatus));

  // Leave loop if program termination requested or board communication not OK
  while (!m->ExitRequest && CommOK) {
    // Read data from socket
    Result = read(Socket, Buffer + Pos, sizeof(Buffer)-Pos);

	// Check result of read
	if (Result == -1) {
	  m->PrintMessage("Error: Could not read from socket, exiting read loop (%s)\n", strerror(errno));
	  CommOK = false;
	  break;
	}
	else if (Result == 0) {
	  m->PrintMessage("Server not existing anymore, exiting read loop\n");
	  CommOK = false;
	  break;
	}
	
	// If not active, discard incoming data
	if (!Active) continue;
	
	// Advance write pointer
	Pos += Result;
	
	// Check if internal buffer full
	if (Pos == sizeof(Buffer)) {
	  m->PrintMessage("Internal buffer full, deleting all data in buffer\n");
	  Pos = 0;
	  continue;
	}
	
	// Check if buffer starts with start_package_flag, remove data if not
	Temp = 0;
	while (ntohs(*((unsigned short *) (Buffer+Temp))) != 0xfb01 && Temp<Pos) Temp++;
	if (Temp != 0) {
	  memmove(Buffer, Buffer+Temp, Pos-Temp);
	  Pos -= Temp;
	  m->PrintMessage("Removed %d bytes because of start_package_flag not found for %s\n", Temp, Name);
	  continue;
	}

	// Wait until the buffer contains at least enough bytes to potentially hold a PEVNT_HEADER
	if (Pos < sizeof(PEVNT_HEADER)) continue;
	
	unsigned int Length = ntohs(Header->package_length)*2*sizeof(char);
	if (Pos < Length) continue;

	// Extract data if event end package flag correct
	if (ntohs(*(unsigned short *) (Buffer+Length-sizeof(unsigned short))) == 0x04FE) {

	  // Prepare pointers to channel data (channels stored in order 0,9,18,27 - 1,10,19,28 - ... - 8,17,26,35)
	  PCHANNEL *Channel[NChips*NChannels], *Pnt=(PCHANNEL *) (Header+1); 
	  for(unsigned int i=0; i<NChips*NChannels; i++) {
		Channel[i] = Pnt;
		Pnt = (PCHANNEL *) ((short *) (Channel[i] + 1) + ntohs(Channel[i]->roi));
	  } 

	  PrevStatus = Status;

	  // Wait until event thread processed the previous data and lock to avoid concurrent access in GetStatus()
	  Lock();
	  while (!Continue) {
		if ((Ret = pthread_cond_wait(&CondVar, &Mutex)) != 0) {
		  m->Message(m->ERROR, "pthread_cond_wait() failed (%s)", strerror(Ret));
		}
	  }
	  gettimeofday(&Status.Update, NULL);

	  // Extract ID and type information
	  Status.BoardID = ntohl(Header->board_id);
	  Status.FirmwareRevision = ntohl(Header->version_no);
	  Status.TriggerID = ntohl(Header->trigger_id);
	  Status.TriggerType = ntohs(Header->trigger_type);

	  // Extract temperatures (MSB indicates if temperature is positive or negative)
	  for (unsigned int i=0; i<NTemp; i++) {
		if ((ntohs(Header->drs_temperature[i]) & 0x8000) == 0) Status.Temp[i] = float(ntohs(Header->drs_temperature[i]) >> 3)/16;
		else Status.Temp[i] = float(0xE000 | (ntohs(Header->drs_temperature[i])) >> 3)/16;
	  }

	  // Extract DAC channels
	  for (unsigned int i=0; i<NDAC; i++) Status.DAC[i] = ntohs(Header->dac[i]);

	  short Buf;
	  for (unsigned int Chip=0; Chip<NChips; Chip++) {
		// Extract trigger cells	  
		Status.TriggerCell[Chip] = (int) ntohs(Channel[Chip]->start_cell);
	  
		for (unsigned int Chan=0; Chan<NChannels; Chan++) {
		  // Extract ROI
		  Status.ROI[Chip][Chan] = ntohs(Channel[Chip+NChips*Chan]->roi);

		  // Extract ADC data (stored in 12 bit signed twis complement with out-of-range-bit and leading zeroes)
		  for (int i=0; i<Status.ROI[Chip][Chan]; i++) {
			  Buf = ntohs(Channel[Chip+NChips*Chan]->adc_data[i]);
			  (Buf <<= 4) >>= 4;			//delete the sign-bit by shifting left and shift back
			  Data[Chip][Chan][i] = Buf;					
		  }
		}
	  }
	  
	  // Prepare predicate for condition variable
	  Continue = false;
	  Unlock();
	  
  	  // Amplitude calibration (will check if Mode is acalib)
	  AmplitudeCalibration();

	  // Update DIM services if necessary
	  if (memcmp(PrevStatus.Temp, Status.Temp, sizeof(Status.Temp)) != 0) {
		DIM_Temp->updateService(Status.Temp, sizeof(Status.Temp));
	  }
	  if (memcmp(PrevStatus.DAC, Status.DAC, sizeof(Status.DAC)) != 0) {
		DIM_DAC->updateService(Status.DAC, sizeof(Status.DAC));
	  }  
	  if (memcmp(PrevStatus.ROI, Status.ROI, sizeof(Status.ROI)) != 0) {
		DIM_ROI->updateService(Status.ROI, sizeof(Status.ROI));
	  }  
	  if (PrevStatus.BoardID != Status.BoardID) {
		DIM_ID->updateService(&Status.BoardID, sizeof(Status.BoardID));
	  }
	  
	  // Inform event thread of new data
	  string Message = string("EVENT")+Name;
	  if (write(m->Pipe[1], Message.data(), Message.size()) == -1) {
		m->Message(m->ERROR, "write() to Pipe[1] failed in class FADBoard (%s)", strerror(errno));
		m->ExitRequest = true;
	  }	    
	}
	else m->PrintMessage("End package flag incorrect, removing corrupt event\n");

	// Remove event data from internal buffer
	memmove(Buffer, Buffer+Length, Pos-Length);
	Pos = Pos-Length;	
  } // while()
  
  // Close socket descriptor
  if (close(Socket) == -1) {
	m->PrintMessage("Could not close socket descriptor for board %s (%s)", Name, strerror(errno));  
  }

}


//
// Launch read thread inside class
//
void FADBoard::LaunchThread(class FADBoard *m) {

  m->ReadLoop();
}


//
// Lock and unlock mutex
//
void FADBoard::Lock() {

  int Ret;

  if ((Ret = pthread_mutex_lock(&Mutex)) != 0) {
	m->Message(m->FATAL, "pthread_mutex_lock() failed in class FADBoard (%s)", strerror(Ret));
  }
}

void FADBoard::Unlock() {

  int Ret;

  if ((Ret = pthread_mutex_unlock(&Mutex)) != 0) {
	m->Message(m->FATAL, "pthread_mutex_unlock() failed in class FADBoard (%s)", strerror(Ret));
  }
}


//
// Open other sockets
//
void FADBoard::threadHandler() {

  int List[] = {5001, 5002, 5003, 5004, 5005, 5006, 5007};
  int Socket[sizeof(List)/sizeof(int)], MaxSocketNum, Ret;
  fd_set DescriptorList;
  char Buffer[1000000];
  
  // Resolve hostname
  struct hostent *Host = gethostbyname(Name);
  if (Host == 0) {
    m->PrintMessage("OtherSockets: Could not resolve host name for %s\n", Name);
    return;
  }

  // Connect to server
  struct sockaddr_in SocketAddress;
  SocketAddress.sin_family = PF_INET;
  SocketAddress.sin_addr = *(struct in_addr*) Host->h_addr;

  for (unsigned int i=0; i<sizeof(List)/sizeof(int); i++) {
	// Open socket descriptor
	if ((Socket[i] = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
      m->PrintMessage("OtherSockets: Could not open socket for port %d (%s)\n", List[i], strerror(errno));
      return;
	}
	MaxSocketNum = *max_element(Socket, Socket+sizeof(List)/sizeof(int));
	 
	// Connect to server
    SocketAddress.sin_port = htons((unsigned short) List[i]);
	if (connect(Socket[i], (struct sockaddr *) &SocketAddress, sizeof(SocketAddress)) == -1) {
      m->PrintMessage("OtherSockets: Could not connect to port %d (%s)\n", List[i], strerror(errno));
      return;
	}
  }
  
  while(true) {
    // Wait for data from sockets
    FD_ZERO(&DescriptorList);   
    for (unsigned int i=0; i<sizeof(List)/sizeof(int); i++) FD_SET(Socket[i], &DescriptorList);
    if (select(MaxSocketNum+1, &DescriptorList, NULL, NULL, NULL) == -1) {
      m->PrintMessage("OtherSockets: Error with select() (%s)\n", strerror(errno));
      break;
    }
	
	// Data from socket
	for (unsigned int i=0; i<sizeof(List)/sizeof(int); i++) if (FD_ISSET(Socket[i], &DescriptorList)) {
	  Ret = read(Socket[i], Buffer, sizeof(Buffer));
      if(Ret == 0) m->PrintMessage("OtherSockets: Connection to port %d not existing anymore\n", List[i]);
      else if (Ret == -1) m->PrintMessage("OtherSockets: Error reading from port %d (%s)\n", List[i], strerror(errno));
    }
  }

  // Close all sockets
  for (unsigned int i=0; i<sizeof(List)/sizeof(int); i++) {
	if ((Socket[i] != -1) && (close(Socket[i]) == -1)) {
	  m->PrintMessage("OtherSockets: Could not close socket of port %d (%s)", List[i], strerror(errno));  
	}
  }
}
